# Error Handling

Proper error handling in async Python code.

## Pattern: Try/Except in Async Functions

```python
import asyncio

async def risky_operation(item_id: int) -> dict:
    """Operation that might fail."""
    await asyncio.sleep(0.1)
    if item_id % 3 == 0:
        raise ValueError(f"Item {item_id} failed")
    return {"id": item_id, "status": "success"}

async def safe_operation(item_id: int):
    """Wrapper with error handling."""
    try:
        return await risky_operation(item_id)
    except ValueError as e:
        print(f"Error: {e}")
        return None
```

## Pattern: gather() with return_exceptions

```python
async def process_items(item_ids: List[int]):
    """Process multiple items with error handling."""
    tasks = [safe_operation(iid) for iid in item_ids]
    results = await asyncio.gather(*tasks, return_exceptions=True)

    # Filter results
    successful = [r for r in results if r is not None and not isinstance(r, Exception)]
    failed = [r for r in results if isinstance(r, Exception)]

    print(f"Success: {len(successful)}, Failed: {len(failed)}")
    return successful
```

## Pattern: Individual Task Error Handling

```python
async def handle_task_errors():
    """Handle errors from individual tasks."""
    task1 = asyncio.create_task(risky_operation(1))
    task2 = asyncio.create_task(risky_operation(3))  # Will fail

    try:
        result1 = await task1
        print(f"Task 1: {result1}")
    except Exception as e:
        print(f"Task 1 failed: {e}")

    try:
        result2 = await task2
        print(f"Task 2: {result2}")
    except Exception as e:
        print(f"Task 2 failed: {e}")
```
