# Performance Best Practices

Optimize async Python applications for best performance.

## 1. Use Connection Pools

```python
import aiohttp

async def with_connection_pool():
    """Use connection pool for efficiency."""
    connector = aiohttp.TCPConnector(limit=100, limit_per_host=10)

    async with aiohttp.ClientSession(connector=connector) as session:
        tasks = [session.get(f"https://api.example.com/item/{i}") for i in range(50)]
        responses = await asyncio.gather(*tasks)
        return responses
```

## 2. Batch Operations

```python
async def batch_process(items: List[str], batch_size: int = 10):
    """Process items in batches."""
    for i in range(0, len(items), batch_size):
        batch = items[i:i + batch_size]
        tasks = [process_item(item) for item in batch]
        await asyncio.gather(*tasks)
        print(f"Processed batch {i // batch_size + 1}")
```

## 3. Avoid Blocking Operations

```python
import asyncio
import concurrent.futures

def blocking_operation(data):
    """CPU-intensive blocking operation."""
    import time
    time.sleep(1)
    return data * 2

async def run_in_executor(data):
    """Run blocking operation in thread pool."""
    loop = asyncio.get_event_loop()
    with concurrent.futures.ThreadPoolExecutor() as pool:
        result = await loop.run_in_executor(pool, blocking_operation, data)
        return result
```

## 4. Limit Concurrent Operations

```python
async def limited_concurrency(urls: List[str], max_concurrent: int = 5):
    """Limit concurrent requests."""
    semaphore = asyncio.Semaphore(max_concurrent)

    async def fetch_with_limit(url):
        async with semaphore:
            return await fetch(url)

    tasks = [fetch_with_limit(url) for url in urls]
    return await asyncio.gather(*tasks)
```

## 5. Reuse Event Loop

```python
# ❌ Bad: Creating new loop each time
def bad_approach():
    loop = asyncio.new_event_loop()
    loop.run_until_complete(my_coroutine())
    loop.close()

# ✅ Good: Use asyncio.run() (Python 3.7+)
asyncio.run(my_coroutine())
```
