# Synchronization

Synchronization primitives for async code.

## Semaphore for Rate Limiting

```python
import asyncio

async def api_call(url: str, semaphore: asyncio.Semaphore) -> dict:
    """Make API call with rate limiting."""
    async with semaphore:
        print(f"Calling {url}")
        await asyncio.sleep(0.5)
        return {"url": url, "status": 200}

async def rate_limited_requests(urls: List[str], max_concurrent: int = 5):
    """Make multiple requests with rate limiting."""
    semaphore = asyncio.Semaphore(max_concurrent)
    tasks = [api_call(url, semaphore) for url in urls]
    results = await asyncio.gather(*tasks)
    return results
```

## Async Locks

```python
class AsyncCounter:
    """Thread-safe async counter."""

    def __init__(self):
        self.value = 0
        self.lock = asyncio.Lock()

    async def increment(self):
        """Safely increment counter."""
        async with self.lock:
            current = self.value
            await asyncio.sleep(0.01)
            self.value = current + 1

    async def get_value(self) -> int:
        async with self.lock:
            return self.value
```

## Events

```python
async def waiter(event: asyncio.Event, name: str):
    """Wait for event."""
    print(f"{name} waiting for event")
    await event.wait()
    print(f"{name} triggered")

async def setter(event: asyncio.Event):
    """Set event after delay."""
    await asyncio.sleep(2)
    print("Setting event")
    event.set()

async def event_example():
    event = asyncio.Event()

    waiters = [asyncio.create_task(waiter(event, f"Waiter-{i}")) for i in range(3)]
    setter_task = asyncio.create_task(setter(event))

    await asyncio.gather(*waiters, setter_task)
```

## Conditions

```python
async def consumer(condition: asyncio.Condition, items: List):
    """Consumer that waits for items."""
    async with condition:
        await condition.wait_for(lambda: len(items) > 0)
        item = items.pop(0)
        print(f"Consumed: {item}")

async def producer(condition: asyncio.Condition, items: List):
    """Producer that adds items."""
    await asyncio.sleep(1)
    async with condition:
        items.append("item")
        condition.notify()
```
