# Testing Async Code

Test async Python code with pytest-asyncio.

## Basic Async Test

```python
import asyncio
import pytest

@pytest.mark.asyncio
async def test_async_function():
    """Test async function."""
    result = await fetch_data("https://api.example.com")
    assert result is not None
    assert "data" in result
```

## Testing with Timeout

```python
@pytest.mark.asyncio
async def test_with_timeout():
    """Test with timeout."""
    with pytest.raises(asyncio.TimeoutError):
        await asyncio.wait_for(slow_operation(5), timeout=1.0)
```

## Testing Concurrent Operations

```python
@pytest.mark.asyncio
async def test_concurrent_fetch():
    """Test concurrent operations."""
    user_ids = [1, 2, 3]
    users = await fetch_all_users(user_ids)

    assert len(users) == 3
    assert all(user["id"] in user_ids for user in users)
```

## Mocking Async Functions

```python
from unittest.mock import AsyncMock

@pytest.mark.asyncio
async def test_with_mock():
    """Test with mocked async function."""
    mock_db = AsyncMock()
    mock_db.fetch_user.return_value = {"id": 1, "name": "Test"}

    result = await get_user_data(mock_db, 1)

    assert result["user"]["name"] == "Test"
    mock_db.fetch_user.assert_called_once_with(1)
```

## Testing Error Handling

```python
@pytest.mark.asyncio
async def test_error_handling():
    """Test error handling in async code."""
    with pytest.raises(ValueError):
        await risky_operation(3)

    # Test safe wrapper
    result = await safe_operation(3)
    assert result is None
```

## Fixtures for Async Tests

```python
@pytest.fixture
async def async_client():
    """Async fixture."""
    client = AsyncClient()
    await client.connect()
    yield client
    await client.disconnect()

@pytest.mark.asyncio
async def test_with_fixture(async_client):
    """Test using async fixture."""
    result = await async_client.query("SELECT 1")
    assert result is not None
```

## Testing Cancellation

```python
@pytest.mark.asyncio
async def test_cancellation():
    """Test task cancellation."""
    task = asyncio.create_task(long_running_task())

    await asyncio.sleep(0.5)
    task.cancel()

    with pytest.raises(asyncio.CancelledError):
        await task
```

## Setup pytest-asyncio

```bash
pip install pytest-asyncio
```

```python
# conftest.py
pytest_plugins = ("pytest_asyncio",)
```
