# Timeouts and Cancellation

Handle timeouts and task cancellation in async code.

## Pattern: wait_for() Timeout

```python
import asyncio

async def slow_operation(delay: int) -> str:
    """Operation that takes time."""
    await asyncio.sleep(delay)
    return f"Completed after {delay}s"

async def with_timeout():
    """Execute operation with timeout."""
    try:
        result = await asyncio.wait_for(slow_operation(5), timeout=2.0)
        print(result)
    except asyncio.TimeoutError:
        print("Operation timed out")

asyncio.run(with_timeout())
```

## Pattern: Task Cancellation

```python
async def cancelable_task():
    """Task that handles cancellation."""
    try:
        while True:
            await asyncio.sleep(1)
            print("Working...")
    except asyncio.CancelledError:
        print("Task cancelled, cleaning up...")
        # Perform cleanup
        raise  # Re-raise to propagate cancellation

async def main():
    task = asyncio.create_task(cancelable_task())

    await asyncio.sleep(3)
    task.cancel()

    try:
        await task
    except asyncio.CancelledError:
        print("Task was cancelled")

asyncio.run(main())
```

## Pattern: Graceful Shutdown

```python
async def worker(name: str):
    """Worker task."""
    try:
        while True:
            await asyncio.sleep(1)
            print(f"{name} working...")
    except asyncio.CancelledError:
        print(f"{name} shutting down...")
        raise

async def main():
    workers = [asyncio.create_task(worker(f"Worker-{i}")) for i in range(3)]

    # Let workers run
    await asyncio.sleep(5)

    # Cancel all workers
    for task in workers:
        task.cancel()

    # Wait for cancellation
    await asyncio.gather(*workers, return_exceptions=True)

asyncio.run(main())
```
