# l10n.yaml Configuration

Complete reference for `l10n.yaml` configuration options used by Flutter's gen-l10n tool.

## Basic Configuration

```yaml
arb-dir: lib/l10n
template-arb-file: app_en.arb
output-localization-file: app_localizations.dart
```

### Required Options

| Option | Description | Default |
|--------|-------------|---------|
| `arb-dir` | Directory containing ARB files | `lib/l10n` |
| `template-arb-file` | Template ARB file used as basis for generation | `app_en.arb` |
| `output-localization-file` | Filename for generated localization classes | `app_localizations.dart` |

## Advanced Configuration Options

### Output Options

| Option | Description | Example |
|--------|-------------|---------|
| `output-dir` | Directory for generated localization files | `lib/generated` |
| `output-class` | Dart class name for localizations | `MyAppLocalizations` |
| `header` | String header to prepend to generated files | `"/// Localizations"` |
| `header-file` | File containing header text | `header.txt` |

### Locale Options

| Option | Description | Example |
|--------|-------------|---------|
| `preferred-supported-locales` | Default locale list | `[en_US, es_ES]` |

### Code Generation Options

| Option | Description | Default |
|--------|-------------|---------|
| `use-escaping` | Enable single quote escaping syntax | `false` |
| `nullable-getter` | Make localizations class getter nullable | `true` |
| `use-named-parameters` | Use named parameters for methods | `false` |
| `format` | Run `dart format` after generation | `true` |

### Generation Tracking

| Option | Description | Example |
|--------|-------------|---------|
| `gen-inputs-and-outputs-list` | Directory for JSON tracking file | `.` |
| `synthetic-package` | Generate as synthetic package | `true` |
| `project-dir` | Root Flutter project directory | `/path/to/project` |

### Translation Tracking

| Option | Description | Example |
|--------|-------------|---------|
| `untranslated-messages-file` | File for untranslated messages | `untranslated.json` |

### Additional Options

| Option | Description | Default |
|--------|-------------|---------|
| `suppress-warnings` | Suppress all warnings | `false` |
| `relax-syntax` | Relaxed syntax parsing | `false` |
| `required-resource-attributes` | Require resource attributes | `false` |
| `use-deferred-loading` | Use deferred loading for locales | `false` |

## Common Configurations

### Standard Setup

```yaml
arb-dir: lib/l10n
template-arb-file: app_en.arb
output-localization-file: app_localizations.dart
```

### Custom Output Directory

```yaml
arb-dir: lib/l10n
template-arb-file: app_en.arb
output-localization-file: app_localizations.dart
output-dir: lib/generated
synthetic-package: false
```

### Custom Class Name

```yaml
arb-dir: lib/l10n
template-arb-file: app_en.arb
output-localization-file: app_localizations.dart
output-class: MyAppLocalizations
```

### Deferred Loading for Web

```yaml
arb-dir: lib/l10n
template-arb-file: app_en.arb
output-localization-file: app_localizations.dart
use-deferred-loading: true
```

### Non-Nullable Getter

```yaml
arb-dir: lib/l10n
template-arb-file: app_en.arb
output-localization-file: app_localizations.dart
nullable-getter: false
```

### Tracking Untranslated Messages

```yaml
arb-dir: lib/l10n
template-arb-file: app_en.arb
output-localization-file: app_localizations.dart
untranslated-messages-file: untranslated.json
```

## Using Deferred Loading

Deferred loading reduces initial bundle size for web apps by loading locale files on demand:

```yaml
use-deferred-loading: true
```

Then import and use:

```dart
import 'package:flutter_gen/gen_l10n/app_localizations.dart';

Future<AppLocalizations> loadLocale(String localeCode) async {
  return await AppLocalizations.delegate.load(Locale(localeCode));
}
```

## Header Configuration

### String Header

```yaml
header: "/// All localized files generated by flutter gen-l10n"
```

### File Header

Create `lib/l10n/header.txt`:
```
/// Copyright 2024 My Company
/// All rights reserved
/// Generated localization files
```

Then configure:

```yaml
header-file: header.txt
```

## Number Formatting

For placeholder number formatting options, see [number-formats.md](number-formats.md).
