# pytest Fixture Examples

Comprehensive examples demonstrating pytest fixture patterns, from basic to advanced usage.

## Table of Contents

1. [Basic Fixtures](#basic-fixtures)
2. [Fixture Scopes](#fixture-scopes)
3. [Fixture Dependencies](#fixture-dependencies)
4. [Fixture Factories](#fixture-factories)
5. [Parametrized Fixtures](#parametrized-fixtures)
6. [Autouse Fixtures](#autouse-fixtures)
7. [Fixture Finalization](#fixture-finalization)
8. [Database Fixtures](#database-fixtures)
9. [API Testing Fixtures](#api-testing-fixtures)
10. [File and Directory Fixtures](#file-and-directory-fixtures)
11. [Mocking Fixtures](#mocking-fixtures)
12. [Complex Fixture Patterns](#complex-fixture-patterns)

## Basic Fixtures

### Example 1: Simple Data Fixture

```python
import pytest

@pytest.fixture
def sample_user():
    """Provide sample user data."""
    return {
        "id": 1,
        "name": "Alice Smith",
        "email": "alice@example.com",
        "age": 30
    }

def test_user_name(sample_user):
    assert sample_user["name"] == "Alice Smith"

def test_user_email(sample_user):
    assert "@" in sample_user["email"]

def test_user_age(sample_user):
    assert sample_user["age"] >= 18
```

### Example 2: Object Instance Fixture

```python
import pytest
from models import User

@pytest.fixture
def user_instance():
    """Create a User instance for testing."""
    return User(
        username="testuser",
        email="test@example.com",
        first_name="Test",
        last_name="User"
    )

def test_user_full_name(user_instance):
    assert user_instance.full_name() == "Test User"

def test_user_is_active(user_instance):
    assert user_instance.is_active is True

def test_user_string_representation(user_instance):
    assert str(user_instance) == "testuser"
```

### Example 3: List Fixture

```python
import pytest

@pytest.fixture
def number_list():
    """Provide a list of numbers for testing."""
    return [1, 2, 3, 4, 5, 6, 7, 8, 9, 10]

def test_list_length(number_list):
    assert len(number_list) == 10

def test_list_sum(number_list):
    assert sum(number_list) == 55

def test_list_contains(number_list):
    assert 5 in number_list
    assert 11 not in number_list
```

## Fixture Scopes

### Example 4: Function Scope (Default)

```python
import pytest

@pytest.fixture  # scope="function" is default
def counter():
    """Create new counter for each test."""
    return {"count": 0}

def test_increment_1(counter):
    counter["count"] += 1
    assert counter["count"] == 1

def test_increment_2(counter):
    # New counter created, starts at 0
    counter["count"] += 1
    assert counter["count"] == 1
```

### Example 5: Class Scope

```python
import pytest

@pytest.fixture(scope="class")
def shared_resource():
    """Shared across all tests in a class."""
    print("\nSetup shared resource")
    resource = {"data": []}
    yield resource
    print("\nTeardown shared resource")

class TestResourceSharing:
    def test_add_item_1(self, shared_resource):
        shared_resource["data"].append(1)
        assert len(shared_resource["data"]) == 1

    def test_add_item_2(self, shared_resource):
        # Same resource from previous test
        shared_resource["data"].append(2)
        assert len(shared_resource["data"]) == 2

class TestOtherClass:
    def test_fresh_resource(self, shared_resource):
        # New resource for new class
        assert len(shared_resource["data"]) == 0
```

### Example 6: Module Scope

```python
import pytest
from database import Database

@pytest.fixture(scope="module")
def database_connection():
    """Create database connection once per module."""
    print("\nConnecting to database...")
    db = Database()
    db.connect()
    yield db
    print("\nDisconnecting from database...")
    db.disconnect()

def test_query_1(database_connection):
    result = database_connection.query("SELECT 1")
    assert result is not None

def test_query_2(database_connection):
    # Same connection used
    result = database_connection.query("SELECT 2")
    assert result is not None
```

### Example 7: Session Scope

```python
import pytest
import tempfile
import shutil

@pytest.fixture(scope="session")
def test_data_directory():
    """Create test directory once for entire session."""
    temp_dir = tempfile.mkdtemp(prefix="test_data_")
    print(f"\nCreated test directory: {temp_dir}")

    # Populate with test data
    with open(f"{temp_dir}/config.json", "w") as f:
        f.write('{"debug": true}')

    yield temp_dir

    # Cleanup after all tests
    print(f"\nRemoving test directory: {temp_dir}")
    shutil.rmtree(temp_dir)

def test_config_exists(test_data_directory):
    import os
    assert os.path.exists(f"{test_data_directory}/config.json")

def test_config_content(test_data_directory):
    import json
    with open(f"{test_data_directory}/config.json") as f:
        config = json.load(f)
    assert config["debug"] is True
```

## Fixture Dependencies

### Example 8: Simple Dependency Chain

```python
import pytest

@pytest.fixture
def database():
    """Database connection."""
    return {"connected": True, "data": {}}

@pytest.fixture
def user_repository(database):
    """User repository depends on database."""
    class UserRepository:
        def __init__(self, db):
            self.db = db

        def create_user(self, name):
            user_id = len(self.db["data"]) + 1
            self.db["data"][user_id] = {"id": user_id, "name": name}
            return self.db["data"][user_id]

    return UserRepository(database)

@pytest.fixture
def sample_user(user_repository):
    """Sample user depends on user_repository."""
    return user_repository.create_user("Alice")

def test_user_creation(sample_user):
    assert sample_user["name"] == "Alice"
    assert sample_user["id"] == 1

def test_repository_create(user_repository):
    user = user_repository.create_user("Bob")
    assert user["name"] == "Bob"
```

### Example 9: Complex Dependency Graph

```python
import pytest

@pytest.fixture
def config():
    """Application configuration."""
    return {
        "db_host": "localhost",
        "db_port": 5432,
        "api_key": "test-key"
    }

@pytest.fixture
def database(config):
    """Database connection using config."""
    class DB:
        def __init__(self, host, port):
            self.host = host
            self.port = port
            self.connected = True

    return DB(config["db_host"], config["db_port"])

@pytest.fixture
def cache(config):
    """Cache service using config."""
    class Cache:
        def __init__(self):
            self.data = {}

        def get(self, key):
            return self.data.get(key)

        def set(self, key, value):
            self.data[key] = value

    return Cache()

@pytest.fixture
def service(database, cache, config):
    """Service depends on database, cache, and config."""
    class Service:
        def __init__(self, db, cache, api_key):
            self.db = db
            self.cache = cache
            self.api_key = api_key

        def get_data(self, key):
            # Try cache first
            cached = self.cache.get(key)
            if cached:
                return cached

            # Otherwise fetch from database
            data = f"data_for_{key}"
            self.cache.set(key, data)
            return data

    return Service(database, cache, config["api_key"])

def test_service_caching(service):
    # First call - cache miss
    data1 = service.get_data("test")
    assert data1 == "data_for_test"

    # Second call - cache hit
    data2 = service.get_data("test")
    assert data2 == data1

def test_service_has_dependencies(service):
    assert service.db.connected
    assert service.cache is not None
    assert service.api_key == "test-key"
```

### Example 10: Optional Fixture Dependencies

```python
import pytest

@pytest.fixture
def base_config():
    """Base configuration always available."""
    return {"env": "test"}

@pytest.fixture
def extended_config(base_config, request):
    """Extended config with optional additions."""
    config = base_config.copy()

    # Add optional features if markers present
    if "feature_x" in request.keywords:
        config["feature_x"] = True

    if "feature_y" in request.keywords:
        config["feature_y"] = True

    return config

def test_basic_config(extended_config):
    assert extended_config["env"] == "test"
    assert "feature_x" not in extended_config

@pytest.mark.feature_x
def test_with_feature_x(extended_config):
    assert extended_config["feature_x"] is True

@pytest.mark.feature_x
@pytest.mark.feature_y
def test_with_both_features(extended_config):
    assert extended_config["feature_x"] is True
    assert extended_config["feature_y"] is True
```

## Fixture Factories

### Example 11: User Factory

```python
import pytest
from models import User

@pytest.fixture
def make_user():
    """Factory for creating users."""
    created_users = []

    def _make_user(username, email=None, **kwargs):
        if email is None:
            email = f"{username}@example.com"

        user = User(username=username, email=email, **kwargs)
        created_users.append(user)
        return user

    yield _make_user

    # Cleanup all created users
    for user in created_users:
        user.delete()

def test_create_single_user(make_user):
    user = make_user("alice")
    assert user.username == "alice"
    assert user.email == "alice@example.com"

def test_create_multiple_users(make_user):
    alice = make_user("alice", is_admin=True)
    bob = make_user("bob", email="bob@test.com")
    charlie = make_user("charlie", age=25)

    assert alice.is_admin is True
    assert bob.email == "bob@test.com"
    assert charlie.age == 25
```

### Example 12: Object Factory with Counter

```python
import pytest

@pytest.fixture
def make_product():
    """Factory for creating unique products."""
    counter = {"count": 0}

    def _make_product(name=None, price=9.99, **kwargs):
        counter["count"] += 1
        if name is None:
            name = f"Product {counter['count']}"

        return {
            "id": counter["count"],
            "name": name,
            "price": price,
            **kwargs
        }

    return _make_product

def test_unique_products(make_product):
    product1 = make_product()
    product2 = make_product()
    product3 = make_product(name="Custom")

    assert product1["id"] == 1
    assert product2["id"] == 2
    assert product3["id"] == 3

    assert product1["name"] == "Product 1"
    assert product2["name"] == "Product 2"
    assert product3["name"] == "Custom"
```

### Example 13: API Request Factory

```python
import pytest
from unittest.mock import Mock

@pytest.fixture
def make_api_request():
    """Factory for creating mock API requests."""
    def _make_request(method="GET", path="/", status=200, data=None):
        request = Mock()
        request.method = method
        request.path = path
        request.status_code = status
        request.json.return_value = data or {}
        request.text = str(data)
        return request

    return _make_request

def test_successful_request(make_api_request):
    request = make_api_request(
        method="POST",
        path="/api/users",
        status=201,
        data={"id": 1, "name": "Alice"}
    )

    assert request.method == "POST"
    assert request.status_code == 201
    assert request.json()["name"] == "Alice"

def test_error_request(make_api_request):
    request = make_api_request(
        status=404,
        data={"error": "Not found"}
    )

    assert request.status_code == 404
    assert "error" in request.json()
```

## Parametrized Fixtures

### Example 14: Database Type Parametrization

```python
import pytest
from database import SQLiteDB, PostgresDB, MySQLDB

@pytest.fixture(params=[
    "sqlite",
    "postgresql",
    "mysql"
])
def database(request):
    """Test runs three times with different databases."""
    db_type = request.param

    if db_type == "sqlite":
        db = SQLiteDB(":memory:")
    elif db_type == "postgresql":
        db = PostgresDB("localhost", "testdb")
    elif db_type == "mysql":
        db = MySQLDB("localhost", "testdb")

    db.connect()
    yield db
    db.disconnect()

def test_database_insert(database):
    """Runs 3 times: sqlite, postgresql, mysql."""
    database.execute("INSERT INTO users (name) VALUES ('test')")
    result = database.execute("SELECT COUNT(*) FROM users")
    assert result[0][0] == 1

def test_database_transaction(database):
    """Also runs 3 times with each database."""
    with database.transaction():
        database.execute("INSERT INTO users (name) VALUES ('test')")
```

### Example 15: Parametrized Fixtures with IDs

```python
import pytest

@pytest.fixture(params=[
    pytest.param("dev", id="development"),
    pytest.param("staging", id="staging"),
    pytest.param("prod", id="production"),
])
def environment(request):
    """Test different environments."""
    return {
        "name": request.param,
        "debug": request.param == "dev",
        "api_url": f"https://api.{request.param}.example.com"
    }

def test_environment_config(environment):
    """Runs as: test_environment_config[development], etc."""
    assert environment["name"] in ["dev", "staging", "prod"]
    assert environment["api_url"].startswith("https://")

# Output:
# test_environment_config[development] PASSED
# test_environment_config[staging] PASSED
# test_environment_config[production] PASSED
```

### Example 16: Combining Parametrized Fixtures

```python
import pytest

@pytest.fixture(params=["sqlite", "postgres"])
def database_type(request):
    return request.param

@pytest.fixture(params=[10, 100, 1000])
def record_count(request):
    return request.param

def test_database_performance(database_type, record_count):
    """Runs 6 times: 2 databases × 3 record counts."""
    # Simulate database operation
    import time
    start = time.time()

    # Simulate operation
    for i in range(record_count):
        pass  # Insert record

    duration = time.time() - start
    print(f"\n{database_type} with {record_count} records: {duration:.4f}s")
    assert duration < 1.0  # Performance requirement
```

## Autouse Fixtures

### Example 17: Automatic Database Reset

```python
import pytest
from database import db

@pytest.fixture(autouse=True)
def reset_database():
    """Automatically run before each test."""
    db.clear_all_tables()
    db.seed_test_data()
    yield
    # Optional cleanup after test

def test_user_count():
    # Database automatically reset before this test
    assert db.users.count() == 0

def test_create_user():
    # Database automatically reset before this test
    db.users.create(name="Alice")
    assert db.users.count() == 1
```

### Example 18: Automatic Logging Configuration

```python
import pytest
import logging

@pytest.fixture(autouse=True, scope="session")
def configure_logging():
    """Configure logging once for entire test session."""
    logging.basicConfig(
        level=logging.DEBUG,
        format='%(asctime)s - %(name)s - %(levelname)s - %(message)s'
    )
    logger = logging.getLogger()
    logger.info("Test session started")
    yield
    logger.info("Test session completed")

def test_logging_works():
    logger = logging.getLogger(__name__)
    logger.info("Test is running")
    assert True
```

### Example 19: Automatic Test Timing

```python
import pytest
import time

@pytest.fixture(autouse=True)
def measure_test_duration(request):
    """Measure and report test duration."""
    start = time.time()
    yield
    duration = time.time() - start
    print(f"\n{request.node.name} took {duration:.4f} seconds")

def test_quick_operation():
    time.sleep(0.1)
    assert True

def test_slow_operation():
    time.sleep(0.5)
    assert True
```

### Example 20: Autouse with Class Scope

```python
import pytest

@pytest.fixture(autouse=True, scope="class")
def class_setup(request):
    """Setup once for entire test class."""
    print(f"\nSetting up for {request.cls.__name__}")
    request.cls.shared_data = []
    yield
    print(f"\nTearing down for {request.cls.__name__}")

class TestDataOperations:
    def test_append(self):
        # shared_data available without requesting fixture
        self.shared_data.append(1)
        assert len(self.shared_data) == 1

    def test_extend(self):
        # Same shared_data instance
        self.shared_data.extend([2, 3])
        assert len(self.shared_data) == 3
```

## Fixture Finalization

### Example 21: Cleanup with Yield

```python
import pytest
import tempfile
import os

@pytest.fixture
def temp_file():
    """Create and cleanup temporary file."""
    # Setup
    fd, path = tempfile.mkstemp()
    os.write(fd, b"test data")
    os.close(fd)

    yield path

    # Cleanup
    if os.path.exists(path):
        os.unlink(path)

def test_file_exists(temp_file):
    assert os.path.exists(temp_file)
    with open(temp_file, 'rb') as f:
        assert f.read() == b"test data"
# File automatically deleted after test
```

### Example 22: Multiple Cleanup Actions

```python
import pytest

@pytest.fixture
def complex_resource():
    """Resource with multiple cleanup steps."""
    resource = {
        "connection": None,
        "cache": None,
        "temp_files": []
    }

    # Setup
    resource["connection"] = open_connection()
    resource["cache"] = create_cache()

    yield resource

    # Cleanup in reverse order
    for temp_file in resource["temp_files"]:
        os.unlink(temp_file)

    if resource["cache"]:
        resource["cache"].clear()

    if resource["connection"]:
        resource["connection"].close()
```

### Example 23: Conditional Cleanup

```python
import pytest

@pytest.fixture
def database_transaction(request):
    """Database transaction with conditional rollback."""
    db = Database()
    db.begin_transaction()

    yield db

    # Only rollback if test failed
    if request.node.rep_call.failed:
        print("\nTest failed - rolling back transaction")
        db.rollback()
    else:
        print("\nTest passed - committing transaction")
        db.commit()

    db.close()
```

## Database Fixtures

### Example 24: SQLAlchemy Session

```python
import pytest
from sqlalchemy import create_engine
from sqlalchemy.orm import sessionmaker, Session
from models import Base

@pytest.fixture(scope="session")
def engine():
    """Create database engine once per session."""
    engine = create_engine("sqlite:///:memory:")
    Base.metadata.create_all(engine)
    yield engine
    engine.dispose()

@pytest.fixture(scope="function")
def db_session(engine):
    """Create new database session for each test."""
    connection = engine.connect()
    transaction = connection.begin()
    session = Session(bind=connection)

    yield session

    session.close()
    transaction.rollback()
    connection.close()

def test_create_user(db_session):
    from models import User
    user = User(name="Alice", email="alice@example.com")
    db_session.add(user)
    db_session.commit()

    assert user.id is not None
    assert db_session.query(User).count() == 1
```

### Example 25: Database with Sample Data

```python
import pytest
from models import User, Post, Comment

@pytest.fixture
def db_with_users(db_session):
    """Database with sample users."""
    users = [
        User(name="Alice", email="alice@example.com"),
        User(name="Bob", email="bob@example.com"),
        User(name="Charlie", email="charlie@example.com"),
    ]
    db_session.add_all(users)
    db_session.commit()
    return db_session

@pytest.fixture
def db_with_posts(db_with_users):
    """Database with users and posts."""
    users = db_with_users.query(User).all()
    posts = [
        Post(title="First Post", user=users[0]),
        Post(title="Second Post", user=users[1]),
        Post(title="Third Post", user=users[0]),
    ]
    db_with_users.add_all(posts)
    db_with_users.commit()
    return db_with_users

def test_user_posts(db_with_posts):
    alice = db_with_posts.query(User).filter_by(name="Alice").first()
    assert len(alice.posts) == 2
```

### Example 26: MongoDB Fixture

```python
import pytest
from pymongo import MongoClient

@pytest.fixture(scope="session")
def mongo_client():
    """Create MongoDB client for test session."""
    client = MongoClient("localhost", 27017)
    yield client
    client.close()

@pytest.fixture
def mongo_db(mongo_client):
    """Provide clean database for each test."""
    db = mongo_client.test_database
    yield db
    # Cleanup: drop all collections
    for collection_name in db.list_collection_names():
        db.drop_collection(collection_name)

@pytest.fixture
def users_collection(mongo_db):
    """Provide users collection with sample data."""
    collection = mongo_db.users
    collection.insert_many([
        {"name": "Alice", "age": 30},
        {"name": "Bob", "age": 25},
        {"name": "Charlie", "age": 35},
    ])
    return collection

def test_find_users(users_collection):
    users = list(users_collection.find({"age": {"$gte": 30}}))
    assert len(users) == 2
    assert users[0]["name"] in ["Alice", "Charlie"]
```

## API Testing Fixtures

### Example 27: REST API Client

```python
import pytest
import requests

@pytest.fixture(scope="session")
def api_base_url():
    """Base URL for API."""
    return "https://api.example.com"

@pytest.fixture
def api_client(api_base_url):
    """HTTP client for API testing."""
    class APIClient:
        def __init__(self, base_url):
            self.base_url = base_url
            self.session = requests.Session()
            self.token = None

        def authenticate(self, username, password):
            response = self.session.post(
                f"{self.base_url}/auth/login",
                json={"username": username, "password": password}
            )
            self.token = response.json()["access_token"]
            self.session.headers["Authorization"] = f"Bearer {self.token}"

        def get(self, path, **kwargs):
            return self.session.get(f"{self.base_url}{path}", **kwargs)

        def post(self, path, **kwargs):
            return self.session.post(f"{self.base_url}{path}", **kwargs)

        def close(self):
            self.session.close()

    client = APIClient(api_base_url)
    yield client
    client.close()

def test_api_get_users(api_client):
    api_client.authenticate("test_user", "test_pass")
    response = api_client.get("/users")
    assert response.status_code == 200
    assert isinstance(response.json(), list)
```

### Example 28: Mock API Responses

```python
import pytest
from unittest.mock import Mock, patch

@pytest.fixture
def mock_api_success():
    """Mock successful API response."""
    mock_response = Mock()
    mock_response.status_code = 200
    mock_response.json.return_value = {
        "id": 1,
        "name": "Test User",
        "email": "test@example.com"
    }
    return mock_response

@pytest.fixture
def mock_api_error():
    """Mock API error response."""
    mock_response = Mock()
    mock_response.status_code = 404
    mock_response.json.return_value = {
        "error": "Not found"
    }
    mock_response.raise_for_status.side_effect = requests.HTTPError("404 Not Found")
    return mock_response

def test_successful_api_call(mock_api_success):
    with patch('requests.get', return_value=mock_api_success):
        response = requests.get("https://api.example.com/users/1")
        assert response.status_code == 200
        assert response.json()["name"] == "Test User"

def test_api_error_handling(mock_api_error):
    with patch('requests.get', return_value=mock_api_error):
        response = requests.get("https://api.example.com/users/999")
        assert response.status_code == 404
        assert "error" in response.json()
```

### Example 29: GraphQL Client Fixture

```python
import pytest
from gql import Client, gql
from gql.transport.requests import RequestsHTTPTransport

@pytest.fixture(scope="session")
def graphql_client():
    """GraphQL client for testing."""
    transport = RequestsHTTPTransport(
        url="https://api.example.com/graphql",
        headers={"Authorization": "Bearer test-token"}
    )
    client = Client(transport=transport, fetch_schema_from_transport=True)
    yield client

@pytest.fixture
def sample_graphql_query():
    """Sample GraphQL query."""
    return gql("""
        query GetUser($id: ID!) {
            user(id: $id) {
                id
                name
                email
            }
        }
    """)

def test_graphql_query(graphql_client, sample_graphql_query):
    result = graphql_client.execute(
        sample_graphql_query,
        variable_values={"id": "1"}
    )
    assert result["user"]["name"] is not None
```

## File and Directory Fixtures

### Example 30: Temporary Directory with Files

```python
import pytest
from pathlib import Path

@pytest.fixture
def temp_workspace(tmp_path):
    """Create temporary workspace with directory structure."""
    # Create directory structure
    (tmp_path / "src").mkdir()
    (tmp_path / "tests").mkdir()
    (tmp_path / "docs").mkdir()

    # Create some files
    (tmp_path / "README.md").write_text("# Test Project")
    (tmp_path / "src" / "main.py").write_text("print('Hello')")
    (tmp_path / "tests" / "test_main.py").write_text("def test(): pass")

    return tmp_path

def test_workspace_structure(temp_workspace):
    assert (temp_workspace / "src").is_dir()
    assert (temp_workspace / "tests").is_dir()
    assert (temp_workspace / "README.md").is_file()

def test_can_create_new_files(temp_workspace):
    new_file = temp_workspace / "config.json"
    new_file.write_text('{"debug": true}')
    assert new_file.exists()
```

### Example 31: Sample Data Files

```python
import pytest
import json
import csv

@pytest.fixture
def json_data_file(tmp_path):
    """Create JSON data file."""
    data_file = tmp_path / "data.json"
    data = {
        "users": [
            {"id": 1, "name": "Alice", "age": 30},
            {"id": 2, "name": "Bob", "age": 25},
        ],
        "posts": [
            {"id": 1, "user_id": 1, "title": "First Post"},
        ]
    }
    data_file.write_text(json.dumps(data, indent=2))
    return data_file

@pytest.fixture
def csv_data_file(tmp_path):
    """Create CSV data file."""
    csv_file = tmp_path / "data.csv"
    with csv_file.open('w', newline='') as f:
        writer = csv.writer(f)
        writer.writerow(["id", "name", "age"])
        writer.writerow([1, "Alice", 30])
        writer.writerow([2, "Bob", 25])
        writer.writerow([3, "Charlie", 35])
    return csv_file

def test_read_json(json_data_file):
    with json_data_file.open() as f:
        data = json.load(f)
    assert len(data["users"]) == 2
    assert data["users"][0]["name"] == "Alice"

def test_read_csv(csv_data_file):
    with csv_data_file.open() as f:
        reader = csv.DictReader(f)
        rows = list(reader)
    assert len(rows) == 3
    assert rows[0]["name"] == "Alice"
```

### Example 32: Configuration File Fixture

```python
import pytest
import configparser

@pytest.fixture
def config_file(tmp_path):
    """Create configuration file."""
    config = configparser.ConfigParser()
    config['database'] = {
        'host': 'localhost',
        'port': '5432',
        'name': 'testdb'
    }
    config['api'] = {
        'url': 'https://api.example.com',
        'timeout': '30'
    }

    config_path = tmp_path / "config.ini"
    with config_path.open('w') as f:
        config.write(f)

    return config_path

def test_read_config(config_file):
    config = configparser.ConfigParser()
    config.read(config_file)

    assert config['database']['host'] == 'localhost'
    assert config['api']['timeout'] == '30'
```

## Mocking Fixtures

### Example 33: Mock External Service

```python
import pytest
from unittest.mock import Mock, patch

@pytest.fixture
def mock_email_service():
    """Mock email service."""
    service = Mock()
    service.send_email.return_value = {
        "status": "sent",
        "message_id": "msg-12345"
    }
    return service

@pytest.fixture
def mock_payment_gateway():
    """Mock payment gateway."""
    gateway = Mock()
    gateway.process_payment.return_value = {
        "success": True,
        "transaction_id": "tx-67890",
        "amount": 99.99
    }
    return gateway

def test_send_email(mock_email_service):
    result = mock_email_service.send_email(
        to="user@example.com",
        subject="Test",
        body="Test message"
    )

    assert result["status"] == "sent"
    mock_email_service.send_email.assert_called_once()

def test_process_payment(mock_payment_gateway):
    result = mock_payment_gateway.process_payment(
        amount=99.99,
        currency="USD",
        card_number="4111111111111111"
    )

    assert result["success"] is True
    assert result["transaction_id"] is not None
```

### Example 34: Mock with Side Effects

```python
import pytest
from unittest.mock import Mock

@pytest.fixture
def mock_api_with_retries():
    """Mock API that fails then succeeds."""
    api = Mock()

    # First two calls fail, third succeeds
    api.fetch_data.side_effect = [
        ConnectionError("Network timeout"),
        ConnectionError("Network timeout"),
        {"status": "success", "data": [1, 2, 3]}
    ]

    return api

def test_api_retry_logic(mock_api_with_retries):
    # Implement retry logic
    max_retries = 3
    for attempt in range(max_retries):
        try:
            result = mock_api_with_retries.fetch_data()
            break
        except ConnectionError:
            if attempt == max_retries - 1:
                raise

    assert result["status"] == "success"
    assert mock_api_with_retries.fetch_data.call_count == 3
```

### Example 35: Monkeypatch Fixture

```python
import pytest
import os

@pytest.fixture
def mock_environment(monkeypatch):
    """Set up mock environment variables."""
    monkeypatch.setenv("DATABASE_URL", "postgresql://test:test@localhost/testdb")
    monkeypatch.setenv("API_KEY", "test-api-key-12345")
    monkeypatch.setenv("DEBUG", "true")

    return monkeypatch

def test_environment_config(mock_environment):
    assert os.getenv("DATABASE_URL").startswith("postgresql://")
    assert os.getenv("API_KEY") == "test-api-key-12345"
    assert os.getenv("DEBUG") == "true"

@pytest.fixture
def mock_datetime(monkeypatch):
    """Mock datetime.now()."""
    from datetime import datetime

    class MockDatetime:
        @staticmethod
        def now():
            return datetime(2025, 1, 1, 12, 0, 0)

    monkeypatch.setattr("datetime.datetime", MockDatetime)

def test_with_fixed_datetime(mock_datetime):
    from datetime import datetime
    now = datetime.now()
    assert now.year == 2025
    assert now.month == 1
    assert now.day == 1
```

## Complex Fixture Patterns

### Example 36: Fixture Composition

```python
import pytest

@pytest.fixture
def smtp_config():
    return {"host": "smtp.example.com", "port": 587}

@pytest.fixture
def email_templates():
    return {
        "welcome": "Welcome {name}!",
        "reset": "Reset your password: {link}"
    }

@pytest.fixture
def email_service(smtp_config, email_templates):
    """Compose email service from multiple fixtures."""
    class EmailService:
        def __init__(self, config, templates):
            self.config = config
            self.templates = templates

        def send_welcome(self, name):
            return self.templates["welcome"].format(name=name)

        def send_reset(self, link):
            return self.templates["reset"].format(link=link)

    return EmailService(smtp_config, email_templates)

def test_email_service(email_service):
    welcome = email_service.send_welcome("Alice")
    assert "Alice" in welcome

    reset = email_service.send_reset("https://example.com/reset")
    assert "https://example.com/reset" in reset
```

### Example 37: Context Manager Fixture

```python
import pytest
from contextlib import contextmanager

@pytest.fixture
def transaction_manager():
    """Fixture that returns a context manager."""
    @contextmanager
    def transaction():
        print("\nBegin transaction")
        try:
            yield
            print("\nCommit transaction")
        except Exception:
            print("\nRollback transaction")
            raise

    return transaction

def test_successful_transaction(transaction_manager):
    with transaction_manager():
        # Perform operations
        print("\nExecuting operations...")
        assert True
    # Transaction committed

def test_failed_transaction(transaction_manager):
    with pytest.raises(ValueError):
        with transaction_manager():
            print("\nExecuting operations...")
            raise ValueError("Something went wrong")
    # Transaction rolled back
```

### Example 38: Dynamic Fixture Selection

```python
import pytest

@pytest.fixture
def get_storage(request):
    """Return different storage based on marker."""
    if "s3" in request.keywords:
        return {"type": "s3", "bucket": "test-bucket"}
    elif "local" in request.keywords:
        return {"type": "local", "path": "/tmp/storage"}
    else:
        return {"type": "memory", "data": {}}

@pytest.mark.s3
def test_s3_storage(get_storage):
    assert get_storage["type"] == "s3"
    assert "bucket" in get_storage

@pytest.mark.local
def test_local_storage(get_storage):
    assert get_storage["type"] == "local"
    assert "path" in get_storage

def test_default_storage(get_storage):
    assert get_storage["type"] == "memory"
```

### Example 39: Fixture with Request Parameter

```python
import pytest

@pytest.fixture
def user(request):
    """Flexible user fixture based on test needs."""
    # Get parameters from test marker if available
    marker = request.node.get_closest_marker("user_config")
    if marker:
        config = marker.kwargs
    else:
        config = {}

    return {
        "name": config.get("name", "Default User"),
        "role": config.get("role", "user"),
        "permissions": config.get("permissions", [])
    }

def test_default_user(user):
    assert user["name"] == "Default User"
    assert user["role"] == "user"

@pytest.mark.user_config(name="Admin", role="admin", permissions=["read", "write", "delete"])
def test_admin_user(user):
    assert user["name"] == "Admin"
    assert user["role"] == "admin"
    assert "delete" in user["permissions"]
```

### Example 40: Nested Fixtures with Cleanup

```python
import pytest

@pytest.fixture
def outer_resource():
    """Outer fixture with cleanup."""
    print("\nSetup outer resource")
    resource = {"outer": True, "data": []}

    yield resource

    print("\nCleanup outer resource")
    resource["data"].clear()

@pytest.fixture
def middle_resource(outer_resource):
    """Middle fixture depending on outer."""
    print("\nSetup middle resource")
    outer_resource["middle"] = True

    yield outer_resource

    print("\nCleanup middle resource")
    del outer_resource["middle"]

@pytest.fixture
def inner_resource(middle_resource):
    """Inner fixture depending on middle."""
    print("\nSetup inner resource")
    middle_resource["inner"] = True

    yield middle_resource

    print("\nCleanup inner resource")
    del middle_resource["inner"]

def test_with_nested_fixtures(inner_resource):
    """Test showing fixture setup/cleanup order."""
    assert inner_resource["outer"] is True
    assert inner_resource["middle"] is True
    assert inner_resource["inner"] is True

# Execution order:
# Setup outer -> Setup middle -> Setup inner
# Test runs
# Cleanup inner -> Cleanup middle -> Cleanup outer
```

---

**Version**: 1.0.0
**Last Updated**: October 2025

These examples demonstrate the full power and flexibility of pytest fixtures. Use them as templates for your own test suites.
