---
name: moai-lang-cpp
description: >
  Modern C++ (C++23/C++20) development specialist covering RAII, smart pointers, concepts, ranges, modules, and CMake. Use when developing high-performance applications, games, system software, or embedded systems.
license: Apache-2.0
compatibility: Designed for Claude Code
allowed-tools: Read Grep Glob Bash(g++:*) Bash(gcc:*) Bash(clang:*) Bash(clang++:*) Bash(cmake:*) Bash(make:*) Bash(ctest:*) Bash(valgrind:*) Bash(gdb:*) mcp__context7__resolve-library-id mcp__context7__get-library-docs
user-invocable: false
metadata:
  version: "1.1.0"
  category: "language"
  status: "active"
  updated: "2026-01-11"
  modularized: "true"
  tags: "language, cpp, c++23, c++20, cmake, raii, smart-pointers, concepts"

# MoAI Extension: Progressive Disclosure
progressive_disclosure:
  enabled: true
  level1_tokens: 100
  level2_tokens: 5000

# MoAI Extension: Triggers
triggers:
  keywords: ["C++", "cpp", "CMake", "RAII", "smart pointer", "concept", "range", ".cpp", ".hpp", "CMakeLists.txt", "vcpkg", "conan"]
  languages: ["cpp", "c++"]
---

## Quick Reference (30 seconds)

Modern C++ (C++23/C++20) Development Specialist - RAII, smart pointers, concepts, ranges, modules, and CMake.

Auto-Triggers: `.cpp`, `.hpp`, `.h`, `CMakeLists.txt`, `vcpkg.json`, `conanfile.txt`, modern C++ discussions

Core Capabilities:

- C++23 Features: std::expected, std::print, std::generator, deducing this
- C++20 Features: Concepts, Ranges, Modules, Coroutines, std::format
- Memory Safety: RAII, Rule of 5, smart pointers (unique_ptr, shared_ptr, weak_ptr)
- STL: Containers, Algorithms, Iterators, std::span, std::string_view
- Build Systems: CMake 3.28+, FetchContent, presets
- Concurrency: std::thread, std::jthread, std::async, atomics, std::latch/barrier
- Testing: Google Test, Catch2
- Package Management: vcpkg, Conan 2.0

### Quick Patterns

Smart Pointer Factory Pattern: Create a class with a static factory method that returns std::unique_ptr. Include a header for memory, define a Widget class with a static create method taking an int value parameter. The create method uses std::make_unique to instantiate and return the Widget. The constructor should be explicit and take the value parameter, storing it in a private member variable.

Concepts Constraint Pattern: Define a concept named Numeric that combines std::integral or std::floating_point constraints. Create a template function square that requires T to satisfy the Numeric concept, taking a value parameter and returning value multiplied by itself.

Ranges Pipeline Pattern: Use std::views::iota to create a range from 1 to 100, pipe it through a filter to select even numbers, then transform by squaring each value, and finally take the first 10 results.

---

## Implementation Guide (5 minutes)

### C++23 New Features

std::expected for Error Handling: Create an enum class ParseError with InvalidFormat and OutOfRange values. Define a parse_int function that takes std::string_view and returns std::expected containing either int or ParseError. Inside, use a try-catch block to call std::stoi. Catch std::invalid_argument and return std::unexpected with InvalidFormat, catch std::out_of_range and return std::unexpected with OutOfRange. On success, return the parsed value directly. Usage involves checking the result with if(result) and accessing the value with asterisk operator or handling the error case.

std::print for Type-Safe Output: Include the print header and use std::println for formatted output with curly brace placeholders. Supports format specifiers like colon followed by hash x for hexadecimal or colon followed by .2f for floating point precision.

Deducing This (Explicit Object Parameter): Define a Builder class with a data_ member string. Create a template method append with template parameter Self that takes this Self and and a string_view parameter. Forward self with std::forward and return Self and and. This enables chaining on both lvalue and rvalue objects.

### C++20 Features

Concepts and Constraints: Define a Hashable concept using requires expression that checks if std::hash can produce a std::size_t. Create template functions with requires clauses to constrain Container types to std::ranges::range. Use abbreviated syntax with std::integral auto for simple constraints on individual parameters.

Modules: Create a module interface file with .cppm extension. Use export module followed by the module name. Define an export namespace containing template functions. In consumer files, use import statements instead of include directives. Import std for standard library access in module-aware compilers.

Ranges Library: Define structs for data types like Person with name and age fields. Use pipe operator to chain views::filter with a lambda checking conditions, then views::transform to extract fields. Iterate with range-based for loops. Use std::ranges::sort with projections for sorting by member fields.

### RAII and Resource Management

Rule of Five: Implement a Resource class managing a raw pointer and size. The constructor allocates with new array syntax. The destructor calls delete array. Copy constructor allocates new memory and uses std::copy. Copy assignment uses copy-and-swap idiom by creating a temp and calling swap. Move constructor uses std::exchange to take ownership and null the source. Move assignment deletes current data and uses std::exchange. The swap member swaps both pointer and size members.

Smart Pointer Patterns: For unique ownership, create static factory methods returning std::unique_ptr via std::make_unique. For shared ownership with cycles, use std::enable_shared_from_this as a base class. Store children in std::vector of shared_ptr and parent as std::weak_ptr to break reference cycles. Use weak_from_this when setting parent relationships.

### CMake Modern Patterns

CMakeLists.txt Structure for C++23: Begin with cmake_minimum_required VERSION 3.28 and project declaration. Set CMAKE_CXX_STANDARD to 23 with STANDARD_REQUIRED ON. Enable CMAKE_EXPORT_COMPILE_COMMANDS. Use generator expressions for compiler-specific warning flags, checking CXX_COMPILER_ID for GNU, Clang, or MSVC. Use FetchContent to declare dependencies with GIT_REPOSITORY and GIT_TAG parameters. Call FetchContent_MakeAvailable to download and configure. Create libraries with add_library, set include directories with target_include_directories, and link with target_link_libraries. For testing, enable_testing, add test executables, link GTest::gtest_main, and use gtest_discover_tests.

### Concurrency

std::jthread and Stop Tokens: Define worker functions taking std::stop_token parameter. Loop while stop_requested returns false, performing work and sleeping. Create std::jthread objects passing the worker function. Call request_stop to signal termination. The thread destructor automatically joins.

Synchronization Primitives: Use std::latch for one-time synchronization by calling count_down. Use std::barrier for repeated synchronization with arrive_and_wait. Use std::counting_semaphore for resource pools with acquire and release calls.

---

## Advanced Implementation (10+ minutes)

For comprehensive coverage including:

- Template metaprogramming patterns
- Advanced concurrency (thread pools, lock-free data structures)
- Memory management and custom allocators
- Performance optimization (SIMD, cache-friendly patterns)
- Production patterns (dependency injection, factories)
- Extended testing with Google Test and Catch2

See:

- [Advanced Patterns](modules/advanced-patterns.md) - Complete advanced patterns guide

---

## Context7 Library Mappings

- /microsoft/vcpkg - Package manager
- /conan-io/conan - Conan package manager
- /google/googletest - Google Test framework
- /catchorg/Catch2 - Catch2 testing framework
- /fmtlib/fmt - fmt formatting library
- /nlohmann/json - JSON for Modern C++
- /gabime/spdlog - Fast logging library

---

## Works Well With

- `moai-lang-rust` - Systems programming comparison and interop
- `moai-domain-backend` - Backend service architecture
- `moai-workflow-testing` - DDD and testing strategies
- `moai-essentials-debug` - Debugging and profiling
- `moai-foundation-quality` - TRUST 5 quality principles

---

## Troubleshooting

Version Check: Run g++ --version to verify GCC 13+ for C++23 support, clang++ --version for Clang 17+, and cmake --version for CMake 3.28+.

Common Compilation Flags: Use -std=c++23 with -Wall -Wextra -Wpedantic -O2 for standard builds. Add -fsanitize=adddess,undefined -g for debugging builds.

vcpkg Integration: Clone the vcpkg repository from GitHub, run bootstrap-vcpkg.sh, then install packages like fmt, nlohmann-json, and gtest using vcpkg install. Configure CMake with -DCMAKE_TOOLCHAIN_FILE pointing to vcpkg's buildsystems/vcpkg.cmake.

---

Last Updated: 2026-01-11
Status: Active (v1.1.0)
