#!/usr/bin/env -S uv run --script
# /// script
# requires-python = ">=3.10"
# dependencies = [
#     "atlassian-python-api>=3.41.0",
#     "click>=8.1.0",
#     "requests>=2.28.0",
# ]
# ///
"""Interactive Jira credential setup - configure authentication interactively."""

import sys
from pathlib import Path
from urllib.parse import urlparse

# ═══════════════════════════════════════════════════════════════════════════════
# Shared library import (TR1.1.1 - PYTHONPATH approach)
# ═══════════════════════════════════════════════════════════════════════════════
_script_dir = Path(__file__).parent
_lib_path = _script_dir.parent / "lib"
if _lib_path.exists():
    sys.path.insert(0, str(_lib_path.parent))

import click
import requests
from atlassian import Jira
from lib.config import DEFAULT_ENV_FILE
from lib.output import success, error, warning

# ═══════════════════════════════════════════════════════════════════════════════
# Exit Codes
# ═══════════════════════════════════════════════════════════════════════════════
EXIT_SUCCESS = 0
EXIT_USER_ABORT = 1
EXIT_VALIDATION_FAILED = 2


def detect_jira_type(url: str) -> str:
    """Detect if URL is Jira Cloud or Server/Data Center.

    Args:
        url: Jira instance URL

    Returns:
        'cloud' for Atlassian Cloud, 'server' for Server/DC
    """
    parsed = urlparse(url)
    netloc = parsed.netloc.lower()
    # Strict check: domain must be exactly atlassian.net or end with .atlassian.net
    # This prevents bypass via malicious domains like attacker-atlassian.net.evil.com
    if netloc == 'atlassian.net' or netloc.endswith('.atlassian.net'):
        return 'cloud'
    return 'server'


def validate_url(url: str) -> tuple[bool, str]:
    """Validate Jira URL is reachable.

    Args:
        url: URL to validate

    Returns:
        Tuple of (success, message)
    """
    if not url.startswith(('http://', 'https://')):
        return False, "URL must start with http:// or https://"

    try:
        response = requests.head(url, timeout=10, allow_redirects=True)
        if response.status_code < 500:
            return True, f"Server reachable (status: {response.status_code})"
        return False, f"Server error (status: {response.status_code})"
    except requests.exceptions.Timeout:
        return False, "Connection timeout - server did not respond"
    except requests.exceptions.ConnectionError as e:
        return False, f"Connection failed: {e}"


def validate_credentials(url: str, auth_type: str, **kwargs) -> tuple[bool, str]:
    """Validate Jira credentials by attempting authentication.

    Args:
        url: Jira instance URL
        auth_type: 'cloud' or 'server'
        **kwargs: Authentication credentials

    Returns:
        Tuple of (success, message/user_info)
    """
    try:
        if auth_type == 'cloud':
            client = Jira(
                url=url,
                username=kwargs['username'],
                password=kwargs['api_token'],
                cloud=True
            )
        else:
            client = Jira(
                url=url,
                token=kwargs['personal_token']
            )

        user = client.myself()
        display_name = user.get('displayName', user.get('name', 'Unknown'))
        email = user.get('emailAddress', '')
        return True, f"{display_name}" + (f" ({email})" if email else "")

    except Exception as e:
        error_msg = str(e)
        if '401' in error_msg or 'Unauthorized' in error_msg:
            return False, "Authentication failed - invalid credentials"
        if '403' in error_msg or 'Forbidden' in error_msg:
            return False, "Access denied - check permissions"
        return False, f"Connection error: {error_msg}"


def write_env_file(path: Path, config: dict) -> None:
    """Write configuration to environment file.

    Security note: Credentials are stored in clear text, similar to standard
    credential files like ~/.netrc, ~/.npmrc, or ~/.aws/credentials. The file
    is protected by restrictive filesystem permissions (0600 - owner read/write
    only). This is an intentional design choice following common CLI tool patterns.

    Args:
        path: Path to write
        config: Configuration dictionary
    """
    lines = [
        "# Jira CLI Configuration",
        "# Generated by jira-setup.py",
        "# Security: This file contains credentials and is protected by 0600 permissions",
        "",
    ]

    for key, value in config.items():
        if value:
            lines.append(f"{key}={value}")

    lines.append("")

    path.write_text("\n".join(lines))
    path.chmod(0o600)  # Restrict permissions


@click.command()
@click.option('--url', help='Jira instance URL (will prompt if not provided)')
@click.option('--type', 'jira_type', type=click.Choice(['cloud', 'server', 'auto']),
              default='auto', help='Jira deployment type')
@click.option('--output', '-o', type=click.Path(), default=str(DEFAULT_ENV_FILE),
              help=f'Output file path (default: {DEFAULT_ENV_FILE})')
@click.option('--force', '-f', is_flag=True, help='Overwrite existing file without prompting')
@click.option('--test-only', is_flag=True, help='Test credentials without saving')
def main(url: str | None, jira_type: str, output: str, force: bool, test_only: bool):
    """Interactive Jira credential setup.

    Guides you through configuring Jira authentication credentials
    and validates them before saving to ~/.env.jira.

    Supports both Jira Cloud (username + API token) and
    Jira Server/Data Center (Personal Access Token).

    \b
    Examples:
      # Interactive setup
      uv run scripts/core/jira-setup.py

      # Pre-fill URL
      uv run scripts/core/jira-setup.py --url https://company.atlassian.net

      # Test credentials without saving
      uv run scripts/core/jira-setup.py --test-only
    """
    click.echo()
    click.echo("=" * 60)
    click.echo("  Jira Credential Setup")
    click.echo("=" * 60)
    click.echo()

    output_path = Path(output)

    # Check for existing file
    if output_path.exists() and not force and not test_only:
        click.echo(f"⚠ Configuration file already exists: {output_path}")
        if not click.confirm("Do you want to overwrite it?", default=False):
            click.echo("\nSetup cancelled.")
            sys.exit(EXIT_USER_ABORT)
        click.echo()

    # Step 1: Get Jira URL
    click.echo("Step 1: Jira Instance URL")
    click.echo("-" * 40)

    if url:
        click.echo(f"Using provided URL: {url}")
    else:
        click.echo("Enter your Jira instance URL.")
        click.echo("Examples:")
        click.echo("  - https://company.atlassian.net (Jira Cloud)")
        click.echo("  - https://jira.company.com (Jira Server/DC)")
        click.echo()
        url = click.prompt("Jira URL", type=str).strip().rstrip('/')

    # Validate URL
    click.echo()
    click.echo("Validating URL...", nl=False)
    url_ok, url_msg = validate_url(url)
    if url_ok:
        click.echo(f" ✓ {url_msg}")
    else:
        click.echo(f" ✗")
        error(f"URL validation failed: {url_msg}")
        sys.exit(EXIT_VALIDATION_FAILED)

    # Step 2: Detect/confirm Jira type
    click.echo()
    click.echo("Step 2: Authentication Type")
    click.echo("-" * 40)

    if jira_type == 'auto':
        detected = detect_jira_type(url)
        click.echo(f"Detected Jira type: {detected.upper()}")

        if detected == 'cloud':
            click.echo("  → Using Username + API Token authentication")
        else:
            click.echo("  → Using Personal Access Token (PAT) authentication")

        if not click.confirm("Is this correct?", default=True):
            jira_type = click.prompt(
                "Select type",
                type=click.Choice(['cloud', 'server']),
                default=detected
            )
        else:
            jira_type = detected

    click.echo()

    # Step 3: Get credentials
    click.echo("Step 3: Credentials")
    click.echo("-" * 40)

    config = {'JIRA_URL': url}

    if jira_type == 'cloud':
        click.echo("Jira Cloud authentication requires:")
        click.echo("  1. Your Atlassian account email")
        click.echo("  2. An API token (create at https://id.atlassian.com/manage-profile/security/api-tokens)")
        click.echo()

        username = click.prompt("Email address", type=str).strip()
        api_token = click.prompt("API Token", type=str, hide_input=True).strip()

        config['JIRA_USERNAME'] = username
        config['JIRA_API_TOKEN'] = api_token

        # Validate
        click.echo()
        click.echo("Validating credentials...", nl=False)
        cred_ok, cred_msg = validate_credentials(url, 'cloud',
                                                  username=username,
                                                  api_token=api_token)
    else:
        click.echo("Jira Server/Data Center authentication requires:")
        click.echo("  - A Personal Access Token (PAT)")
        click.echo("  - Create one in Jira: Profile → Personal Access Tokens → Create token")
        click.echo()

        personal_token = click.prompt("Personal Access Token", type=str, hide_input=True).strip()

        config['JIRA_PERSONAL_TOKEN'] = personal_token

        # Validate
        click.echo()
        click.echo("Validating credentials...", nl=False)
        cred_ok, cred_msg = validate_credentials(url, 'server',
                                                  personal_token=personal_token)

    if cred_ok:
        click.echo(f" ✓")
        success(f"Authenticated as: {cred_msg}")
    else:
        click.echo(f" ✗")
        error(f"Authentication failed: {cred_msg}")

        if jira_type == 'cloud':
            click.echo()
            click.echo("Troubleshooting tips:")
            click.echo("  1. Verify your email address is correct")
            click.echo("  2. Generate a new API token at:")
            click.echo("     https://id.atlassian.com/manage-profile/security/api-tokens")
            click.echo("  3. Make sure you're using the token, not your password")
        else:
            click.echo()
            click.echo("Troubleshooting tips:")
            click.echo("  1. Create a new PAT in Jira: Profile → Personal Access Tokens")
            click.echo("  2. Ensure the token has not expired")
            click.echo("  3. Check that you have access to the Jira instance")

        sys.exit(EXIT_VALIDATION_FAILED)

    # Step 4: Save configuration
    if test_only:
        click.echo()
        click.echo("=" * 60)
        success("Credentials validated successfully!")
        click.echo("(--test-only mode: not saving to file)")
        sys.exit(EXIT_SUCCESS)

    click.echo()
    click.echo("Step 4: Save Configuration")
    click.echo("-" * 40)

    click.echo(f"Configuration will be saved to: {output_path}")
    click.echo("File permissions will be set to 600 (owner read/write only)")

    if click.confirm("Save configuration?", default=True):
        write_env_file(output_path, config)
        click.echo()
        click.echo("=" * 60)
        success(f"Configuration saved to {output_path}")
        click.echo()
        click.echo("You can now use the Jira CLI scripts:")
        click.echo("  uv run scripts/core/jira-validate.py --verbose")
        click.echo("  uv run scripts/core/jira-issue.py get PROJ-123")
    else:
        click.echo("\nConfiguration not saved.")
        sys.exit(EXIT_USER_ABORT)

    sys.exit(EXIT_SUCCESS)


if __name__ == '__main__':
    main()
