# Baremetal Setup for STM32 with Embedded Swift

Program a STM32 microcontroller directly with low-level Swift code

This tutorial will guide you through setting up a baremetal STM32 project with Embedded Swift to create a simple LED blinking application with text output to UART, the "Hello World" of embedded systems. Concretely, we'll be using the STM32F746G-DISCO (discovery) development board, but the setup will work (with only small tweaks) on most other STM32 devices.

We will be writing a full firmware from scratch including low-level boot code, direct hardware register configuration, a custom linker script and more. Besides educational purposes, this level of control is typically only needed in specialized setups because it involves more work and more complexity. If you want a simpler path to get started, you can integrate with an existing embedded SDK (like STCube or BSPs) instead, which offers higher-level APIs and hardware abstraction. Check out [Integrating with embedded platforms](../SDKSupport/IntegratingWithPlatforms.md) for guidance on common integration patterns.

## Overall plan

The entire baremetal project will consist of:
- A Package.swift file describing the overall structure of the project
- Register definitions for GPIO registers generated by Swift MMIO
- A toolset.json file defining compilation and linking flags
- A Makefile that will serve as a simple shortcut for building and flashing
- A simple linker script
- An interrupt vector and a reset function implementing basic startup code
- Finally, the application logic in Embedded Swift that will set up UART and print "Hello World"

Let's get started!

## Prerequisites

- Mac or Linux
- STM32F746G-DISCO board connected over a USB cable
- Swift toolchain installed using swiftly

## Step 1: Create a New Swift Package

Create a new directory for your project and initialize a Swift package:

```bash
mkdir STM32BlinkLED
cd STM32BlinkLED
swift package init --type executable
```

The initial Swift code that this generates will be in `Sources/STM32BlinkLED/STM32BlinkLED.swift` and we can keep its current content for now:

```swift
// The Swift Programming Language
// https://docs.swift.org/swift-book

@main
struct STM32BlinkLED {
    static func main() {
        print("Hello, world!")
    }
}
```

Let's work on getting this print to actually work and produce text into UART.

## Step 2: Download Arm Toolchain for Embedded

We're going to use the [Arm Toolchain for Embedded](https://github.com/arm/arm-toolchain/tree/arm-software/arm-software/embedded) (formerly called "LLVM Toolchain for ARM") to provide us with basic C-level helper code, concretely memset, memcpy, and an allocator (malloc, free).

Go to [https://github.com/arm/arm-toolchain/releases](https://github.com/arm/arm-toolchain/releases) and download the latest released version of "ATfe" for your host OS:
- On macOS, download e.g. `ATfE-20.1.0-Darwin-universal.dmg`.
- On x86_64 Linux, download e.g. `ATfE-20.1.0-Linux-x86_64.tar.xz`.

Expand and copy out the contents of the toolchain into a subdirectory `llvm-toolchain` of our project. We should end up with a structure of:
```shell
STM32BlinkLED
|- llvm-toolchain
   |- bin/
   |- CHANGELOG.md
   |- docs/
   |- include/
   |- lib/
   |- README.md
   |- ...
|- Sources
|- Package.swift
```

## Step 3: Configure the Package

Edit the `Package.swift` file to configure your project for embedded development, specifically let's use Swift MMIO as a dependency, and let's create two helper targets "Registers" and "Support":

```swift
// swift-tools-version: 5.10
import PackageDescription

let package = Package(
  name: "STM32BlinkLED",
  platforms: [.macOS(.v11)],
  products: [
    .executable(name: "STM32BlinkLED", targets: ["STM32BlinkLED"])
  ],
  dependencies: [
    .package(url: "https://github.com/swiftlang/swift-mmio", branch: "main"),
  ],
  targets: [
    .executableTarget(
      name: "STM32BlinkLED",
      dependencies: ["Registers", "Support"]),
    .target(
      name: "Registers",
      dependencies: [.product(name: "MMIO", package: "swift-mmio")]),
    .target(
      name: "Support"),
  ])
```

Then let's create the respective source directories for the new targets (empty for now):

```shell
$ mkdir Sources/Registers
$ mkdir Sources/Support
$ mkdir Sources/Support/include
```

## Step 4: Generate MMIO register descriptions using SVD2Swift

SVD2Swift is a tool provided by Swift MMIO that automatically generates Swift code from SVD (System View Description) files. SVD files contain detailed descriptions of all the memory-mapped registers in a microcontroller, making them invaluable for embedded development. Using this generated code gives us type-safe access to the hardware registers of the STM32F7 microcontroller.

First, we'll build the SVD2Swift tool from the swift-mmio package we added as a dependency, then download an SVD file for our specific microcontroller, and finally generate the register definitions we need:

```shell
$ swift build --product SVD2Swift
$ curl -L "https://github.com/swiftlang/swift-embedded-examples/raw/refs/heads/main/Tools/SVDs/stm32f7x6.patched.svd" -O
$ .build/debug/SVD2Swift --input stm32f7x6.patched.svd --output Sources/Registers --access-level public \
  --peripherals RCC USART1 GPIOA GPIOB GPIOC GPIOD GPIOE GPIOF GPIOG GPIOH GPIOI GPIOJ GPIOK
```

Tip: If a build fails for any reason, it's often useful to add the `--verbose` flag to `swift build` to see a full list of commands the build system runs.

## Step 5: Create a toolset.json file

Create a `toolset.json` file in the project directory to configure the build with essential settings for ARMv7-based STM32 microcontrollers. This file defines options for the Swift compiler, C compiler, and linker:

```json
{
  "schemaVersion": "1.0",
  "swiftCompiler": {
    "extraCLIOptions": [
      "-enable-experimental-feature", "Embedded",
      "-Xfrontend", "-mergeable-symbols"
    ]
  }
}
```

After this is done, let's attempt a build using `swift build`. We have to specify the path to the toolset file, and also the right target triple:

```shell
$ swift build --configuration release --triple armv7em-none-none-eabi --toolset toolset.json
```

Currently, this should succeed during compilation, but fail to link (because we haven't yet defined a valid linker script for embedded usage):

```shell
$ swift build --configuration release --triple armv7em-none-none-eabi --toolset toolset.json
...
error: link command failed with exit code 1 (use -v to see invocation)
ld.lld: error: unable to find library -lc
ld.lld: error: unable to find library -lm
ld.lld: error: libclang_rt.builtins.a: No such file or directory
clang: error: ld.lld command failed with exit code 1 (use -v to see invocation)
```

## Step 6: Create a Linker Script and Startup Code

Let's now create a linker script, an interrupt vector and startup code.

Create a linker script file named `stm32f4.ld` in the project root directory:

```
MEMORY
{
   flash (rx)      : ORIGIN = 0x08000000, LENGTH = 1024K  /* end: 0x08100000 */
   sram_stack (rw) : ORIGIN = 0x20000000, LENGTH = 32K    /* end: 0x20008000 */
   sram_data (rw)  : ORIGIN = 0x20008000, LENGTH = 160K   /* end: 0x20030000 */
   sram_heap (rw)  : ORIGIN = 0x20030000, LENGTH = 128K   /* end: 0x20050000 */
}

SECTIONS
{
  .text     : { *(.vectors*) ; *(.text*) } > flash
  .rodata   : { *(.rodata*) ; *(.got*) } > flash
   
  __flash_data_start = (. + 3) & ~ 3; /* 4-byte aligned end of text is where data is going to be placed (by elf2hex) */ 

  .bss      : { *(.bss*) } > sram_data
  .tbss      : { *(.tbss*) } > sram_data
  .data     : { *(.data*) } > sram_data

  __flash_data_len   = . - ORIGIN(sram_data);
  
  /DISCARD/ : { *(.swift_modhash*) }  
  /* ARM metadata sections */
  /DISCARD/ : { *(.ARM.attributes*) *(.ARM.exidx) }
  /* ELF metadata sections */
  .symtab   : { *(.symtab) }
  .strtab   : { *(.strtab) }
  .shstrtab : { *(.shstrtab) }
  .debug    : { *(.debug*) }
  .comment  : { *(.comment) }
}

__stack_start      = ORIGIN(sram_stack);
__stack_end        = ORIGIN(sram_stack) + LENGTH(sram_stack);

__data_start      = ORIGIN(sram_data);
__data_end        = ORIGIN(sram_data) + LENGTH(sram_data);

__heap_start      = ORIGIN(sram_heap);
__heap_end        = ORIGIN(sram_heap) + LENGTH(sram_heap);
```

Create startup code in C that defines the interrupt table and reset handler, in `Sources/Support/Startup.c`:

```c
#include <stddef.h>
#include <stdint.h>

void enable_fpu(void) {
  *(volatile uint32_t *)0xE000ED88 |= (0xF << 20); // set CP10 and CP11 Full Access
}

// Reset entrypoint
__attribute__((naked)) __attribute__((noreturn)) void ResetISR(void) {
  asm volatile("bl    enable_fpu");
  asm volatile("ldr   r0, =__data_start // dst");
  asm volatile("ldr   r1, =__flash_data_start // src");
  asm volatile("ldr   r2, =__flash_data_len // size");
  asm volatile("bl    memcpy"); // Relocate data section to RAM
  asm volatile("bl    main");

  // If main returns, spin.
  asm volatile("b     .");
}

void IntDefaultHandlerISR() { __builtin_trap(); }

// These are provided by the linker script
extern void *__stack_start;
extern void *__stack_end;

// Primary interrupt vector table
__attribute__((section(".vectors"))) const void *Vectors[120] = {
    (void *)(((uintptr_t)&__stack_end) - 4), // initial SP
    ResetISR,                                //  1 0x04 The reset handler
    IntDefaultHandlerISR,                    //  2 0x08 The NMI handler
    // All other interrupts are not handled
};

// ELF entrypoint, not actually called at runtime, but it's a GC root
void *_start_elf(void) { return (void *)&Vectors; }
```

Finally, let's update `toolset.json` to configure the linker settings for our embedded target. We'll need to add several important options:
1. The `-T` flag to specify our custom linker script
2. Library paths to the ARM toolchain we downloaded
3. Standard C library and runtime library linkage
4. Disable standard library and stack protector features that aren't appropriate for baremetal code.

```json
{
  "schemaVersion": "1.0",
  "swiftCompiler": {
    "extraCLIOptions": [
      "-enable-experimental-feature", "Embedded",
      "-Xfrontend", "-mergeable-symbols",
      "-Xfrontend", "-disable-stack-protector",
      "-Xclang-linker", "-nostdlib",
    ]
  },
  "linker": {
    "extraCLIOptions": [
      "-T", "stm32f4.ld",
      "-e", "_start_elf",
      "-Lllvm-toolchain/lib/clang-runtimes/arm-none-eabi/armv7m_soft_fpv4_sp_d16_exn_rtti/lib",
      "-lc",
      "-lclang_rt.builtins",
    ]
  }
}
```

At this point, linking will *almost* succeed, the only unresolved reference should be stdout/putchar:

```shell
$ swift build --configuration release --triple armv7em-none-none-eabi --toolset toolset.json
ld.lld: error: undefined symbol: stdout
>>> referenced by putchar.c
>>>               libc_tinystdio_putchar.c.o:(putchar) in archive llvm-toolchain/lib/clang-runtimes/arm-none-eabi/armv7m_soft_fpv4_sp_d16_exn_rtti/lib/libc.a
```

We're not going to actually try to provide the `stdout` symbol, instead let's provide our own custom `putchar` that will be routed to the UART.

## Step 7: Add UART code

Create a new file at `Sources/STM32BlinkLED/UART.swift`:

```swift
import Registers

extension STM32BlinkLED {
  static func initUartOutput() {
    // A9 is UART1 TX, which is relayed by ST-LINK over USB

    // Clock configuration
    rcc.ahb1enr.modify { $0.raw.gpioaen = 1 }  // Enable AHB clock to port A
    rcc.apb2enr.modify { $0.raw.usart1en = 1 }  // Enable APB clock to usart 1

    // Configure A9 as UART1 TX
    gpioa.moder.modify { $0.raw.moder9 = 0b10 }  // Put Pin A9 into alternate function mode
    gpioa.otyper.modify { $0.raw.ot9 = 0b0 }  // Put Pin A9 into push pull
    gpioa.ospeedr.modify { $0.raw.ospeedr9 = 0b00 }  // Put Pin A9 into low speed
    gpioa.pupdr.modify { $0.raw.pupdr9 = 0b00 }  // Disable pull up/down on Pin A9
    gpioa.afrh.modify { $0.raw.afrh9 = 0b0111 }  // Set alternate function usart1 on Pin A9

    // Configure UART1, set the baud rate to 115200 (we boot at 16 MHz)
    usart1.brr.modify { $0.raw.storage = 16_000_000 / 115_200 }

    usart1.cr1.modify {
      $0.raw.ue = 1  // Enable USART 1
      $0.raw.te = 1  // Enable TX
    }
  }
}

func waitTxBufferEmpty() {
  // Spin while tx buffer not empty
  while usart1.isr.read().raw.txe == 0 {}
}

func tx(value: UInt8) {
  usart1.tdr.write { $0.raw.tdr_field = UInt32(value) }
}

@_cdecl("putchar")
public func putchar(_ value: CInt) -> CInt {
  waitTxBufferEmpty()
  tx(value: UInt8(value))
  waitTxBufferEmpty()
  return 0
}
```

## Step 8: Package up and Boot the Firmware

The firmware should now build successfully, and we are finally ready to boot the firmware! We're going to use the `elf2hex` script to convert the ELF file that `swift build` produced into a format that's suitable for flashing, and then we'll use the `st-flash` tool from the opensource stlink package to actually run the firmware. We'll also use the `minicom` program to receive the text over UART, which will be presented on the host system as a serial port (also known as "COM port").

First let's make sure we have st-link and minicom installed:
```shell
$ brew install stlink
$ st-info --probe
... TODO
$ brew install minicom
$ minicom --version
minicom version 2.10 (compiled Feb 22 2025)
Copyright (C) Miquel van Smoorenburg.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version
2 of the License, or (at your option) any later version.
```

Then let's fetch the elf2hex tool:

```shell
$ curl -L "https://raw.githubusercontent.com/swiftlang/swift-embedded-examples/refs/heads/main/Tools/elf2hex.py" -O
$ chmod +x elf2hex.py
```

Next, let's build, package, and flash the firmware:

```shell
$ swift build --configuration release --triple armv7em-none-none-eabi --toolset toolset.json
$ ./elf2hex.py .build/release/STM32BlinkLED .build/release/STM32BlinkLED.hex
```

```shell
$ ls -al .build/release/STM32BlinkLED.hex
-rw-r--r--  1 kuba  staff    13K May 18 10:12 .build/release/STM32BlinkLED.hex
$ st-flash --connect-under-reset --format ihex write .build/release/STM32BlinkLED.hex
... TODO
```

Flashing should succeed and the firmware will run, but at this point, you won't see any UART output yet, and that's expected. While we've created the functions to initialize UART and transmit data, we haven't actually called the initialization routine from our main application. The LED won't blink either since we haven't configured the GPIO pins for it. In the next step, we'll update our main application file to call our UART initialization function and configure the LED pin properly.

## Step 9: Configuring pins for UART and LED

Now let's update the main application file to actually use our UART setup and blink an LED. Open the `Sources/STM32BlinkLED/STM32BlinkLED.swift` file and replace its contents with:

```swift
import Registers

@main
struct STM32BlinkLED {
    static func main() {
        // Initialize UART for output
        initUartOutput()

        // Initialize LED (Pin I1 on STM32F746G-DISCO)
        let ledPin: UInt32 = 1

        // Enable clock for GPIO port I
        rcc.ahb1enr.modify { $0.raw.gpioien = 1 }

        // Configure I1 as output
        gpioi.moder.modify { $0.raw.moder1 = 0b01 } // Output mode
        gpioi.otyper.modify { $0.raw.ot1 = 0b0 }    // Push-pull mode
        gpioi.ospeedr.modify { $0.raw.ospeedr1 = 0b00 } // Low speed

        print("Hello from Embedded Swift on STM32F7!")

        // Main loop - toggle LED and print message
        var count = 0
        while true {
            // Toggle the LED
            let ledState = count % 2 == 0
            gpioi.bsrr.write { 
                if ledState {
                    $0.raw.bs1 = 1  // Set pin (LED on)
                } else {
                    $0.raw.br1 = 1  // Reset pin (LED off)
                }
            }

            // Print status message every iteration
            print("LED is now \(ledState ? "ON" : "OFF") - count: \(count)")

            // Delay using a simple counter
            for _ in 0..<500_000 {
                // Empty loop to create delay
                // This is not accurate timing, just a busy-wait
            }

            count += 1
        }
    }
}
```

Now we need to create a Makefile to simplify the build and flash process. Create a file named `Makefile` in the project root:

```makefile
.PHONY: build flash clean

# Serial port for UART output viewing (change to match your system)
SERIAL_PORT ?= /dev/tty.usbmodem14203

build:
 swift build --configuration release --triple armv7em-none-none-eabi --toolset toolset.json
 ./elf2hex.py .build/release/STM32BlinkLED .build/release/STM32BlinkLED.hex

flash: build
 st-flash --connect-under-reset --format ihex write .build/release/STM32BlinkLED.hex

monitor:
 minicom -D $(SERIAL_PORT) -b 115200

clean:
 swift package clean
 rm -f .build/release/STM32BlinkLED.hex
```

Now you can build and flash your firmware with a single command:

```shell
$ make flash
```

To view the UART output, determine which serial port your STM32F7 board appears as on your system. The ST-Link on the discovery board presents itself as a USB-to-Serial device. Once you've identified the correct port (update the SERIAL_PORT variable in the Makefile if needed), run:

```shell
$ make monitor
```

You should now see "Hello from Embedded Swift on STM32F7!" followed by LED status messages in the minicom terminal, and the LED on your board should be blinking.

## Conclusion

Congratulations! You've successfully set up and programmed an STM32 microcontroller in baremetal mode using Embedded Swift. This simple LED blinking project demonstrates the fundamental concepts of embedded programming:

1. Setting up hardware registers
2. Configuring GPIO pins
3. Creating delay functions
4. Implementing a main loop

From here, you can expand the project to include more complex functionality like interfacing with sensors, implementing communication protocols, or adding user input.

## Next Steps

- Add button input to control the LED patterns
- Implement proper timer-based delays instead of the busy-wait approach
- Add UART communication to send debug messages to your computer
- Explore other peripherals like ADC, I2C, or SPI

Happy embedded programming with Swift!
