# Data Structure - `.pm/` Folder

Complete specification of the `.pm/` folder structure, file formats, and staleness tracking.

---

## Directory Structure

```
.pm/
├── config.md                    # PM preferences and configuration
├── product/
│   ├── inventory.md             # Feature inventory from analysis
│   └── architecture.md          # Technical constraints and moats
├── competitors/
│   ├── _landscape.md            # Market overview (underscore = meta file)
│   └── [competitor-name].md     # Per-competitor profiles
├── gaps/
│   └── [YYYY-MM-DD]-analysis.md # Gap analyses (date-prefixed)
├── requests/
│   └── [issue-id].md            # Local copies of filed requests
└── cache/
    └── last-updated.json        # Staleness tracking
```

---

## File Templates

### config.md

Created on first use of skill. User can edit to customize behavior.

```markdown
# PM Configuration

## Product Positioning

**One-liner:** [Your product in one sentence]

**Target users:** [Primary ICP]

**Key differentiators:**
1. [Differentiator 1]
2. [Differentiator 2]
3. [Differentiator 3]

## Scoring Preferences

### WINNING Thresholds
- **FILE threshold:** 40 (default)
- **WAIT threshold:** 25 (default)

### Staleness Rules
- **Competitor data stale after:** 30 days (default)
- **Gap analysis stale after:** 14 days (default)

## GitHub Integration

**Repository:** [owner/repo]
**Auto-create labels:** true
**Issue prefix:** [PM] (optional)

## Competitors to Track

1. [Competitor 1]
2. [Competitor 2]
3. [Competitor 3]

---
*Last updated: YYYY-MM-DD*
```

---

### product/inventory.md

Generated by `/pm analyze`. Contains feature inventory.

```markdown
# Product Inventory

**Generated:** YYYY-MM-DD
**Codebase:** [repo path]

## Core Value Proposition

[From user interview]

## Target Users

- **Primary:** [ICP]
- **Secondary:** [Other segments]

## Feature Inventory

### Core Features (Table Stakes)

| Feature | Location | Technical Notes |
|---------|----------|-----------------|
| [Feature] | [Path/route] | [Notes] |

### Differentiators (Our USPs)

| Feature | Location | Moat Level | Notes |
|---------|----------|------------|-------|
| [Feature] | [Path] | High/Med/Low | [Why it's a moat] |

### Nice-to-Haves

| Feature | Location | Notes |
|---------|----------|-------|
| [Feature] | [Path] | [Notes] |

## Summary Stats

- **Total features:** X
- **Core/Table Stakes:** X
- **Differentiators:** X
- **Nice-to-Haves:** X

---
*Source: Codebase scan + user interview*
```

---

### product/architecture.md

Generated by `/pm analyze`. Contains technical constraints.

```markdown
# Architecture Overview

**Generated:** YYYY-MM-DD

## Tech Stack

- **Frontend:** [Framework, version]
- **Backend:** [Framework, version]
- **Database:** [Type, version]
- **Infrastructure:** [Cloud provider, services]

## Architectural Constraints

### What's Easy to Change
- [Thing 1] - [Why easy]
- [Thing 2] - [Why easy]

### What's Hard to Change
- [Thing 1] - [Why hard, estimated effort]
- [Thing 2] - [Why hard, estimated effort]

## Technical Moats

| Moat | Description | Defensibility |
|------|-------------|---------------|
| [Moat] | [What it is] | High/Med/Low |

## Technical Debt

| Area | Severity | Impact on New Features |
|------|----------|----------------------|
| [Area] | High/Med/Low | [How it affects development] |

## Dependencies

- **Critical:** [Dependencies that block features]
- **Optional:** [Nice-to-have integrations]

---
*Source: Codebase scan*
```

---

### competitors/_landscape.md

Market overview. The underscore prefix indicates a meta-file.

```markdown
# Competitive Landscape

**Generated:** YYYY-MM-DD

## Market Overview

[Brief description of the market/space]

## Competitors

| Competitor | Positioning | Strength | Weakness |
|------------|-------------|----------|----------|
| [Name] | [One-liner] | [Key strength] | [Key weakness] |

## Market Trends

1. **[Trend]** - [Description and evidence]
2. **[Trend]** - [Description and evidence]

## Our Position

[Where we fit in the landscape]

## Profiles Available

- `[competitor-1].md` - Last updated: YYYY-MM-DD
- `[competitor-2].md` - Last updated: YYYY-MM-DD

---
*Run `/pm scout [name]` to update individual profiles*
```

---

### competitors/[name].md

Per-competitor deep dive. Created by `/pm scout [name]`.

```markdown
# [Competitor Name]

**Website:** [URL]
**Last Updated:** YYYY-MM-DD

## Overview

**Positioning:** [Their one-liner]
**Founded:** [Year]
**Funding:** [Amount/stage if known]
**Team size:** [Approximate]

## Pricing

| Tier | Price | Key Features |
|------|-------|--------------|
| Free | $0 | [Features] |
| Pro | $X/mo | [Features] |
| Enterprise | Custom | [Features] |

## Features

### Tablestakes (Everyone Has)
- [Feature]
- [Feature]

### Differentiators (Their USPs)
- [Feature] - [Why it's unique]
- [Feature] - [Why it's unique]

### Emerging (New Bets)
- [Feature] - [When launched, user reception]

### Deprecated (They Gave Up On)
- [Feature] - [Why they dropped it]

## User Sentiment

**Source:** G2, Capterra, Reddit, Twitter

### What Users Love
- "[Quote]" - [Source]
- "[Quote]" - [Source]

### What Users Hate
- "[Quote]" - [Source]
- "[Quote]" - [Source]

## Recent Activity

| Date | Activity | Signal |
|------|----------|--------|
| [Date] | [What happened] | [What it means] |

## Hiring Signals

**Open roles:** [Key roles that hint at direction]

---
*Data sources: Website, G2, Product Hunt, LinkedIn*
```

---

### gaps/[YYYY-MM-DD]-analysis.md

Gap analysis output. Date-prefixed for history.

```markdown
# Gap Analysis - YYYY-MM-DD

## Summary

- **Product data from:** YYYY-MM-DD (X days old)
- **Competitor data from:** YYYY-MM-DD (X days old)
- **Total gaps identified:** X
- **Recommended to FILE:** X (score ≥40)
- **Recommended to WAIT:** X (score 25-39)
- **Recommended to SKIP:** X (score <25)

## Gaps

### 1. [Gap Name]

**WINNING Score: XX/60** → Recommended: FILE/WAIT/SKIP

| Criterion | Score | Notes |
|-----------|-------|-------|
| Pain Intensity | X/10 | [Evidence] |
| Market Timing | X/10 | [Evidence] |
| Execution Capability | X/10 | [Reasoning] |
| Strategic Fit | X/10 | [Reasoning] |
| Revenue Potential | X/10 | [Reasoning] |
| Competitive Moat | X/10 | [Reasoning] |

**Competitor Evidence:**
- [Competitor A]: [How they do it]
- [Competitor B]: [How they do it]

**User Evidence:**
- [Review/ticket quote]

**Decision:** [ ] FILE  [ ] WAIT  [ ] SKIP

---

### 2. [Gap Name]

...

---

## Decision Summary

| Gap | Score | Decision | Notes |
|-----|-------|----------|-------|
| [Gap 1] | XX/60 | FILE | |
| [Gap 2] | XX/60 | FILE | |
| [Gap 3] | XX/60 | WAIT | Revisit Q2 |
| [Gap 4] | XX/60 | SKIP | Not strategic |

---
*Run `/pm review` to update decisions*
*Run `/pm file` to create GitHub Issues for FILE items*
```

---

### requests/[issue-id].md

Local copy of filed GitHub Issue for offline reference.

```markdown
# [Issue Title]

**GitHub Issue:** #[number]
**Created:** YYYY-MM-DD
**Status:** Open/In Progress/Closed

**Labels:** pm:feature-request, winning:high, priority:now

---

[Full issue body content - copy of GitHub Issue]

---

*Source: gap-analysis-YYYY-MM-DD*
```

---

### cache/last-updated.json

Staleness tracking for all PM data.

```json
{
  "product_inventory": "2025-12-26T10:30:00Z",
  "product_architecture": "2025-12-26T10:30:00Z",
  "competitor_landscape": "2025-12-20T14:00:00Z",
  "competitors": {
    "competitor-a": "2025-12-15T09:00:00Z",
    "competitor-b": "2025-12-18T11:30:00Z"
  },
  "gap_analyses": [
    "2025-12-26",
    "2025-12-10"
  ]
}
```

---

## Staleness Rules

### Default Thresholds

| Data Type | Stale After | Warning Message |
|-----------|-------------|-----------------|
| Competitor profiles | 30 days | "⚠️ [Competitor] data is X days old" |
| Gap analysis | 14 days | "⚠️ Gap analysis is X days old" |
| Product inventory | 60 days | "⚠️ Product inventory may be outdated" |

### Staleness Check Logic

```python
def check_staleness(data_type, last_updated):
    thresholds = {
        "competitor": 30,
        "gap_analysis": 14,
        "product": 60
    }

    days_old = (now - last_updated).days
    threshold = thresholds.get(data_type, 30)

    if days_old > threshold:
        return f"⚠️ {data_type} data is {days_old} days old. Consider refreshing."
    return None
```

### When to Check

- **On `/pm gaps`:** Check all competitor data, warn if any stale
- **On `/pm backlog`:** Check gap analysis date
- **On `/pm standup`:** Summary of all staleness

---

## File Naming Conventions

| Type | Convention | Example |
|------|------------|---------|
| Competitor profiles | kebab-case | `notion.md`, `linear-app.md` |
| Gap analyses | Date prefix | `2025-12-26-analysis.md` |
| Request copies | Issue ID | `42.md`, `feature-43.md` |
| Meta files | Underscore prefix | `_landscape.md` |

---

## Initialization

On first use, create minimal structure:

```bash
mkdir -p .pm/{product,competitors,gaps,requests,cache}
touch .pm/config.md
echo '{}' > .pm/cache/last-updated.json
```

Add to `.gitignore` (optional):
```
# PM skill cache (regeneratable)
.pm/cache/
```

Keep in git (valuable context):
```
.pm/config.md
.pm/product/
.pm/competitors/
.pm/gaps/
.pm/requests/
```
