# Pull Request Template

Use this template when creating pull requests to open source projects. Adapt as needed based on the project's CONTRIBUTING.md guidelines.

---

## Standard PR Template (What/Why/How)

```markdown
## What?
[Brief, clear description of what this PR does - 1-2 sentences]

## Why?
[Explain the reasoning, problem being solved, or business value - 2-3 sentences]

## How?
[Describe the implementation approach and key decisions - bullet points work well]
- Key change 1
- Key change 2
- Key change 3

## Testing
[Step-by-step instructions for reviewers to test the changes]
1. Step 1
2. Step 2
3. Expected result

## Checklist
- [ ] Tests added/updated
- [ ] Documentation updated
- [ ] CI passing
- [ ] Breaking changes documented (if any)
- [ ] Follows project code style

## Related Issues
Closes #[issue number]
Relates to #[issue number]

## Screenshots (if applicable)
[Add screenshots for visual changes, UI updates, or bug fixes]

## Additional Notes
[Any other context, concerns, or questions for reviewers]
```

---

## Example: Feature Addition

```markdown
## What?
Add OAuth2 authentication support for Google and GitHub providers

## Why?
Users have requested social login to reduce friction during signup. This implements Key Result 2 of Q4 OKR1 and addresses multiple user requests from issue #234.

## How?
- Implemented OAuth2 flow using passport.js strategy pattern
- Added provider configuration via environment variables
- Created callback routes for each provider (/auth/google/callback, /auth/github/callback)
- Updated user model to link social accounts with existing email-based accounts
- Added middleware to merge accounts if user already exists

## Testing
1. Set up OAuth apps in Google and GitHub developer consoles
2. Add credentials to `.env` file (see `.env.example` for required variables)
3. Run `npm install` to ensure passport dependencies are installed
4. Start server: `npm start`
5. Navigate to `/login`
6. Click "Login with Google" button
7. Verify OAuth flow redirects correctly
8. Verify user profile data merges correctly with existing account (if applicable)
9. Test GitHub provider following same steps

## Checklist
- [x] Tests added for OAuth flow (see tests/auth/oauth.test.js)
- [x] Documentation updated (see docs/authentication.md)
- [x] CI passing
- [x] No breaking changes
- [x] Follows existing code style
- [x] Environment variables documented in .env.example

## Related Issues
Closes #234
Relates to #156 (social login epic)

## Screenshots
![OAuth Login Buttons](./screenshots/oauth-buttons.png)
*New social login buttons on login page*

## Additional Notes
- Chose passport.js over custom implementation for security and maintenance
- Used strategy pattern to make adding new providers easier in future
- Account merging happens automatically based on email address
- Consider adding rate limiting for OAuth endpoints in future PR
```

---

## Example: Bug Fix

```markdown
## What?
Fix memory leak in worker process shutdown

## Why?
The worker pool wasn't properly cleaning up connections when shutting down gracefully, leading to memory leaks over time. This was causing production issues on long-running instances and reported in #456.

## How?
- Added connection cleanup in worker shutdown handler
- Implemented timeout for graceful shutdown (30s default)
- Added mutex locks to prevent race conditions during cleanup
- Updated tests to verify proper cleanup

## Testing
1. Run load test: `npm run test:load`
2. Monitor memory usage: `npm run monitor:memory`
3. Trigger graceful shutdown: `kill -SIGTERM <pid>`
4. Verify memory is released (check monitor output)
5. Run for 24 hours to verify no gradual memory increase

Manual testing:
- Tested on staging for 48 hours with production load
- Memory usage remained stable at ~120MB
- Previous behavior showed gradual increase to 2GB+ over 24h

## Checklist
- [x] Tests added for shutdown cleanup
- [x] Documentation updated (shutdown behavior in README)
- [x] CI passing
- [x] No breaking changes
- [x] Tested on staging environment

## Related Issues
Fixes #456

## Additional Notes
- Considered force-killing workers after timeout, but went with graceful degradation
- Mutex implementation follows existing pattern in connection-pool.js
- Backward compatible with existing shutdown handlers
```

---

## Example: Refactoring

```markdown
## What?
Refactor authentication middleware to improve testability and reduce duplication

## Why?
The authentication logic was duplicated across 5 different route handlers, making it difficult to test and maintain. This refactoring consolidates the logic into reusable middleware.

## How?
- Extracted auth logic into `middleware/authenticate.js`
- Created composable middleware functions:
  - `requireAuth()` - Basic authentication check
  - `requireRole(role)` - Role-based access control
  - `optionalAuth()` - Sets user if authenticated, continues if not
- Updated all routes to use new middleware
- Maintained backward compatibility with existing behavior

## Testing
1. Run full test suite: `npm test`
2. Verify authentication still works:
   - Try accessing protected route without token (should get 401)
   - Access with valid token (should succeed)
   - Access with invalid token (should get 401)
3. Verify role-based access:
   - User role trying admin endpoint (should get 403)
   - Admin role trying admin endpoint (should succeed)

All existing tests pass without modification, demonstrating backward compatibility.

## Checklist
- [x] All existing tests pass
- [x] Added tests for new middleware functions
- [x] Documentation updated
- [x] CI passing
- [x] No breaking changes
- [x] Code coverage maintained

## Related Issues
Relates to #301 (technical debt epic)

## Additional Notes
- No changes to API contracts - purely internal refactoring
- Reduces code duplication from ~200 lines to ~50 lines
- Future PRs can add new auth strategies more easily
- All 5 route handlers tested individually to verify behavior unchanged
```

---

## Example: Documentation Update

```markdown
## What?
Update installation instructions for v2.0 and add troubleshooting section

## Why?
Users have reported confusion about new installation requirements in v2.0. Multiple support requests (#567, #589, #601) asking about the same issues.

## How?
- Updated README.md with new installation steps
- Added prerequisites section (Node.js 18+, npm 9+)
- Created troubleshooting guide for common issues
- Added examples for different deployment scenarios
- Fixed typos and outdated links

## Testing
- Followed installation steps on fresh Ubuntu 22.04 VM
- Tested on macOS Ventura
- Verified all links work
- Asked non-technical user to follow guide (successfully completed)

## Checklist
- [x] All links verified working
- [x] Code examples tested
- [x] Markdown formatting validated
- [x] Screenshots updated to v2.0 UI
- [x] Spelling and grammar checked

## Related Issues
Closes #567, #589, #601

## Screenshots
N/A (documentation only)

## Additional Notes
- Kept v1.0 migration guide in separate file (MIGRATION.md)
- Added FAQ section based on common support questions
- Consider adding video walkthrough in future
```

---

## Tips for Writing Good PR Descriptions

### Do:
✅ Be specific and clear
✅ Explain WHY, not just WHAT
✅ Provide testing instructions
✅ Link related issues
✅ Include screenshots for visual changes
✅ Note breaking changes prominently
✅ Keep it concise but complete

### Don't:
❌ Just repeat the commit message
❌ Leave sections empty ("TODO", "TBD")
❌ Assume reviewers know the context
❌ Include implementation details better suited for code comments
❌ Write a novel (keep it scannable)
❌ Skip testing instructions

---

## Project-Specific Variations

Some projects have specific PR template requirements. Check for:
- `.github/PULL_REQUEST_TEMPLATE.md` in the repository
- Instructions in CONTRIBUTING.md
- Examples in recently merged PRs

**Always adapt this template to match project expectations.**
