# Files to Exclude from Pull Requests

A comprehensive reference of files that should NEVER be included in open source pull requests, organized by category.

---

## Personal Development Artifacts

### Session & Notes Files
```
❌ SESSION.md               # Session tracking
❌ NOTES.md                 # Development notes
❌ TODO.md                  # Personal todo lists
❌ SCRATCH.md               # Scratch notes
❌ DEBUGGING.md             # Debugging notes
❌ TESTING.md               # Testing notes
❌ JOURNAL.md               # Development journal
❌ IDEAS.md                 # Personal ideas
❌ THOUGHTS.md              # Random thoughts
❌ WIP.md                   # Work in progress notes
```

### Planning Documents
```
❌ planning/                # Entire planning directory
❌ IMPLEMENTATION_PHASES.md # Phase-based planning
❌ DATABASE_SCHEMA.md       # Database planning (unless adding to project)
❌ ARCHITECTURE.md          # Architecture planning (unless adding to project)
❌ API_ENDPOINTS.md         # API planning (unless adding to project)
❌ UI_COMPONENTS.md         # UI planning (unless adding to project)
❌ PROJECT_SPEC.md          # Project specifications
❌ ROADMAP.md               # Personal roadmap
❌ MILESTONES.md            # Personal milestones
```

### Research & Reference
```
❌ research-logs/           # Research directory
❌ references/              # Personal references (skill-specific)
❌ research-*.md            # Research documents
❌ analysis-*.md            # Analysis documents
❌ comparison-*.md          # Comparison documents
❌ evaluation-*.md          # Evaluation documents
```

---

## Screenshots & Visual Assets

### Debug & Development Screenshots
```
❌ screenshots/debug-*.png      # Debugging screenshots
❌ screenshots/test-*.png       # Testing screenshots
❌ screenshots/scratch-*.png    # Scratch screenshots
❌ screenshot-*.png             # Ad-hoc screenshots
❌ screen-recording-*.mp4       # Screen recordings
❌ before-after-local.png       # Local comparisons
❌ demo-local.*                 # Local demos
❌ temp-visual.*                # Temporary visuals
```

### When Screenshots ARE Okay
```
✅ docs/assets/ui-example.png   # Documentation assets
✅ screenshots/feature-demo.png # Demonstrating feature in PR description
✅ docs/images/architecture.png # Architecture diagrams for project docs
```

**Rule of Thumb**: Only include screenshots if they:
1. Demonstrate a feature for the PR description
2. Are part of documentation updates
3. Would be useful to all users/developers

---

## Test Files (Situational)

### Temporary Test Files (NEVER Include)
```
❌ test-manual.js           # Manual testing scripts
❌ test-debug.ts            # Debugging tests
❌ test-quick.py            # Quick validation tests
❌ scratch-test.sh          # Scratch test scripts
❌ example-local.json       # Local test data
❌ quick-test.*             # Quick test files
❌ debug-test.*             # Debug test files
❌ temp-test.*              # Temporary tests
❌ playground.*             # Playground files
❌ experiment.*             # Experimental files
```

### Proper Test Files (Include These)
```
✅ tests/feature.test.js    # Proper test suite
✅ tests/fixtures/data.json # Required test fixtures
✅ __tests__/component.tsx  # Component tests
✅ spec/feature_spec.rb     # RSpec tests
✅ test_feature.py          # Python tests
```

**Rule**: Only include tests that:
1. Are part of the project's test suite structure
2. Follow project's testing conventions
3. Will be run by CI/other developers
4. Test the specific feature/fix in the PR

---

## Build Artifacts & Dependencies

### Build Output
```
❌ dist/                    # Build output
❌ build/                   # Build directory
❌ out/                     # Output directory
❌ target/                  # Rust/Java build directory
❌ bin/                     # Binary output (usually)
❌ lib/                     # Library output (usually)
❌ *.exe, *.dll, *.so       # Compiled binaries
❌ *.o, *.obj               # Object files
❌ *.pyc, *.pyo             # Python compiled files
❌ __pycache__/             # Python cache
❌ .next/                   # Next.js build
❌ .nuxt/                   # Nuxt build
❌ .output/                 # Nitro build
```

### Dependencies
```
❌ node_modules/            # Node dependencies
❌ vendor/                  # Ruby/PHP dependencies
❌ venv/                    # Python virtual environment
❌ .venv/                   # Python virtual environment
❌ env/                     # Environment directory
❌ Cargo.lock               # Rust dependencies (situational)
❌ package-lock.json        # NPM lock (situational)
❌ yarn.lock                # Yarn lock (situational)
❌ pnpm-lock.yaml           # PNPM lock (situational)
```

**Lock File Rule**: Only include lock files if:
- Project explicitly requires them (check CONTRIBUTING.md)
- You're adding/updating dependencies
- Project uses lock files (check existing files in repo)

### Cache & Temporary Build Files
```
❌ .cache/                  # Cache directory
❌ .tmp/                    # Temporary files
❌ .temp/                   # Temporary files
❌ tmp/                     # Temporary directory
❌ temp/                    # Temporary directory
❌ *.cache                  # Cache files
❌ .eslintcache             # ESLint cache
❌ .parcel-cache/           # Parcel cache
❌ .turbo/                  # Turborepo cache
```

---

## IDE & Editor Files

### VS Code
```
❌ .vscode/                 # VS Code settings (use global gitignore)
❌ *.code-workspace         # Workspace files
```

### JetBrains (IntelliJ, WebStorm, etc.)
```
❌ .idea/                   # IntelliJ settings
❌ *.iml                    # Module files
❌ *.ipr                    # Project files
❌ *.iws                    # Workspace files
```

### Vim
```
❌ *.swp                    # Swap files
❌ *.swo                    # Swap files
❌ *~                       # Backup files
❌ .*.sw?                   # Swap files pattern
```

### Emacs
```
❌ *~                       # Backup files
❌ \#*\#                    # Auto-save files
❌ .\#*                     # Lock files
```

### Sublime Text
```
❌ *.sublime-project        # Project files
❌ *.sublime-workspace      # Workspace files
```

**Exception**: If the project provides official IDE configurations (like .vscode/settings.json in the repo), you may update those if needed for the feature.

---

## Operating System Files

### macOS
```
❌ .DS_Store                # Finder metadata
❌ .AppleDouble              # Resource forks
❌ .LSOverride              # Icon metadata
❌ ._*                      # Resource forks
```

### Windows
```
❌ Thumbs.db                # Thumbnail cache
❌ ehthumbs.db              # Thumbnail cache
❌ Desktop.ini              # Folder settings
❌ $RECYCLE.BIN/            # Recycle bin
```

### Linux
```
❌ .directory               # KDE directory settings
❌ .Trash-*/                # Trash directory
```

---

## Secrets & Credentials (CRITICAL!)

### Environment Files
```
❌ .env                     # Environment variables (NEVER!)
❌ .env.local               # Local environment
❌ .env.development         # Development environment
❌ .env.production          # Production environment (NEVER!)
❌ .env.*.local             # Any local env files
❌ config/local.json        # Local configuration
❌ config/secrets.json      # Secrets configuration
```

### Credentials
```
❌ credentials.json         # Credentials file
❌ secrets.json             # Secrets file
❌ auth.json                # Authentication file
❌ token.txt                # Token files
❌ api-keys.json            # API keys
❌ service-account.json     # Service account credentials
```

### Keys & Certificates
```
❌ *.key                    # Private keys
❌ *.pem                    # PEM certificates
❌ *.p12                    # PKCS#12 certificates
❌ *.pfx                    # PFX certificates
❌ id_rsa                   # SSH private key
❌ id_dsa                   # SSH private key
❌ *.crt (sometimes)        # Certificates
```

### Password & Secret Patterns
Look for these in file contents:
```
❌ password=
❌ api_key=
❌ api-key=
❌ apiKey=
❌ secret=
❌ token=
❌ access_token=
❌ private_key=
```

**CRITICAL**: Even if deleted later, secrets in git history are compromised. Use `git filter-branch` or BFG Repo-Cleaner if secrets are committed.

---

## Logs & Debugging

### Log Files
```
❌ *.log                    # Log files
❌ logs/                    # Logs directory
❌ debug.log                # Debug logs
❌ error.log                # Error logs
❌ npm-debug.log            # NPM debug logs
❌ yarn-debug.log           # Yarn debug logs
❌ yarn-error.log           # Yarn error logs
❌ lerna-debug.log          # Lerna debug logs
```

### Debug Files
```
❌ debug-*.js               # Debug scripts
❌ debug-*.py               # Debug scripts
❌ trace-*.txt              # Trace files
❌ profile-*.json           # Profiling output
❌ *.prof                   # Profiling files
❌ *.trace                  # Trace files
```

### Crash Dumps
```
❌ core                     # Core dumps
❌ core.*                   # Core dumps
❌ *.dmp                    # Dump files
❌ crash-*.log              # Crash logs
```

---

## Database & Data Files

### Database Files (Local Development)
```
❌ *.db                     # SQLite databases (local)
❌ *.sqlite                 # SQLite databases (local)
❌ *.sqlite3                # SQLite databases (local)
❌ dump.sql                 # Database dumps
❌ backup.sql               # Database backups
❌ *.mdb                    # Access databases
```

### Data Files (Local/Personal)
```
❌ data/local/              # Local data directory
❌ data/personal/           # Personal data
❌ data/test-data.json      # Test data (unless fixtures)
❌ sample-data-local.json   # Local sample data
```

**Exception**: Include database files if:
- They're part of the project's test fixtures
- They're example/seed data for the project
- Project explicitly includes them (check existing repo)

---

## Coverage & Reports

### Test Coverage
```
❌ coverage/                # Coverage reports
❌ .coverage               # Coverage data
❌ htmlcov/                 # HTML coverage
❌ .nyc_output/             # NYC coverage
❌ lcov.info                # LCOV coverage
```

### Reports
```
❌ reports/                 # Generated reports
❌ test-results/            # Test results
❌ junit.xml                # JUnit reports
❌ cypress/videos/          # Cypress videos
❌ cypress/screenshots/     # Cypress screenshots (unless demonstrating bug)
```

---

## Version Control (Other Than Git)

### SVN
```
❌ .svn/                    # SVN metadata
```

### Mercurial
```
❌ .hg/                     # Mercurial metadata
❌ .hgignore                # Mercurial ignore
```

### CVS
```
❌ CVS/                     # CVS metadata
❌ .cvsignore               # CVS ignore
```

---

## What SHOULD Be Included

For reference, these ARE okay to include:

### Source Code
```
✅ src/                     # Source code
✅ lib/                     # Library code (if source, not compiled)
✅ app/                     # Application code
✅ components/              # Component files
✅ utils/                   # Utility functions
```

### Tests
```
✅ tests/                   # Test directory
✅ __tests__/               # Jest tests
✅ spec/                    # RSpec tests
✅ test_*.py                # Python tests
```

### Documentation
```
✅ README.md                # Project readme
✅ CHANGELOG.md             # Changelog
✅ CONTRIBUTING.md          # Contributing guide
✅ LICENSE                  # License file
✅ docs/                    # Documentation directory
```

### Configuration (Project-level)
```
✅ .gitignore               # Git ignore rules
✅ .eslintrc                # ESLint config (if updating)
✅ .prettierrc              # Prettier config (if updating)
✅ tsconfig.json            # TypeScript config (if updating)
✅ package.json             # NPM package file (if updating)
✅ Cargo.toml               # Rust config (if updating)
✅ pyproject.toml           # Python config (if updating)
```

### CI/CD (if part of feature)
```
✅ .github/workflows/       # GitHub Actions
✅ .gitlab-ci.yml           # GitLab CI
✅ .travis.yml              # Travis CI
✅ Jenkinsfile              # Jenkins
```

### Migrations & Schema (if part of feature)
```
✅ migrations/              # Database migrations
✅ schema.sql               # Database schema (if adding to project)
✅ seeds/                   # Seed data (if part of project)
```

---

## How to Prevent Including These Files

### 1. Project .gitignore
Add patterns that benefit ALL developers:
```gitignore
# Build
dist/
build/
*.pyc

# Dependencies
node_modules/
vendor/

# Logs
*.log

# Secrets
.env
.env.local
*.key
*.pem
```

### 2. Global .gitignore (Recommended)
Add personal/OS-specific patterns:
```bash
# Configure global gitignore
git config --global core.excludesfile ~/.gitignore_global

# Add your patterns
echo ".DS_Store" >> ~/.gitignore_global
echo ".vscode/" >> ~/.gitignore_global
echo ".idea/" >> ~/.gitignore_global
echo "*.swp" >> ~/.gitignore_global
```

### 3. Local Exclusions (.git/info/exclude)
For patterns specific to YOUR workflow only:
```bash
echo "SESSION.md" >> .git/info/exclude
echo "NOTES.md" >> .git/info/exclude
echo "planning/" >> .git/info/exclude
echo "screenshots/debug-*" >> .git/info/exclude
```

**Difference**:
- `.gitignore` → Committed, affects everyone
- `~/.gitignore_global` → Your global settings, affects all your repos
- `.git/info/exclude` → This repo only, not committed

---

## Quick Check Commands

### List all tracked files:
```bash
git ls-files
```

### Check for specific patterns:
```bash
git ls-files | grep -E "SESSION|NOTES|TODO|planning"
```

### Find large files:
```bash
git ls-files | while read file; do
  [ -f "$file" ] && stat -f%z "$file" "$file"
done | sort -rn | head -20
```

### Search for secrets in staged files:
```bash
git diff --cached | grep -iE "password|secret|api[_-]?key|token"
```

### Use the pre-PR check script:
```bash
./scripts/pre-pr-check.sh
```

---

## Resources

- **Pre-PR Check Script**: Automated scanning for these patterns
- **Clean Branch Script**: Remove common artifacts safely
- **GitHub's gitignore templates**: https://github.com/github/gitignore

---

**Remember**: When in doubt, DON'T include it. You can always add files later if needed, but removing them from git history is much harder.
