# Pull Request Submission Checklist

A comprehensive checklist to ensure your PR is ready for review and meets open source contribution standards.

---

## Pre-Contribution Phase

**Before starting any work:**

- [ ] **Read CONTRIBUTING.md** - Found in root, .github/, or docs/
- [ ] **Read CODE_OF_CONDUCT.md** - Understand community expectations
- [ ] **Check if issue exists** - Search existing issues for your topic
- [ ] **Create issue first (if needed)** - For significant changes, discuss before coding
- [ ] **Comment on issue to claim work** - Prevents duplicate effort
  ```
  "Hi! I'd like to work on this. My approach would be to [brief outline]."
  ```
- [ ] **Wait for acknowledgment** - Especially for large changes
- [ ] **Fork the repository** - If you don't have write access
- [ ] **Set up upstream remote**
  ```bash
  git remote add upstream https://github.com/ORIGINAL/repo.git
  ```
- [ ] **Create feature branch** - NEVER work on main/master
  ```bash
  git checkout -b feature/descriptive-name
  ```
- [ ] **Understand testing requirements** - Check what tests are expected
- [ ] **Identify code style tools** - Look for .eslintrc, .prettierrc, etc.

---

## Development Phase

**While coding:**

- [ ] **Follow project style guidelines** - Match existing code patterns
- [ ] **Write tests for new functionality** - Most projects require this
- [ ] **Update tests for changed behavior** - Keep tests in sync with code
- [ ] **Add/update documentation** - README, API docs, inline comments
- [ ] **Keep commits atomic** - One logical change per commit
- [ ] **Write good commit messages** - Follow Conventional Commits format
  ```
  feat(scope): brief description

  Longer explanation if needed

  Fixes #123
  ```
- [ ] **Run linters and formatters** - Fix style issues during development
  ```bash
  npm run lint
  npm run format
  ```
- [ ] **Test locally frequently** - Don't wait until the end
- [ ] **Keep PR scope focused** - One feature/fix per PR
- [ ] **Sync with upstream regularly** - Avoid merge conflicts
  ```bash
  git fetch upstream
  git rebase upstream/main
  ```

---

## Pre-Submission Phase

### Code Quality

- [ ] **All tests pass locally**
  ```bash
  npm test
  # or
  pytest
  # or
  cargo test
  # or project's test command
  ```
- [ ] **Code builds successfully**
  ```bash
  npm run build
  ```
- [ ] **Linter passes**
  ```bash
  npm run lint
  ```
- [ ] **Formatter applied**
  ```bash
  npm run format
  ```
- [ ] **Code coverage maintained** - If project has minimum thresholds
- [ ] **No compiler warnings** - Clean build output
- [ ] **Manual testing completed** - Test the actual functionality

### Code Review (Self)

- [ ] **Review your own diff**
  ```bash
  git diff origin/main
  ```
- [ ] **No debugging code** - Remove console.logs, debugger statements
- [ ] **No commented-out code** - Remove dead code
- [ ] **No TODO comments** - Complete work or create follow-up issues
- [ ] **Consistent naming** - Variables, functions, files match conventions
- [ ] **Proper error handling** - Don't swallow errors silently
- [ ] **Edge cases considered** - Null checks, empty arrays, etc.

### Cleanup (Critical!)

Run the pre-PR check script:
```bash
./scripts/pre-pr-check.sh
```

- [ ] **Remove SESSION.md** - Personal session tracking
- [ ] **Remove NOTES.md** - Development notes
- [ ] **Remove TODO.md** - Personal todo lists
- [ ] **Remove planning/* directory** - Project planning documents
- [ ] **Remove debug screenshots** - Screenshots used during debugging
  - Keep only screenshots demonstrating features for PR description
- [ ] **Remove temporary test files**
  - test-manual.js, test-debug.ts, quick-test.py
- [ ] **Remove personal workflow files**
  - scratch.*, temp.*, debug.*
- [ ] **No IDE/editor files**
  - .vscode/, .idea/, *.swp
  - Should be in global .gitignore, not committed
- [ ] **No OS-specific files**
  - .DS_Store, Thumbs.db, desktop.ini
- [ ] **No build artifacts**
  - dist/, build/, node_modules/, __pycache__/
- [ ] **No secrets or credentials**
  - .env, credentials.json, *.key, *.pem
  - Double-check with: `git diff | grep -i "password\|secret\|key"`
- [ ] **No large binary files** - Unless necessary for the PR
- [ ] **No unrelated changes** - Only changes relevant to this PR

### Git Hygiene

- [ ] **Commits are clean** - No "WIP" or "fix typo" commits
  - Consider squashing if needed
- [ ] **Commit messages follow conventions** - Conventional Commits format
- [ ] **No merge conflicts** - Rebase on latest upstream/main
  ```bash
  git fetch upstream
  git rebase upstream/main
  ```
- [ ] **Branch is up to date** - Latest changes from main included
- [ ] **On feature branch** - Not on main/master

---

## PR Creation Phase

### PR Description

- [ ] **Title follows conventions** - Conventional Commits format
  ```
  feat(auth): add OAuth2 support
  fix(api): resolve memory leak
  docs(readme): update installation
  ```
- [ ] **Uses What/Why/How structure**
  - What: Brief description of changes
  - Why: Reasoning and context
  - How: Implementation approach
- [ ] **Testing instructions included** - Step-by-step how to test
- [ ] **Screenshots for visual changes** - Before/after if applicable
- [ ] **Breaking changes noted** - If any
- [ ] **Links to related issues** - Use closing keywords
  ```
  Closes #123
  Fixes #456
  Relates to #789
  ```
- [ ] **Checklist included** - Tests, docs, CI status
- [ ] **Description is clear** - Reviewer can understand without asking questions

### PR Quality

- [ ] **PR is reasonably sized**
  - Ideal: < 50 lines
  - Good: < 200 lines
  - Max: < 400 lines
  - If larger, explain why or split into multiple PRs
- [ ] **PR is focused** - One feature/fix/refactor, not multiple unrelated changes
- [ ] **No unrelated changes** - No "drive-by fixes" in unrelated files
- [ ] **All changed files are intentional** - Review git status

### GitHub Settings

- [ ] **Pushed to feature branch on fork**
  ```bash
  git push origin feature/my-feature
  ```
- [ ] **PR targets correct branch** - Usually main or develop
- [ ] **Assigned labels** - If you have permission
- [ ] **Requested reviewers** - If known and appropriate
- [ ] **Linked to project/milestone** - If applicable
- [ ] **Set as draft** - If not ready for full review yet
  ```bash
  gh pr create --draft
  ```

---

## Post-Submission Phase

### Monitor CI/Checks

- [ ] **CI is running** - Green checkmarks appearing
- [ ] **All checks pass** - No failures
- [ ] **Fix failures immediately** - Don't wait for review if CI fails
- [ ] **Watch for build notifications** - Email/GitHub notifications

### Communication

- [ ] **Responsive to feedback** - Reply within 24-48 hours
- [ ] **Address all review comments** - Even if just "Acknowledged"
- [ ] **Mark conversations resolved** - After addressing feedback
- [ ] **Request re-review** - After making changes
  ```bash
  gh pr ready  # if was draft
  ```
- [ ] **Thank reviewers** - Shows appreciation for their time
- [ ] **Professional tone** - Courteous and respectful
- [ ] **Ask for clarification** - If feedback is unclear
- [ ] **Be patient** - Reviewers are often volunteers

### Updates

- [ ] **Keep PR updated** - Rebase if main moves forward
  ```bash
  git fetch upstream
  git rebase upstream/main
  git push origin feature/my-feature --force-with-lease
  ```
- [ ] **Fix requested changes** - Implement feedback
- [ ] **Update documentation** - If requirements change
- [ ] **Squash commits** - If maintainer requests

---

## Ready to Submit?

**Final verification:**

```bash
# 1. Run pre-PR check
./scripts/pre-pr-check.sh

# 2. Review changes
git status
git diff origin/main --stat

# 3. Verify tests pass
npm test

# 4. Verify build succeeds
npm run build

# 5. Check commit messages
git log --oneline -5

# 6. Push to your fork
git push origin feature/my-feature

# 7. Create PR
gh pr create --fill
# or
gh pr create --title "feat: ..." --body "$(cat pr-description.md)"
```

---

## Common Mistakes Checklist

Avoid these common errors:

- [ ] ❌ Not reading CONTRIBUTING.md
- [ ] ❌ Including personal artifacts (SESSION.md, planning/*)
- [ ] ❌ Submitting massive PR (>400 lines)
- [ ] ❌ Not testing before submission
- [ ] ❌ Working on already assigned issue
- [ ] ❌ Not discussing large changes first
- [ ] ❌ Being impatient or unresponsive
- [ ] ❌ Not updating documentation
- [ ] ❌ Ignoring code style
- [ ] ❌ Ignoring CI failures
- [ ] ❌ Including unrelated changes
- [ ] ❌ Not linking issues properly
- [ ] ❌ Committing secrets
- [ ] ❌ Force-pushing without warning
- [ ] ❌ Working on main/master branch

---

## Project-Specific Checklist

Add project-specific items here based on CONTRIBUTING.md:

- [ ] _[Project-specific requirement 1]_
- [ ] _[Project-specific requirement 2]_
- [ ] _[Project-specific requirement 3]_

---

## Quick Reference: Essential Commands

```bash
# Setup
git clone https://github.com/YOUR-USERNAME/repo.git
git remote add upstream https://github.com/ORIGINAL/repo.git
git checkout -b feature/my-feature

# Development
npm run lint
npm test
npm run build
git add .
git commit -m "feat(scope): description"

# Pre-submission
./scripts/pre-pr-check.sh
git status
git diff origin/main

# Submission
git push origin feature/my-feature
gh pr create --fill

# After feedback
git fetch upstream
git rebase upstream/main
# make changes
git push origin feature/my-feature --force-with-lease
```

---

## Resources

- **Pre-PR Check Script**: `./scripts/pre-pr-check.sh`
- **Clean Branch Script**: `./scripts/clean-branch.sh`
- **PR Template**: `./references/pr-template.md`
- **Commit Message Guide**: `./references/commit-message-guide.md`
- **Files to Exclude**: `./references/files-to-exclude.md`

---

**Remember**: The goal is to make the maintainer's job as easy as possible. A well-prepared PR shows respect for their time and increases the likelihood of quick acceptance.
