#!/bin/bash

# Clean Branch Script
# Safely removes common personal development artifacts before PR submission
# Part of the open-source-contributions skill

set -e

# Colors for output
RED='\033[0;31m'
YELLOW='\033[1;33m'
GREEN='\033[0;32m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

echo -e "${BLUE}====================================${NC}"
echo -e "${BLUE}  Clean Branch for PR${NC}"
echo -e "${BLUE}====================================${NC}"
echo ""

# Check if we're in a git repository
if ! git rev-parse --git-dir > /dev/null 2>&1; then
    echo -e "${RED}✗ Not in a git repository${NC}"
    exit 1
fi

# Interactive mode flag
INTERACTIVE=true
if [ "$1" = "--non-interactive" ] || [ "$1" = "-y" ]; then
    INTERACTIVE=false
fi

# Files to potentially remove
PERSONAL_FILES=(
    "SESSION.md"
    "NOTES.md"
    "TODO.md"
    "SCRATCH.md"
    "DEBUGGING.md"
    "TESTING.md"
)

FOUND_FILES=()

echo -e "${BLUE}Scanning for personal development artifacts...${NC}"
echo ""

# Check for personal files
for file in "${PERSONAL_FILES[@]}"; do
    if git ls-files --error-unmatch "$file" > /dev/null 2>&1; then
        FOUND_FILES+=("$file")
        echo -e "${YELLOW}  Found: $file${NC}"
    fi
done

# Check for planning directory
if git ls-files | grep -q "^planning/"; then
    FOUND_FILES+=("planning/")
    echo -e "${YELLOW}  Found: planning/* directory${NC}"
    git ls-files | grep "^planning/" | head -5 | sed 's/^/    - /'
    PLANNING_COUNT=$(git ls-files | grep "^planning/" | wc -l)
    if [ "$PLANNING_COUNT" -gt 5 ]; then
        echo -e "${YELLOW}    ... and $((PLANNING_COUNT - 5)) more files${NC}"
    fi
fi

# Check for debug screenshots
DEBUG_SCREENSHOTS=$(git ls-files | grep -E "(debug|test|scratch).*\.(png|jpg|jpeg|gif)" || true)
if [ -n "$DEBUG_SCREENSHOTS" ]; then
    echo -e "${YELLOW}  Found: debug/test screenshots${NC}"
    echo "$DEBUG_SCREENSHOTS" | sed 's/^/    - /'
    FOUND_FILES+=("screenshots-debug")
fi

# Check for temporary test files
TEMP_TESTS=$(git ls-files | grep -iE "(test-manual|test-debug|quick-test|scratch-test|example-local)" || true)
if [ -n "$TEMP_TESTS" ]; then
    echo -e "${YELLOW}  Found: temporary test files${NC}"
    echo "$TEMP_TESTS" | sed 's/^/    - /'
    FOUND_FILES+=("temp-tests")
fi

echo ""

# If nothing found, exit
if [ ${#FOUND_FILES[@]} -eq 0 ]; then
    echo -e "${GREEN}✓ No personal artifacts found!${NC}"
    echo -e "${GREEN}Your branch is clean.${NC}"
    exit 0
fi

# Confirm removal
if [ "$INTERACTIVE" = true ]; then
    echo -e "${YELLOW}These files should not be included in your PR.${NC}"
    echo -e "${BLUE}Would you like to remove them? (y/n)${NC}"
    read -r response
    if [[ ! "$response" =~ ^[Yy]$ ]]; then
        echo -e "${YELLOW}Cancelled. No files removed.${NC}"
        exit 0
    fi
fi

echo ""
echo -e "${BLUE}Removing files...${NC}"

REMOVED_COUNT=0

# Remove personal files
for file in "${PERSONAL_FILES[@]}"; do
    if git ls-files --error-unmatch "$file" > /dev/null 2>&1; then
        git rm --cached "$file" 2>/dev/null || git rm "$file" 2>/dev/null || true
        echo -e "${GREEN}  ✓ Removed: $file${NC}"
        ((REMOVED_COUNT++))
    fi
done

# Remove planning directory
if git ls-files | grep -q "^planning/"; then
    git rm --cached -r "planning/" 2>/dev/null || git rm -r "planning/" 2>/dev/null || true
    echo -e "${GREEN}  ✓ Removed: planning/* directory${NC}"
    ((REMOVED_COUNT++))
fi

# Remove debug screenshots (interactive)
if [ -n "$DEBUG_SCREENSHOTS" ]; then
    if [ "$INTERACTIVE" = true ]; then
        echo ""
        echo -e "${YELLOW}Found debug screenshots. Remove these too? (y/n)${NC}"
        echo "$DEBUG_SCREENSHOTS" | sed 's/^/    - /'
        read -r response
        if [[ "$response" =~ ^[Yy]$ ]]; then
            echo "$DEBUG_SCREENSHOTS" | while read -r file; do
                git rm --cached "$file" 2>/dev/null || git rm "$file" 2>/dev/null || true
                echo -e "${GREEN}  ✓ Removed: $file${NC}"
            done
            ((REMOVED_COUNT++))
        fi
    else
        echo "$DEBUG_SCREENSHOTS" | while read -r file; do
            git rm --cached "$file" 2>/dev/null || git rm "$file" 2>/dev/null || true
        done
        echo -e "${GREEN}  ✓ Removed: debug screenshots${NC}"
        ((REMOVED_COUNT++))
    fi
fi

# Remove temporary test files (interactive)
if [ -n "$TEMP_TESTS" ]; then
    if [ "$INTERACTIVE" = true ]; then
        echo ""
        echo -e "${YELLOW}Found temporary test files. Remove these too? (y/n)${NC}"
        echo "$TEMP_TESTS" | sed 's/^/    - /'
        read -r response
        if [[ "$response" =~ ^[Yy]$ ]]; then
            echo "$TEMP_TESTS" | while read -r file; do
                git rm --cached "$file" 2>/dev/null || git rm "$file" 2>/dev/null || true
                echo -e "${GREEN}  ✓ Removed: $file${NC}"
            done
            ((REMOVED_COUNT++))
        fi
    else
        echo "$TEMP_TESTS" | while read -r file; do
            git rm --cached "$file" 2>/dev/null || git rm "$file" 2>/dev/null || true
        done
        echo -e "${GREEN}  ✓ Removed: temporary test files${NC}"
        ((REMOVED_COUNT++))
    fi
fi

echo ""
echo -e "${BLUE}====================================${NC}"

if [ $REMOVED_COUNT -gt 0 ]; then
    echo -e "${GREEN}✓ Cleaned $REMOVED_COUNT artifact(s)${NC}"
    echo ""
    echo -e "${BLUE}Next steps:${NC}"
    echo -e "  1. Review changes: ${YELLOW}git status${NC}"
    echo -e "  2. Commit removal: ${YELLOW}git commit -m 'chore: remove personal development artifacts'${NC}"
    echo -e "  3. Re-run check: ${YELLOW}./scripts/pre-pr-check.sh${NC}"
    echo -e "  4. Push changes: ${YELLOW}git push${NC}"
else
    echo -e "${GREEN}✓ No files removed${NC}"
fi

echo ""
echo -e "${YELLOW}Note:${NC} Files are removed from git tracking but still exist locally."
echo -e "Add them to .git/info/exclude to prevent re-adding:"
echo ""
echo -e "${BLUE}echo 'SESSION.md' >> .git/info/exclude${NC}"
echo -e "${BLUE}echo 'planning/' >> .git/info/exclude${NC}"
