# React Native Version Decision Tree

**Last Updated:** 2025-11-22
**Purpose:** Help choose the right React Native version for your project

---

## Quick Decision Flowchart

```
START: What type of project?
│
├─ NEW PROJECT (starting from scratch)
│  │
│  └─ React Native 0.82+ ✅
│     ├─ New Architecture mandatory (no legacy)
│     ├─ Hermes only (JSC moving to community)
│     ├─ React 19 (no propTypes, no forwardRef)
│     ├─ Swift iOS template (default)
│     └─ Latest CSS properties
│
└─ EXISTING PROJECT (migrating)
   │
   ├─ Currently on 0.75 or EARLIER?
   │  │
   │  ├─ YES → Upgrade to 0.76-0.81 FIRST ⚠️
   │  │  ├─ Why: Get interop layer for migration
   │  │  ├─ Test with New Architecture enabled
   │  │  ├─ Fix incompatible dependencies
   │  │  └─ THEN upgrade to 0.82+
   │  │
   │  └─ NO (already on 0.76+) → Continue to next question
   │
   ├─ Are ALL dependencies compatible with New Architecture?
   │  │
   │  ├─ YES → Upgrade to 0.82+ ✅
   │  │
   │  ├─ NO → Do you NEED those incompatible dependencies?
   │  │  │
   │  │  ├─ YES, CRITICAL → Stay on 0.76-0.81 (interop layer)
   │  │  │  ├─ Legacy architecture frozen (no updates)
   │  │  │  ├─ Find alternatives if possible
   │  │  │  └─ Plan migration before 0.81 EOL
   │  │  │
   │  │  └─ NO, can replace → Replace with compatible libraries ✅
   │  │     └─ Then upgrade to 0.82+
   │  │
   │  └─ UNSURE → Use scripts/check-rn-version.sh
   │     └─ It will detect incompatible dependencies
   │
   └─ Are you using Expo?
      │
      ├─ YES → Use Expo SDK 52+ (React Native 0.76+)
      │  ├─ Expo Go: New Architecture required
      │  ├─ Hermes only (JSC removed)
      │  └─ Custom dev client: More flexibility
      │
      └─ NO → Continue with version choice above
```

---

## Version Comparison Table

| Feature | 0.72-0.75 | 0.76-0.81 | 0.82+ |
|---------|-----------|-----------|-------|
| **New Architecture** | Optional, opt-in | Default, can disable | **Mandatory** |
| **Legacy Architecture** | Default | Frozen, deprecated | ❌ **Removed** |
| **Interop Layer** | N/A | ✅ Available | ❌ Gone |
| **React Version** | 18.x | 18.x / 19.x | 19.1+ |
| **propTypes Support** | ✅ Yes | ✅ Yes (0.76-0.77) / ❌ No (0.78+) | ❌ **Removed** |
| **forwardRef** | ✅ Required | ✅ Required (0.76-0.77) / ⚠️ Deprecated (0.78+) | ⚠️ **Deprecated** |
| **iOS Template** | Objective-C | Objective-C (0.76) / Swift (0.77+) | Swift |
| **JSC Engine** | ✅ Bundled | ✅ Bundled | ⚠️ Community package |
| **Metro Logs** | ✅ Works | ✅ Works (0.76) / ❌ Removed (0.77+) | ❌ **Removed** |
| **Chrome Debugger** | ✅ Works | ⚠️ Deprecated (0.79+) | ❌ **Removed** |
| **CSS: display: contents** | ❌ No | ✅ Yes (0.77+) | ✅ Yes |
| **CSS: mixBlendMode** | ❌ No | ✅ Yes (0.77+) | ✅ Yes |
| **CSS: outline** | ❌ No | ✅ Yes (0.77+) | ✅ Yes |
| **Android XML Drawables** | ❌ No | ✅ Yes (0.78+) | ✅ Yes |
| **Deep Imports** | ✅ Works | ⚠️ Deprecated (0.80+) | ⚠️ **Deprecated** |

---

## Detailed Version Guide

### React Native 0.82+ (Latest)

**Choose this if:**
- ✅ Starting a new project
- ✅ All dependencies support New Architecture
- ✅ Want latest features and performance
- ✅ Ready for React 19 migration

**Pros:**
- Fastest performance (Hermes V1 experimental)
- Latest CSS properties
- Active support and updates
- No legacy baggage

**Cons:**
- New Architecture mandatory (can't disable)
- Some libraries may not be compatible yet
- React 19 breaking changes (propTypes removed)
- Steeper migration if coming from old version

**Best for:**
- New projects
- Apps with modern dependencies
- Teams comfortable with bleeding edge

---

### React Native 0.76-0.81 (Transition)

**Choose this if:**
- ✅ Migrating from 0.75 or earlier
- ✅ Need interop layer during migration
- ✅ Have some incompatible dependencies
- ✅ Want time to test New Architecture

**Pros:**
- Interop layer helps migration
- Can disable New Architecture (if needed)
- Access to new CSS features (0.77+)
- Time to fix dependency issues

**Cons:**
- Legacy architecture frozen (no updates)
- Short support window (transitional release)
- Will need to upgrade to 0.82+ eventually
- Missing some latest features

**Best for:**
- Migration projects
- Large codebases with many dependencies
- Teams needing gradual migration path

**Warning:** Do NOT stay on 0.76-0.81 long-term. Plan to upgrade to 0.82+ within 3-6 months.

---

### React Native 0.72-0.75 (Legacy)

**Choose this if:**
- ⚠️ Stuck with incompatible dependencies
- ⚠️ Cannot migrate to New Architecture yet
- ⚠️ Need stable version for critical app

**Pros:**
- Mature, stable
- Most libraries compatible
- Well-documented issues

**Cons:**
- Legacy architecture (deprecated)
- Missing new features
- Security/bug fixes only (no new features)
- End of life approaching

**Best for:**
- Maintaining old apps
- Very short-term only

**Warning:** Avoid for new projects. Plan migration to 0.76+ ASAP.

---

## Migration Paths

### Path 1: New Project (Recommended)

```bash
# Start with latest
npx create-expo-app@latest my-app
cd my-app

# Verify versions
npm list react-native react

# Should see:
# react-native@0.82.x
# react@19.1.x

# ✅ You're done! New Architecture enabled by default
```

---

### Path 2: Old Project → Modern (Safe Migration)

```bash
# Step 1: Check current version
npx react-native --version
# Example: 0.72.0

# Step 2: Upgrade to 0.81 (last interop version)
npm install react-native@0.81
npx expo install --fix  # If using Expo

# Step 3: Enable New Architecture (test mode)
# Android: gradle.properties
newArchEnabled=true

# iOS
RCT_NEW_ARCH_ENABLED=1 pod install

# Step 4: Test thoroughly
npm run ios
npm run android
# Test ALL features, navigation, state management, etc.

# Step 5: Fix incompatible dependencies
# See: references/new-architecture-errors.md
# Common fixes:
#   - Redux → Redux Toolkit
#   - i18n-js → react-i18next
#   - Update React Navigation

# Step 6: Migrate to React 19 (if upgrading to 0.78+)
npx @codemod/react-19 upgrade

# Step 7: Upgrade to 0.82+ (final step)
npm install react-native@0.82 react@19
npx expo install --fix

# Step 8: Test again
npm run ios
npm run android
```

---

### Path 3: Skip Migration (Use 0.82 Immediately)

**⚠️ Only if:**
- New project OR
- Already on 0.76+ with New Architecture enabled

```bash
# Install latest
npm install react-native@0.82 react@19

# Update dependencies
npx expo install --fix

# New Architecture is mandatory (cannot disable)
# ✅ You're done!
```

---

## Compatibility Checker

Use this script to check your project:

```bash
# Run version checker
./scripts/check-rn-version.sh

# It will tell you:
# - Current React Native version
# - Whether New Architecture is enabled
# - Incompatible dependencies (Redux, i18n, etc.)
# - Recommended next steps
```

---

## Decision Helper: Questions to Ask

**Q1: Is this a new project?**
- YES → **Use 0.82+**
- NO → Continue to Q2

**Q2: What React Native version are you on now?**
- 0.75 or earlier → **Upgrade to 0.76-0.81 first**
- 0.76-0.81 → Continue to Q3
- 0.82+ → **You're already current**

**Q3: Do you have any of these dependencies?**
- `redux` + `redux-thunk` (not Redux Toolkit)
- `i18n-js`
- `react-native-code-push`
- Other libraries without New Architecture support

**If YES:**
- Can you replace them? → **Replace, then upgrade to 0.82+**
- Cannot replace? → **Stay on 0.76-0.81 temporarily, find alternatives**

**Q4: Are you using Expo?**
- YES, Expo Go → **Must use 0.76+ (New Architecture required)**
- YES, custom dev client → **Can use 0.76-0.81 or 0.82+**
- NO (bare React Native) → **Choose based on Q1-Q3**

---

## Summary Recommendations

| Scenario | Recommended Version | Notes |
|----------|-------------------|-------|
| **New project** | 0.82+ | Start with latest, no migration needed |
| **Migration from 0.75-** | 0.76-0.81 → 0.82+ | Two-step migration with interop layer |
| **Already on 0.76-0.81** | Upgrade to 0.82+ | Test thoroughly, fix dependencies |
| **Incompatible deps** | 0.76-0.81 (temporary) | Replace deps, plan migration |
| **Expo Go** | 0.76+ (SDK 52+) | New Architecture required |
| **Production app** | 0.82+ | Best performance, active support |
| **Legacy app (no migration)** | 0.72-0.75 | Only if absolutely necessary |

---

**Bottom Line:** Use React Native 0.82+ for new projects. For existing projects, upgrade to 0.76-0.81 first (get interop layer), then to 0.82+ after testing.
