# React 19 Migration Guide for React Native

**Last Updated:** 2025-11-22
**React Native Versions:** 0.78+ (React 19 support added)
**Source:** [React 19 Upgrade Guide](https://react.dev/blog/2024/04/25/react-19-upgrade-guide)

---

## Breaking Changes Summary

| Removed API | Replacement | Required Action |
|-------------|-------------|-----------------|
| `propTypes` | TypeScript | Remove all `propTypes` declarations |
| `forwardRef` | Regular `ref` prop | Remove wrapper, pass `ref` directly |
| `defaultProps` (function components) | Default parameters | Use JS default params |
| Legacy Context (`contextTypes`) | `useContext` hook | Migrate to modern Context API |
| String refs (`ref="input"`) | Callback/object refs | Use `useRef` hook |

---

## 1. propTypes Removed

### What Changed

React 19 **completely removed** runtime `propTypes` validation. No errors, no warnings - they're simply ignored.

### Before (Old Code)

```typescript
import PropTypes from 'prop-types';

function Button({ title, onPress, variant }) {
  return <Pressable onPress={onPress}><Text>{title}</Text></Pressable>;
}

Button.propTypes = {  // ❌ Silently ignored in React 19
  title: PropTypes.string.isRequired,
  onPress: PropTypes.func.isRequired,
  variant: PropTypes.oneOf(['primary', 'secondary'])
};

Button.defaultProps = {  // ❌ Also ignored for function components
  variant: 'primary'
};
```

### After (Use TypeScript)

```typescript
type ButtonProps = {
  title: string;
  onPress: () => void;
  variant?: 'primary' | 'secondary';
};

function Button({ title, onPress, variant = 'primary' }: ButtonProps) {
  return <Pressable onPress={onPress}><Text>{title}</Text></Pressable>;
}
```

### Migration Steps

1. **Install TypeScript (if not already)**:
   ```bash
   npm install --save-dev typescript @types/react @types/react-native
   ```

2. **Run React 19 codemod**:
   ```bash
   npx @codemod/react-19 upgrade
   ```

3. **Manually convert remaining propTypes**:
   - Find all `propTypes` declarations: `grep -r "\.propTypes" src/`
   - Convert to TypeScript types
   - Remove `prop-types` package: `npm uninstall prop-types`

---

## 2. forwardRef Deprecated

### What Changed

`ref` is now a regular prop - no need for `forwardRef` wrapper.

### Before (Old Code)

```typescript
import { forwardRef } from 'react';

const TextInput = forwardRef((props, ref) => {  // ❌ Deprecated
  return <NativeTextInput ref={ref} {...props} />;
});

// Usage
<TextInput ref={inputRef} />
```

### After (React 19)

```typescript
function TextInput({ ref, ...props }) {  // ✅ ref is a regular prop
  return <NativeTextInput ref={ref} {...props} />;
}

// Usage (same)
<TextInput ref={inputRef} />
```

### Migration Steps

1. **Find all forwardRef usages**:
   ```bash
   grep -r "forwardRef" src/
   ```

2. **Unwrap components**:
   - Remove `forwardRef()` wrapper
   - Add `ref` to function parameters
   - Keep everything else the same

3. **Codemod can help**:
   ```bash
   npx @codemod/react-19 upgrade
   ```

---

## 3. New Hooks

React 19 introduces several new hooks that replace common patterns.

### `useActionState` (replaces form state patterns)

**Replaces:** Manual form state management with `useState` + async functions

```typescript
import { useActionState } from 'react';

function LoginForm() {
  const [state, loginAction, isPending] = useActionState(
    async (prevState, formData) => {
      try {
        const user = await api.login(formData);
        return { success: true, user };
      } catch (error) {
        return { success: false, error: error.message };
      }
    },
    { success: false }  // Initial state
  );

  return (
    <View>
      <TextInput name="email" />
      <TextInput name="password" secureTextEntry />
      <Button onPress={() => loginAction(new FormData())}>
        {isPending ? 'Logging in...' : 'Login'}
      </Button>
      {!state.success && state.error && <Text>{state.error}</Text>}
    </View>
  );
}
```

### `useOptimistic` (optimistic UI updates)

**Use case:** Update UI immediately, then sync with server

```typescript
import { useOptimistic } from 'react';

function TodoItem({ todo, onToggle }) {
  const [optimisticTodo, addOptimisticToggle] = useOptimistic(
    todo,
    (current, toggle) => ({ ...current, completed: toggle })
  );

  async function handleToggle() {
    addOptimisticToggle(!optimisticTodo.completed);  // UI updates immediately
    await onToggle(todo.id);  // Server syncs in background
  }

  return (
    <Pressable onPress={handleToggle}>
      <Text style={{ textDecorationLine: optimisticTodo.completed ? 'line-through' : 'none' }}>
        {optimisticTodo.text}
      </Text>
    </Pressable>
  );
}
```

### `use` (read promises/contexts during render)

**Replaces:** `React.Suspense` + manual promise handling

```typescript
import { use, Suspense } from 'react';

function UserProfile({ userPromise }) {
  const user = use(userPromise);  // Suspends if promise is pending

  return <Text>{user.name}</Text>;
}

// Usage
<Suspense fallback={<Text>Loading...</Text>}>
  <UserProfile userPromise={fetchUser(id)} />
</Suspense>
```

---

## 4. Common Migration Errors

### Error: "propTypes is not a function"

**Cause:** Trying to use `propTypes` in React 19

**Fix:** Remove all `propTypes` declarations, use TypeScript

```bash
# Find all propTypes
grep -r "\.propTypes" src/

# Remove prop-types package
npm uninstall prop-types
```

### Error: "Warning: forwardRef is deprecated"

**Cause:** Using `forwardRef` wrapper

**Fix:** Remove wrapper, add `ref` as regular prop parameter

```typescript
// Before
const Component = forwardRef((props, ref) => <View ref={ref} />);

// After
function Component({ ref, ...props }) {
  return <View ref={ref} />;
}
```

### Error: "defaultProps is ignored for function components"

**Cause:** Using `defaultProps` on function components

**Fix:** Use JavaScript default parameters

```typescript
// Before
function Button({ title, variant }) { ... }
Button.defaultProps = { variant: 'primary' };

// After
function Button({ title, variant = 'primary' }) { ... }
```

---

## 5. Recommended Migration Order

1. **Upgrade React Native to 0.78+** (includes React 19)
   ```bash
   npm install react-native@0.78 react@19
   ```

2. **Run codemod**
   ```bash
   npx @codemod/react-19 upgrade
   ```

3. **Install TypeScript** (if not already)
   ```bash
   npm install --save-dev typescript @types/react @types/react-native
   ```

4. **Convert propTypes to TypeScript**
   - Use codemod or manually convert
   - Remove `prop-types` package when done

5. **Remove forwardRef wrappers**
   - Codemod handles most cases
   - Manually check complex components

6. **Update defaultProps**
   - Replace with default parameters
   - Class components can still use `defaultProps`

7. **Test thoroughly**
   - Run on both iOS and Android
   - Check all form submissions
   - Verify ref forwarding still works

---

## 6. Compatibility Notes

### What Still Works

✅ **Class components** - Fully supported, no changes needed
✅ **Hooks** - All existing hooks work the same
✅ **Context API** - No changes
✅ **Memo/useMemo** - No changes
✅ **TypeScript** - Better type inference

### What Doesn't Work

❌ **propTypes** - Removed, use TypeScript
❌ **forwardRef** - Deprecated, use regular ref prop
❌ **defaultProps** (function components) - Use default params
❌ **String refs** - Removed, use `useRef`
❌ **Legacy Context** - Removed, use modern Context API

---

## 7. Resources

- **Official Upgrade Guide**: https://react.dev/blog/2024/04/25/react-19-upgrade-guide
- **Codemod Tool**: https://github.com/codemod-com/react-19
- **React 19 Release**: https://react.dev/blog/2024/12/05/react-19
- **TypeScript Handbook**: https://www.typescriptlang.org/docs/handbook/react.html

---

## 8. Quick Reference: Before & After

```typescript
// ❌ OLD (React 18 / RN 0.72-0.77)
import PropTypes from 'prop-types';
import { forwardRef } from 'react';

const Button = forwardRef(({ title, onPress, variant }, ref) => {
  return <Pressable ref={ref} onPress={onPress}><Text>{title}</Text></Pressable>;
});

Button.propTypes = {
  title: PropTypes.string.isRequired,
  onPress: PropTypes.func.isRequired,
  variant: PropTypes.oneOf(['primary', 'secondary'])
};

Button.defaultProps = {
  variant: 'primary'
};

// ✅ NEW (React 19 / RN 0.78+)
type ButtonProps = {
  title: string;
  onPress: () => void;
  variant?: 'primary' | 'secondary';
  ref?: React.Ref<View>;
};

function Button({ title, onPress, variant = 'primary', ref }: ButtonProps) {
  return <Pressable ref={ref} onPress={onPress}><Text>{title}</Text></Pressable>;
}
```

---

**Bottom Line:** React 19 removes runtime validation (propTypes) in favor of compile-time validation (TypeScript). Use the codemod to automate most of the migration, then manually convert remaining propTypes to TypeScript types.
