# New CSS Features Cheatsheet (React Native 0.77+)

**Last Updated:** 2025-11-22
**React Native Version:** 0.77+ (New Architecture only)
**Source:** [React Native 0.77 Release Notes](https://reactnative.dev/blog/2025/01/14/release-0.77)

---

## Overview

React Native 0.77+ added several CSS properties previously only available on web. These bring React Native styling closer to web CSS parity.

**Requirements:**
- React Native 0.77+
- **New Architecture enabled** (these properties only work with Fabric)

---

## 1. `display: contents`

### What It Does
Makes an element "invisible" in the layout tree, but keeps its children.

### Use Case
Wrapper components that shouldn't affect layout hierarchy.

### Example

```typescript
function Card({ children }) {
  return (
    <View style={{ flexDirection: 'row', gap: 16 }}>
      <View style={{ display: 'contents' }}>
        {/* This View disappears, but children render as if they're direct children of Card */}
        <Image source={require('./icon.png')} />
        <Text>Title</Text>
      </View>
      <Button title="Action" />
    </View>
  );
}

// Rendered layout (View with display: contents is gone):
// <View flexDirection="row" gap={16}>
//   <Image />       ← Direct child (not wrapped)
//   <Text />        ← Direct child (not wrapped)
//   <Button />
// </View>
```

### Without `display: contents`

```typescript
// ❌ Without display: contents
<View style={{ flexDirection: 'row', gap: 16 }}>
  <View>
    <Image />   ← Inside wrapper View
    <Text />    ← Inside wrapper View
  </View>
  <Button />
</View>

// Layout is affected by wrapper View (extra nesting)
```

### Real-World Use

```typescript
// Conditional wrapper that shouldn't affect layout
function ConditionalWrapper({ condition, wrapper, children }) {
  if (condition) {
    return wrapper(children);
  }
  // Without display: contents, we'd have inconsistent layout
  return <View style={{ display: 'contents' }}>{children}</View>;
}

// Usage
<ConditionalWrapper
  condition={isLoggedIn}
  wrapper={(children) => <PremiumBadge>{children}</PremiumBadge>}
>
  <Avatar />
  <UserName />
</ConditionalWrapper>
```

---

## 2. `boxSizing`

### What It Does
Controls whether padding and border are included in width/height calculations.

### Values
- `border-box` (default) - Padding/border inside the box
- `content-box` - Padding/border outside the box

### Example

```typescript
// border-box (default)
<View style={{
  boxSizing: 'border-box',  // Default
  width: 100,
  padding: 10,
  borderWidth: 2
}}>
  {/* Total width: 100px (padding/border inside) */}
  {/* Content area: 100 - 20 (padding) - 4 (border) = 76px */}
</View>

// content-box
<View style={{
  boxSizing: 'content-box',
  width: 100,
  padding: 10,
  borderWidth: 2
}}>
  {/* Total width: 124px (100 + 20 padding + 4 border) */}
  {/* Content area: 100px */}
</View>
```

### Visual Comparison

```
border-box (default):
┌─────────────────────┐
│  border: 2          │  ← 100px total
│ ┌─────────────────┐ │
│ │ padding: 10     │ │
│ │ ┌─────────────┐ │ │
│ │ │   content   │ │ │  ← 76px (shrinks to fit)
│ │ │   76px      │ │ │
│ │ └─────────────┘ │ │
│ └─────────────────┘ │
└─────────────────────┘

content-box:
┌─────────────────────────────┐
│  border: 2                  │  ← 124px total (grows)
│ ┌───────────────────────────┐ │
│ │ padding: 10               │ │
│ │ ┌───────────────────────┐ │ │
│ │ │   content             │ │ │  ← 100px (fixed)
│ │ │   100px               │ │ │
│ │ └───────────────────────┘ │ │
│ └───────────────────────────┘ │
└─────────────────────────────┘
```

### When to Use

- **border-box** (default): Most cases - easier to reason about total size
- **content-box**: When you need exact content dimensions (rare in React Native)

---

## 3. `mixBlendMode` + `isolation`

### What It Does
Blends colors like Photoshop layer blend modes.

### Available Modes (16 total)

| Mode | Effect |
|------|--------|
| `multiply` | Darkens (like overlaying inks) |
| `screen` | Lightens (like projecting light) |
| `overlay` | Contrast boost |
| `darken` | Keeps darkest colors |
| `lighten` | Keeps lightest colors |
| `color-dodge` | Brightens by reducing contrast |
| `color-burn` | Darkens by increasing contrast |
| `hard-light` | Strong overlay |
| `soft-light` | Gentle overlay |
| `difference` | Inverts based on difference |
| `exclusion` | Like difference but lower contrast |
| `hue` | Uses hue of top layer |
| `saturation` | Uses saturation of top layer |
| `color` | Uses hue+saturation of top layer |
| `luminosity` | Uses luminosity of top layer |

### Examples

```typescript
// Multiply (darkening effect)
<View style={{ backgroundColor: '#ff0000' }}>  {/* Red */}
  <View style={{
    mixBlendMode: 'multiply',
    backgroundColor: '#0000ff'  {/* Blue */}
  }}>
    {/* Result: #000000 (black) - red × blue */}
  </View>
</View>

// Screen (lightening effect)
<View style={{ backgroundColor: '#ff0000' }}>  {/* Red */}
  <View style={{
    mixBlendMode: 'screen',
    backgroundColor: '#00ff00'  {/* Green */}
  }}>
    {/* Result: #ffff00 (yellow) - red + green */}
  </View>
</View>

// Overlay (text on gradient)
<View style={{
  background: 'linear-gradient(to right, #ff0000, #0000ff)'
}}>
  <Text style={{
    mixBlendMode: 'overlay',
    color: 'white'
  }}>
    Blended Text
  </Text>
</View>
```

### `isolation` Property

Prevents blend modes from affecting parent layers:

```typescript
<View style={{ backgroundColor: 'red' }}>
  <View style={{ isolation: 'isolate' }}>
    {/* Blend modes inside here won't affect red background */}
    <View style={{ mixBlendMode: 'multiply', backgroundColor: 'blue' }}>
      {/* Only blends with siblings, not with red parent */}
    </View>
  </View>
</View>
```

### Real-World Use: Glowing Button

```typescript
function GlowButton({ title, onPress }) {
  return (
    <Pressable
      onPress={onPress}
      style={{
        backgroundColor: '#3b82f6',
        padding: 16,
        borderRadius: 8,
        // Prevent blending with background
        isolation: 'isolate'
      }}
    >
      {/* Glow effect */}
      <View style={{
        position: 'absolute',
        inset: 0,
        backgroundColor: 'white',
        opacity: 0.2,
        mixBlendMode: 'screen',  // Lightens button
        borderRadius: 8
      }} />

      <Text style={{ color: 'white', fontWeight: 'bold' }}>
        {title}
      </Text>
    </Pressable>
  );
}
```

---

## 4. `outline` Properties

### What It Does
Draws an outline that doesn't affect layout (unlike `border`).

### Properties

```typescript
outlineWidth: number      // Thickness (default: 0)
outlineStyle: string      // 'solid' | 'dashed' | 'dotted'
outlineColor: string      // Color (default: black)
outlineOffset: number     // Space between element and outline (default: 0)
outlineSpread: number     // Expand outline beyond offset (default: 0)
```

### Example

```typescript
<View style={{
  width: 100,
  height: 100,
  backgroundColor: 'blue',

  // Outline doesn't change size (still 100x100)
  outlineWidth: 2,
  outlineStyle: 'solid',
  outlineColor: 'red',
  outlineOffset: 4,       // 4px gap between element and outline
  outlineSpread: 2        // Outline extends 2px beyond offset
}} />
```

### Visual Explanation

```
Without outline:
┌────────────┐
│   Element  │  100x100
│   (blue)   │
└────────────┘

With outline (doesn't affect layout):
              ┌──────────────────┐  ← outlineSpread: 2
              │   red outline    │
              │ ┌──────────────┐ │  ← outlineOffset: 4 (gap)
              │ │              │ │
              │ │   Element    │ │  ← Still 100x100 (no layout change)
              │ │   (blue)     │ │
              │ │              │ │
              │ └──────────────┘ │
              └──────────────────┘
```

### `outline` vs `border`

| Property | Affects Layout? | Use Case |
|----------|----------------|----------|
| `border` | ✅ Yes (increases size) | Actual borders, dividers |
| `outline` | ❌ No (drawn on top) | Focus indicators, highlights |

### Real-World Use: Focus Indicator

```typescript
function FocusableButton({ focused, title, onPress }) {
  return (
    <Pressable
      onPress={onPress}
      style={{
        padding: 16,
        backgroundColor: focused ? '#3b82f6' : '#gray',

        // Outline appears on focus without changing layout
        outlineWidth: focused ? 2 : 0,
        outlineStyle: 'solid',
        outlineColor: '#60a5fa',
        outlineOffset: 2  // Small gap from button
      }}
    >
      <Text>{title}</Text>
    </Pressable>
  );
}
```

### Dashed/Dotted Outlines

```typescript
// Dashed outline
<View style={{
  outlineWidth: 2,
  outlineStyle: 'dashed',  // ← Dashed
  outlineColor: 'gray'
}} />

// Dotted outline
<View style={{
  outlineWidth: 2,
  outlineStyle: 'dotted',  // ← Dotted
  outlineColor: 'gray'
}} />
```

---

## Complete Example: Product Card

Combining all new CSS features:

```typescript
function ProductCard({ product, onPress, focused }) {
  return (
    <Pressable
      onPress={onPress}
      style={{
        // Outline for focus (no layout impact)
        outlineWidth: focused ? 2 : 0,
        outlineStyle: 'solid',
        outlineColor: '#3b82f6',
        outlineOffset: 4,

        // Isolation prevents blend from affecting siblings
        isolation: 'isolate',

        padding: 16,
        backgroundColor: 'white',
        borderRadius: 8,

        // Using border-box (default) for predictable sizing
        boxSizing: 'border-box',
        width: 200
      }}
    >
      {/* Product image with overlay */}
      <View style={{ position: 'relative' }}>
        <Image source={product.image} style={{ width: '100%', height: 120 }} />

        {/* Sale badge with blend mode */}
        {product.onSale && (
          <View style={{
            position: 'absolute',
            top: 8,
            right: 8,
            backgroundColor: '#ff0000',
            padding: 4,
            borderRadius: 4,
            mixBlendMode: 'multiply'  // Darkens over image
          }}>
            <Text style={{ color: 'white', fontSize: 12 }}>SALE</Text>
          </View>
        )}
      </View>

      {/* Product details - display: contents for layout flexibility */}
      <View style={{ display: 'contents' }}>
        <Text style={{ fontWeight: 'bold', marginTop: 8 }}>
          {product.name}
        </Text>
        <Text style={{ color: 'gray' }}>
          ${product.price}
        </Text>
      </View>

      {/* Glow effect on hover/focus */}
      {focused && (
        <View style={{
          position: 'absolute',
          inset: 0,
          backgroundColor: 'white',
          opacity: 0.1,
          mixBlendMode: 'screen',
          borderRadius: 8,
          pointerEvents: 'none'
        }} />
      )}
    </Pressable>
  );
}
```

---

## Browser Compatibility Note

These CSS properties work **identically** to their web counterparts, making React Native styling more consistent with React (web) development.

| Property | Web Support | React Native Support |
|----------|-------------|---------------------|
| `display: contents` | ✅ Modern browsers | ✅ RN 0.77+ (New Arch) |
| `boxSizing` | ✅ All browsers | ✅ RN 0.77+ (New Arch) |
| `mixBlendMode` | ✅ Modern browsers | ✅ RN 0.77+ (New Arch) |
| `outline` | ✅ All browsers | ✅ RN 0.77+ (New Arch) |

---

## Performance Notes

- **All properties are performant** - rendered natively by Fabric
- **No JavaScript overhead** - handled entirely by native rendering
- **No layout recalculations** for `outline` (unlike `border`)
- **`isolation`** creates a new stacking context (minor cost)

---

## Quick Reference Table

| Property | Effect | Layout Impact | Use Case |
|----------|--------|---------------|----------|
| `display: contents` | Makes wrapper invisible | ✅ Affects | Remove wrapper from layout tree |
| `boxSizing` | Changes size calculation | ✅ Affects | Control padding/border sizing |
| `mixBlendMode` | Blends colors | ❌ None | Visual effects, overlays |
| `isolation` | Contains blend effects | ❌ Minor | Prevent blend propagation |
| `outline` | Draws outline | ❌ None | Focus indicators, highlights |

---

**Bottom Line:** These CSS properties bring React Native 50% closer to web CSS parity. Use them to create sophisticated visual effects without JavaScript or extra layout complexity.

**Requirement:** React Native 0.77+ with New Architecture enabled.
