# Embeddings & Vector Search

Vector search is a core Convex feature. How you store embeddings critically impacts performance and cost.

## Default Pattern: Co-location (Same Table)

For most applications, store embedding vector **in the same table** as source data. Simplest and most effective.

✅ **Best practice: Co-located embeddings**
```typescript
// convex/schema.ts
defineTable({
  text: v.string(),
  embedding: v.array(v.float64()), // OpenAI embeddings are 1536 dims
})
.vectorIndex("by_embedding", {
  vectorField: "embedding",
  dimensions: 1536,
  filterFields: [], // Add fields here to filter during vector search
});
```

## Advanced Pattern: Separate Embeddings Table

**Only use separate table when:**
- Multiple embeddings per document (e.g., different models or chunked text)
- Source document is extremely large (keep main table lean)
- Embeddings generated by decoupled process

This pattern requires "joins" (two `ctx.db.get()` calls) or denormalization, adding complexity.

❌ **Avoid unless necessary: Separate tables**
```typescript
// Main documents table
defineTable({
  title: v.string(),
  content: v.string(),
});

// Separate embeddings table
defineTable({
  documentId: v.id("documents"),
  embedding: v.array(v.float64()),
})
.index("by_document", ["documentId"])
.vectorIndex("by_embedding", {
  vectorField: "embedding",
  dimensions: 1536,
});
```

## Vector Search Must Use Actions

Vector searches **must** be performed inside `actions` using `ctx.vectorSearch`.

✅ **Best practice: Vector search in action**
```typescript
// convex/similarItems.ts
import { action } from "./_generated/server";
import { api, internal } from "./_generated/api";

export const findSimilar = action({
  args: { id: v.id("documents") },
  handler: async (ctx, args) => {
    const doc = await ctx.runQuery(api.documents.get, { id: args.id });
    if (!doc || !doc.embedding) {
      throw new Error("Document or embedding not found");
    }

    const results = await ctx.vectorSearch("documents", "by_embedding", {
      vector: doc.embedding,
      limit: 10,
    });
    return results;
  },
});
```

## Batch Embeddings API Calls

Fetch multiple embeddings at once to reduce overhead.

❌ **Anti-pattern: Sequential API calls**
```typescript
for (const doc of docs) {
  const embedding = await openai.embeddings.create({
    input: doc.text,
    model: "text-embedding-ada-002",
  });
  await ctx.db.patch(doc._id, { embedding: embedding.data[0].embedding });
}
```

✅ **Best practice: Batch embeddings**
```typescript
// Batch all texts together
const embeddings = await openai.embeddings.create({
  input: docs.map(d => d.text),
  model: "text-embedding-ada-002",
});

// Update all documents
for (let i = 0; i < docs.length; i++) {
  await ctx.db.patch(docs[i]._id, {
    embedding: embeddings.data[i].embedding
  });
}
```

## Model Consistency

Different embedding models produce incompatible arrays. **Commit to a single model** (e.g., `text-embedding-ada-002`) rather than experimenting across providers.

## When to Separate Embeddings (Critical Decision)

**Keep embeddings WITH source data when:**
- ✅ One embedding per document
- ✅ Source document is small-to-medium size (<10KB)
- ✅ Embeddings generated inline with document creation
- ✅ Embeddings are always needed with the document

**Separate embeddings ONLY when:**
- ⚠️ Multiple embeddings per document (chunks, different models)
- ⚠️ Source document is very large (>100KB)
- ⚠️ Embeddings generated asynchronously/decoupled
- ⚠️ **Bandwidth optimization**: Embeddings 1536 floats (~6KB each) — if frequently fetching documents WITHOUT needing embeddings, separation reduces bandwidth

## Embeddings Decision Tree

```
Need vector search?
  → One embedding per doc + small source? → Co-locate (same table)
  → Multiple embeddings per doc? → Separate table
  → Very large source docs (>100KB)? → Separate table
  → Frequently fetch docs WITHOUT embeddings? → Separate table
  → Otherwise → Co-locate (default)
```
