#!/bin/bash
# Detect if Sentry is configured in current project
# Exit codes: 0=fully configured, 1=not configured, 2=partially configured

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
source "$SCRIPT_DIR/_common.sh"

# Default options
OUTPUT_JSON=false

HELP_TEXT="Usage: $(basename "$0") [OPTIONS]

Detect if Sentry is configured in the current project.

Options:
  --json        Output results as JSON
  --help        Show this help message

Exit codes:
  0  Fully configured (all checks pass)
  1  Not configured (no Sentry detected)
  2  Partially configured (some checks pass)

Example:
  $(basename "$0")
  $(basename "$0") --json"

# Parse arguments
while [[ $# -gt 0 ]]; do
  case $1 in
    --json)
      OUTPUT_JSON=true
      shift
      ;;
    --help|-h)
      show_help "$HELP_TEXT"
      exit 0
      ;;
    *)
      die "Unknown option: $1. Use --help for usage."
      ;;
  esac
done

# Initialize check results
declare -A checks
checks[packages]=false
checks[client_config]=false
checks[server_config]=false
checks[dsn_env]=false
checks[vercel_integration]=false

# Check 1: @sentry/* packages in package.json
check_packages() {
  if [ -f "package.json" ]; then
    if grep -q '"@sentry/' package.json 2>/dev/null; then
      checks[packages]=true
      return 0
    fi
  fi
  return 1
}

# Check 2: sentry.client.config.ts exists
check_client_config() {
  local files=("sentry.client.config.ts" "sentry.client.config.js" "src/sentry.client.config.ts")
  for f in "${files[@]}"; do
    if [ -f "$f" ]; then
      checks[client_config]=true
      return 0
    fi
  done
  return 1
}

# Check 3: sentry.server.config.ts exists
check_server_config() {
  local files=("sentry.server.config.ts" "sentry.server.config.js" "src/sentry.server.config.ts" "instrumentation.ts")
  for f in "${files[@]}"; do
    if [ -f "$f" ]; then
      checks[server_config]=true
      return 0
    fi
  done
  return 1
}

# Check 4: SENTRY_DSN in .env.local or .env
check_dsn_env() {
  local env_files=(".env.local" ".env" ".env.development")
  for f in "${env_files[@]}"; do
    if [ -f "$f" ]; then
      if grep -q "SENTRY_DSN\|NEXT_PUBLIC_SENTRY_DSN" "$f" 2>/dev/null; then
        checks[dsn_env]=true
        return 0
      fi
    fi
  done
  return 1
}

# Check 5: Vercel Integration connected
check_vercel_integration() {
  # Check for sentry.properties (generated by wizard)
  if [ -f "sentry.properties" ]; then
    checks[vercel_integration]=true
    return 0
  fi
  # Check for .sentryclirc
  if [ -f ".sentryclirc" ]; then
    checks[vercel_integration]=true
    return 0
  fi
  return 1
}

# Run all checks
check_packages
check_client_config
check_server_config
check_dsn_env
check_vercel_integration

# Count passed checks
passed=0
total=5
for check in "${!checks[@]}"; do
  if [ "${checks[$check]}" = true ]; then
    ((passed++))
  fi
done

# Determine overall status
if [ $passed -eq 0 ]; then
  status="not_configured"
  exit_code=1
elif [ $passed -eq $total ]; then
  status="configured"
  exit_code=0
else
  status="partial"
  exit_code=2
fi

# Output results
if [ "$OUTPUT_JSON" = true ]; then
  cat <<EOF
{
  "status": "$status",
  "passed": $passed,
  "total": $total,
  "checks": {
    "packages": ${checks[packages]},
    "client_config": ${checks[client_config]},
    "server_config": ${checks[server_config]},
    "dsn_env": ${checks[dsn_env]},
    "vercel_integration": ${checks[vercel_integration]}
  }
}
EOF
else
  echo ""
  echo "Sentry Configuration Status: $(echo $status | tr '_' ' ' | sed 's/\b\(.\)/\u\1/g')"
  echo "================================"
  echo ""

  # Show check results
  for check in packages client_config server_config dsn_env vercel_integration; do
    if [ "${checks[$check]}" = true ]; then
      echo -e "${GREEN}✓${NC} $check"
    else
      echo -e "${RED}✗${NC} $check"
    fi
  done

  echo ""
  echo "Result: $passed/$total checks passed"

  if [ "$status" = "not_configured" ]; then
    echo ""
    echo "To set up Sentry, run:"
    echo "  ~/.claude/skills/sentry-observability/scripts/init_sentry.sh"
  elif [ "$status" = "partial" ]; then
    echo ""
    echo "Some configuration is missing. Run verify_setup.sh for details:"
    echo "  ~/.claude/skills/sentry-observability/scripts/verify_setup.sh"
  fi
fi

exit $exit_code
