---
name: "Pandas & NumPy"
description: "Master Pandas dataframes and NumPy arrays for efficient data manipulation, analysis, and transformation"
category: "Programming"
level: "Intermediate"
duration: "5-6 weeks"
---

# Pandas & NumPy

## Quick Start

Pandas and NumPy are essential tools for data manipulation. Within your first week, you'll load data into dataframes and perform basic operations. By week four, you'll merge datasets, create pivot tables, and perform complex aggregations programmatically.

**First Task (30 minutes):**
1. Load a CSV file into a Pandas dataframe
2. Inspect data (shape, info, describe)
3. Filter rows and select columns
4. Calculate simple statistics
5. Save cleaned data to new CSV

## Key Concepts

### 1. NumPy Arrays Fundamentals
**What it is:** Efficient numerical array operations for mathematical computing.

**Array Creation & Operations:**
```python
import numpy as np

# Creating arrays
array1d = np.array([1, 2, 3, 4, 5])
array2d = np.array([[1, 2, 3], [4, 5, 6]])

# Creating special arrays
zeros = np.zeros(5)              # [0., 0., 0., 0., 0.]
ones = np.ones((2, 3))           # 2x3 matrix of ones
range_arr = np.arange(0, 10, 2)  # [0, 2, 4, 6, 8]
linspace = np.linspace(0, 1, 5)  # 5 evenly spaced values

# Array properties
print(array2d.shape)    # (2, 3) - dimensions
print(array2d.dtype)    # int64 - data type
print(array2d.size)     # 6 - total elements

# Element-wise operations (very fast)
a = np.array([1, 2, 3])
b = np.array([4, 5, 6])
print(a + b)    # [5, 7, 9]
print(a * 2)    # [2, 4, 6]
print(a ** 2)   # [1, 4, 9]
```

**Array Indexing & Slicing:**
```python
arr = np.array([10, 20, 30, 40, 50])

# Indexing
print(arr[0])       # 10
print(arr[-1])      # 50 (last element)

# Slicing
print(arr[1:4])     # [20, 30, 40]
print(arr[::2])     # [10, 30, 50] (every other)

# 2D array indexing
array2d = np.array([[1, 2, 3], [4, 5, 6]])
print(array2d[0, 1])    # 2 (row 0, column 1)
print(array2d[:, 1])    # [2, 5] (all rows, column 1)

# Boolean indexing
mask = arr > 25
print(arr[mask])    # [30, 40, 50]
```

### 2. Pandas DataFrames
**What it is:** Tabular data structure (like Excel sheet) with powerful operations.

**DataFrame Creation & Loading:**
```python
import pandas as pd

# Create from dictionary
data = {
    'customer_id': [1, 2, 3, 4],
    'name': ['Alice', 'Bob', 'Charlie', 'Diana'],
    'sales': [1000, 1500, 800, 2000]
}
df = pd.DataFrame(data)

# Load from CSV
df = pd.read_csv('data.csv')

# Load from Excel
df = pd.read_excel('data.xlsx', sheet_name='Sales')

# Load from SQL database
from sqlalchemy import create_engine
engine = create_engine('postgresql://user:password@localhost/db')
df = pd.read_sql('SELECT * FROM customers', engine)
```

**Exploring DataFrames:**
```python
# Basic info
print(df.shape)          # (1000, 5) - rows, columns
print(df.info())         # Data types, memory usage
print(df.head())         # First 5 rows
print(df.tail(10))       # Last 10 rows
print(df.describe())     # Summary statistics

# Column info
print(df.columns)        # Column names
print(df.dtypes)         # Data types per column
print(df['sales'].dtype) # Single column type
```

### 3. Data Selection & Filtering
**What it is:** Selecting specific data based on conditions.

**Column Selection:**
```python
# Single column (returns Series)
series = df['name']
series = df.name  # Alternative syntax

# Multiple columns
subset = df[['name', 'sales']]

# Add new column
df['region'] = 'North'
df['sales_x100'] = df['sales'] * 100
```

**Row Filtering:**
```python
# Single condition
high_sales = df[df['sales'] > 1000]

# Multiple conditions
filtered = df[(df['sales'] > 1000) & (df['region'] == 'North')]

# Using isin() for multiple values
subset = df[df['region'].isin(['North', 'South'])]

# Using query() method (more readable)
result = df.query('sales > 1000 and region == "North"')

# Filtering out nulls
clean_df = df[df['column'].notna()]
```

### 4. Data Cleaning & Transformation
**What it is:** Preparing messy data for analysis.

**Handling Missing Values:**
```python
# Identify missing values
print(df.isnull().sum())  # Count nulls per column

# Remove rows with any null
df_clean = df.dropna()

# Remove rows with null in specific column
df_clean = df.dropna(subset=['customer_id'])

# Fill missing values
df['region'].fillna('Unknown', inplace=True)
df['age'].fillna(df['age'].mean(), inplace=True)  # Fill with mean
```

**Removing Duplicates:**
```python
# Find duplicates
print(df.duplicated().sum())

# Remove duplicate rows
df_unique = df.drop_duplicates()

# Remove duplicates on specific column
df_unique = df.drop_duplicates(subset=['customer_id'])
```

**Data Type Conversion:**
```python
# Convert data types
df['age'] = df['age'].astype(int)
df['purchase_date'] = pd.to_datetime(df['purchase_date'])

# Example of conversion
df['sales'] = pd.to_numeric(df['sales'], errors='coerce')  # Convert, invalid → NaN
```

**String Operations:**
```python
# Clean text data
df['name'] = df['name'].str.lower()     # Lowercase
df['name'] = df['name'].str.strip()     # Remove whitespace
df['name'] = df['name'].str.replace(' ', '_')

# Extract from string
df['domain'] = df['email'].str.split('@').str[1]

# Check if contains
has_gmail = df['email'].str.contains('gmail')
```

### 5. Aggregation & Grouping
**What it is:** Summarizing data by categories and calculating metrics.

**GroupBy Operations:**
```python
# Simple grouping
by_region = df.groupby('region')['sales'].sum()

# Multiple aggregations
summary = df.groupby('region').agg({
    'sales': 'sum',
    'customer_id': 'count',
    'age': 'mean'
}).rename(columns={'customer_id': 'num_customers'})

# Multiple group columns
by_region_category = df.groupby(['region', 'product_category'])['sales'].sum()

# Custom aggregation function
def custom_agg(group):
    return {
        'total': group['sales'].sum(),
        'average': group['sales'].mean(),
        'count': len(group)
    }

result = df.groupby('region').apply(custom_agg)
```

**Pivot Tables:**
```python
# Create pivot table
pivot = df.pivot_table(
    values='sales',
    index='region',
    columns='product_category',
    aggfunc='sum',
    fill_value=0
)

# Multiple aggregations
pivot = df.pivot_table(
    values=['sales', 'quantity'],
    index='region',
    columns='month',
    aggfunc={'sales': 'sum', 'quantity': 'count'}
)
```

**Merging & Joining Data:**
```python
# Merge (like SQL JOIN)
result = pd.merge(customers, orders,
                 left_on='customer_id',
                 right_on='cust_id',
                 how='inner')

# Concatenate (stack dataframes)
combined = pd.concat([df1, df2], ignore_index=True)

# Append rows
df = pd.concat([df, new_row_df], ignore_index=True)
```

## Tools and Resources

**Installation:**
```bash
pip install pandas numpy matplotlib jupyter
# or use Anaconda which includes both
```

**Documentation:**
- Pandas Official: https://pandas.pydata.org/docs/
- NumPy Official: https://numpy.org/doc/
- Real Python: Pandas and NumPy tutorials

**Learning Resources:**
- DataCamp: Pandas courses
- Coursera: Data Science with Python
- "Python for Data Analysis" book by Wes McKinney

## Best Practices

1. **Check Data First:** Always run describe(), info(), head()
2. **Preserve Original:** Work on copies for exploratory analysis
3. **Handle Missing Appropriately:** Decide between dropping or filling
4. **Use Vectorized Operations:** Avoid for loops when possible (much faster)
5. **Memory Efficient:** Use appropriate data types (int32 vs int64)
6. **Chain Operations:** Use method chaining for readability
7. **Document Data Quality:** Track issues found and decisions made
8. **Validate After Transform:** Spot-check results after operations
9. **Use Meaningful Names:** df_cleaned not df2
10. **Avoid Modifying In Place:** Unless explicitly necessary

## Next Steps

1. **Week 1-2:** Master dataframe basics and data loading
2. **Week 2-3:** Learn filtering, selection, and data cleaning
3. **Week 3-4:** Master aggregation and grouping
4. **Week 4-5:** Practice merging and complex transformations
5. **Week 5-6:** Build complete data analysis projects
6. **After:** Move to R programming or statistical modeling
7. **Progression:** Pandas/NumPy → Advanced Analytics → Machine Learning
