---
name: "R Programming"
description: "Learn R basics, dplyr data manipulation, ggplot2 visualization, tidyverse ecosystem, and statistical functions"
category: "Programming"
level: "Intermediate"
duration: "5-6 weeks"
---

# R Programming

## Quick Start

R is designed for statistical analysis and visualization. Within your first week, you'll write scripts and manipulate data with dplyr. By week four, you'll create publication-quality graphics with ggplot2 and perform statistical analyses efficiently.

**First Task (30 minutes):**
1. Install R and RStudio
2. Load a CSV dataset
3. Use dplyr to filter, select, and summarize data
4. Create a basic ggplot2 visualization
5. Save results and plot to files

## Key Concepts

### 1. R Fundamentals
**What it is:** Core R syntax for data analysis.

**Data Types & Structures:**
```r
# Vectors (fundamental data structure)
numbers <- c(1, 2, 3, 4, 5)
names <- c("Alice", "Bob", "Charlie")
mixed <- c(1, "text", TRUE)  # Converts to common type (character)

# Named vectors
ages <- c(Alice = 25, Bob = 30, Charlie = 28)
ages["Alice"]  # 25

# Sequences
seq1 <- 1:10
seq2 <- seq(0, 1, by=0.1)

# Lists (heterogeneous collections)
person <- list(name="John", age=30, city="NYC")
person$name  # "John"
person[["age"]]  # 30

# Matrices
matrix1 <- matrix(1:6, nrow=2, ncol=3)

# Data frames (tabular data)
df <- data.frame(
  name = c("Alice", "Bob", "Charlie"),
  age = c(25, 30, 28),
  salary = c(50000, 60000, 55000)
)
```

**Control Flow:**
```r
# If/else
if (age >= 18) {
  print("Adult")
} else if (age >= 13) {
  print("Teen")
} else {
  print("Child")
}

# Loops
for (i in 1:5) {
  print(i)
}

for (name in c("Alice", "Bob", "Charlie")) {
  print(paste("Hello,", name))
}

# While loop
i <- 1
while (i <= 5) {
  print(i)
  i <- i + 1
}

# apply family (vectorized operations)
sapply(1:5, function(x) x^2)  # [1, 4, 9, 16, 25]
lapply(list(1, 2, 3), function(x) x * 2)
```

### 2. Dplyr Data Manipulation
**What it is:** Intuitive grammar for data transformation.

**Core dplyr Functions:**
```r
library(dplyr)

# Filter rows
high_earners <- df %>%
  filter(salary > 55000)

# Select columns
subset <- df %>%
  select(name, salary)

# Create new columns
df_with_bonus <- df %>%
  mutate(bonus = salary * 0.1,
         total_comp = salary + bonus)

# Rename columns
df_renamed <- df %>%
  rename(employee_name = name,
         annual_salary = salary)

# Arrange (sort)
df_sorted <- df %>%
  arrange(salary)  # Ascending
  arrange(desc(salary))  # Descending

# Summarize (aggregate)
summary_stats <- df %>%
  summarise(
    avg_salary = mean(salary),
    max_salary = max(salary),
    count = n()
  )

# Group by
by_department <- df %>%
  group_by(department) %>%
  summarise(
    total_salary = sum(salary),
    avg_salary = mean(salary),
    num_employees = n()
  )
```

**Pipe Operator (%>%):**
```r
# Pipe chains operations left to right (more readable)
result <- df %>%
  filter(salary > 50000) %>%
  select(name, salary) %>%
  arrange(desc(salary)) %>%
  mutate(salary_thousands = salary / 1000)

# Same as:
result <- mutate(
  arrange(
    select(
      filter(df, salary > 50000),
      name, salary
    ),
    desc(salary)
  ),
  salary_thousands = salary / 1000
)
```

**Joining Data:**
```r
library(dplyr)

# Create sample tables
customers <- tibble(
  customer_id = 1:3,
  name = c("Alice", "Bob", "Charlie")
)

orders <- tibble(
  order_id = 1:5,
  customer_id = c(1, 1, 2, 3, 3),
  amount = c(100, 200, 150, 300, 50)
)

# Inner join (only matching rows)
inner_join(customers, orders, by="customer_id")

# Left join (all from left, matching from right)
left_join(customers, orders, by="customer_id")

# Bind rows
combined <- bind_rows(df1, df2)

# Bind columns
combined <- bind_cols(col1, col2)
```

### 3. ggplot2 Visualization
**What it is:** Grammar of graphics for creating publication-quality plots.

**ggplot2 Structure:**
```r
library(ggplot2)

# Basic ggplot template
ggplot(data, aes(x=variable1, y=variable2)) +
  geom_point() +
  labs(title="Chart Title", x="X Label", y="Y Label") +
  theme_minimal()

# Bar chart
ggplot(df, aes(x=category, y=value)) +
  geom_col(fill="steelblue") +
  theme(axis.text.x = element_text(angle=45, hjust=1))

# Line plot with multiple groups
ggplot(sales, aes(x=month, y=revenue, color=region)) +
  geom_line() +
  geom_point() +
  facet_wrap(~year) +
  scale_color_brewer(palette="Set1")

# Histogram
ggplot(df, aes(x=age)) +
  geom_histogram(binwidth=5, fill="lightblue", color="black") +
  labs(title="Age Distribution")

# Box plot
ggplot(df, aes(x=region, y=salary)) +
  geom_boxplot(fill="lightgreen") +
  geom_jitter(alpha=0.5, width=0.2)

# Scatter plot with regression line
ggplot(df, aes(x=years_experience, y=salary)) +
  geom_point() +
  geom_smooth(method="lm", se=TRUE) +
  theme_classic()
```

**ggplot2 Customization:**
```r
# Color scales
scale_color_manual(values=c("red", "blue", "green"))
scale_fill_brewer(palette="Spectral")

# Themes
theme_minimal()
theme_classic()
theme_dark()
theme_void()

# Faceting (small multiples)
facet_wrap(~region)        # Wrap by region
facet_grid(year~region)    # Grid by year and region

# Axis customization
scale_x_continuous(limits=c(0, 100), breaks=seq(0,100,25))
scale_y_log10()  # Log scale

# Labels and legend
labs(title="Title", subtitle="Subtitle", caption="Source: Data")
theme(legend.position="bottom")
```

### 4. Tidyverse Ecosystem
**What it is:** Suite of packages designed for data science workflows.

**Key Tidyverse Packages:**
```r
# Load all main packages
library(tidyverse)

# Data import
library(readr)   # CSV, TSV, fixed-width files
df <- read_csv("data.csv")

# Data tidying
library(tidyr)   # Reshape, pivot
df_long <- df %>%
  pivot_longer(cols=-id, names_to="variable", values_to="value")
df_wide <- df %>%
  pivot_wider(names_from=variable, values_from=value)

# String manipulation
library(stringr)
df %>%
  mutate(name = str_to_lower(name),
         initials = str_sub(name, 1, 1))

# Factors
library(forcats)
df %>%
  mutate(category = fct_reorder(category, value))

# Date/time
library(lubridate)
df %>%
  mutate(year = year(date),
         month = month(date),
         day_of_week = wday(date))
```

### 5. Statistical Analysis in R
**What it is:** R's primary strength - statistical functions built-in.

**Common Statistical Tests:**
```r
# T-test
t.test(group1, group2, paired=FALSE)

# ANOVA
aov_model <- aov(salary ~ department, data=df)
summary(aov_model)

# Correlation
cor(df$age, df$salary)
cor.test(df$age, df$salary)

# Linear regression
model <- lm(salary ~ years_experience, data=df)
summary(model)
predictions <- predict(model, new_data)

# Logistic regression
logit_model <- glm(purchased ~ age + income,
                   data=df, family=binomial)
```

## Tools and Resources

**Installation:**
```r
# Install packages
install.packages("dplyr")
install.packages("ggplot2")
install.packages("tidyverse")

# Load packages
library(dplyr)
library(ggplot2)
```

**Development Environment:**
- RStudio: Free IDE for R (rstudio.com)
- R Console: Built-in editor
- Jupyter: R kernel available

**Learning Resources:**
- "R for Data Science" by Hadley Wickham (free online)
- RStudio Cheat Sheets (official, free PDF)
- DataCamp: R for Data Science courses
- Stack Overflow: R tag for troubleshooting

**Data Sources:**
- Kaggle: Public datasets
- R built-in: mtcars, iris, diamonds
- Government sources: Federal, state, local data

## Best Practices

1. **Use tidyverse:** Modern, consistent approach to data science
2. **Pipe Operations:** Use %>% for readable code chains
3. **Name Objects Clearly:** df_cleaned not x
4. **Document Code:** Comments and explanations
5. **Check Data Structure:** Always str() and head() first
6. **Vectorize Operations:** Avoid loops when possible
7. **Use RMarkdown:** Combine code, output, and text
8. **Version Control:** Use Git for scripts
9. **Create Reproducible Reports:** RMarkdown for sharing
10. **Test Assumptions:** Check normality, homogeneity for tests

## Next Steps

1. **Week 1-2:** Master R basics and data frame operations
2. **Week 2-3:** Learn dplyr for data transformation
3. **Week 3-4:** Create visualizations with ggplot2
4. **Week 4-5:** Combine tidyverse tools for complete workflows
5. **Week 5-6:** Perform statistical analyses and create reports
6. **After:** Advanced modeling, machine learning in R
7. **Progression:** R Basics → Tidyverse → Statistical Modeling → ML
