# DevOps Scripting Guide
## Bash Essentials
```bash
#!/bin/bash
set -euo pipefail  # Strict mode

# Variables
readonly APP_NAME="myapp"
LOG_FILE="/var/log/${APP_NAME}.log"

# Functions
log() { echo "[$(date '+%Y-%m-%d %H:%M:%S')] $*" | tee -a "$LOG_FILE"; }
die() { log "ERROR: $*"; exit 1; }

# Argument parsing
while getopts "v:h" opt; do
  case $opt in
    v) VERSION="$OPTARG" ;;
    h) usage; exit 0 ;;
    *) usage; exit 1 ;;
  esac
done
```

## Python for DevOps
```python
#!/usr/bin/env python3
import subprocess
import sys
from pathlib import Path

def run_cmd(cmd: list) -> str:
    result = subprocess.run(cmd, capture_output=True, text=True, check=True)
    return result.stdout.strip()

def main():
    version = run_cmd(["git", "describe", "--tags"])
    print(f"Current version: {version}")
```

## Best Practices
- Use shellcheck for bash linting
- Implement proper error handling
- Add logging and verbose modes
- Make scripts idempotent
- Use configuration files over hardcoding
