#!/usr/bin/env npx ts-node
// ABOUTME: Analyzes module exports and import dependencies for understanding codebase structure.
// ABOUTME: Maps public API surface and shows dependency graphs between modules.

import { Project, SourceFile, ExportDeclaration, ExportAssignment, Node } from "ts-morph";
import * as path from "path";

interface ExportInfo {
  name: string;
  kind: "function" | "class" | "interface" | "type" | "variable" | "const" | "namespace" | "reexport" | "default";
  file: string;
  line: number;
  fromModule?: string;
}

interface ModuleInfo {
  file: string;
  exports: ExportInfo[];
  imports: { module: string; names: string[] }[];
}

interface AnalyzeOptions {
  followReexports: boolean;
  showDeps: boolean;
  jsonOutput: boolean;
}

function getExportsFromSourceFile(sourceFile: SourceFile): ExportInfo[] {
  const exports: ExportInfo[] = [];
  const filePath = path.relative(process.cwd(), sourceFile.getFilePath());

  // Direct exports (export function, export class, etc.)
  for (const fn of sourceFile.getFunctions()) {
    if (fn.isExported()) {
      exports.push({
        name: fn.getName() || "(anonymous)",
        kind: "function",
        file: filePath,
        line: fn.getStartLineNumber(),
      });
    }
  }

  for (const cls of sourceFile.getClasses()) {
    if (cls.isExported()) {
      exports.push({
        name: cls.getName() || "(anonymous)",
        kind: "class",
        file: filePath,
        line: cls.getStartLineNumber(),
      });
    }
  }

  for (const iface of sourceFile.getInterfaces()) {
    if (iface.isExported()) {
      exports.push({
        name: iface.getName(),
        kind: "interface",
        file: filePath,
        line: iface.getStartLineNumber(),
      });
    }
  }

  for (const typeAlias of sourceFile.getTypeAliases()) {
    if (typeAlias.isExported()) {
      exports.push({
        name: typeAlias.getName(),
        kind: "type",
        file: filePath,
        line: typeAlias.getStartLineNumber(),
      });
    }
  }

  for (const varDecl of sourceFile.getVariableDeclarations()) {
    const varStmt = varDecl.getVariableStatement();
    if (varStmt?.isExported()) {
      const isConst = varStmt.getDeclarationKind() === 0; // VariableDeclarationKind.Const
      exports.push({
        name: varDecl.getName(),
        kind: isConst ? "const" : "variable",
        file: filePath,
        line: varDecl.getStartLineNumber(),
      });
    }
  }

  // Re-exports (export { x } from './module')
  for (const exportDecl of sourceFile.getExportDeclarations()) {
    const moduleSpecifier = exportDecl.getModuleSpecifierValue();

    if (exportDecl.isNamespaceExport()) {
      exports.push({
        name: "*",
        kind: "reexport",
        file: filePath,
        line: exportDecl.getStartLineNumber(),
        fromModule: moduleSpecifier,
      });
    } else {
      for (const namedExport of exportDecl.getNamedExports()) {
        const exportedName = namedExport.getAliasNode()?.getText() || namedExport.getName();
        exports.push({
          name: exportedName,
          kind: "reexport",
          file: filePath,
          line: exportDecl.getStartLineNumber(),
          fromModule: moduleSpecifier,
        });
      }
    }
  }

  // Default export
  const defaultExport = sourceFile.getDefaultExportSymbol();
  if (defaultExport) {
    const decl = defaultExport.getDeclarations()[0];
    if (decl) {
      exports.push({
        name: "default",
        kind: "default",
        file: filePath,
        line: decl.getStartLineNumber(),
      });
    }
  }

  return exports;
}

function getImportsFromSourceFile(sourceFile: SourceFile): { module: string; names: string[] }[] {
  const imports: { module: string; names: string[] }[] = [];

  for (const importDecl of sourceFile.getImportDeclarations()) {
    const moduleSpecifier = importDecl.getModuleSpecifierValue();
    const names: string[] = [];

    // Default import
    const defaultImport = importDecl.getDefaultImport();
    if (defaultImport) {
      names.push(`default as ${defaultImport.getText()}`);
    }

    // Namespace import
    const namespaceImport = importDecl.getNamespaceImport();
    if (namespaceImport) {
      names.push(`* as ${namespaceImport.getText()}`);
    }

    // Named imports
    for (const namedImport of importDecl.getNamedImports()) {
      const alias = namedImport.getAliasNode();
      if (alias) {
        names.push(`${namedImport.getName()} as ${alias.getText()}`);
      } else {
        names.push(namedImport.getName());
      }
    }

    if (names.length > 0) {
      imports.push({ module: moduleSpecifier, names });
    }
  }

  return imports;
}

function buildDependencyGraph(modules: ModuleInfo[]): Map<string, Set<string>> {
  const graph = new Map<string, Set<string>>();

  for (const mod of modules) {
    const deps = new Set<string>();

    for (const imp of mod.imports) {
      // Only include local imports (not from node_modules)
      if (imp.module.startsWith(".") || imp.module.startsWith("/")) {
        // Normalize the path
        const basePath = path.dirname(mod.file);
        const resolvedPath = path.normalize(path.join(basePath, imp.module));
        deps.add(resolvedPath);
      }
    }

    graph.set(mod.file, deps);
  }

  return graph;
}

function findCircularDeps(graph: Map<string, Set<string>>): string[][] {
  const cycles: string[][] = [];
  const visited = new Set<string>();
  const recursionStack = new Set<string>();
  const path: string[] = [];

  function dfs(node: string): void {
    visited.add(node);
    recursionStack.add(node);
    path.push(node);

    const deps = graph.get(node) || new Set();
    for (const dep of deps) {
      if (!visited.has(dep)) {
        dfs(dep);
      } else if (recursionStack.has(dep)) {
        // Found a cycle
        const cycleStart = path.indexOf(dep);
        if (cycleStart !== -1) {
          cycles.push([...path.slice(cycleStart), dep]);
        }
      }
    }

    path.pop();
    recursionStack.delete(node);
  }

  for (const node of graph.keys()) {
    if (!visited.has(node)) {
      dfs(node);
    }
  }

  return cycles;
}

function printDependencyGraph(graph: Map<string, Set<string>>): void {
  console.log("\n=== Dependency Graph ===\n");

  for (const [file, deps] of graph) {
    if (deps.size > 0) {
      console.log(`${file}`);
      for (const dep of deps) {
        console.log(`  → ${dep}`);
      }
      console.log("");
    }
  }

  const cycles = findCircularDeps(graph);
  if (cycles.length > 0) {
    console.log("⚠️  Circular Dependencies Detected:\n");
    for (const cycle of cycles) {
      console.log(`  ${cycle.join(" → ")}`);
    }
    console.log("");
  }
}

function printUsage(): void {
  console.log(`
Usage: analyze-exports.ts <path> [options]

Arguments:
  path                 File or directory to analyze

Options:
  --follow-reexports   Follow and resolve re-exports to their source
  --deps               Show dependency graph between modules
  --json               Output as JSON
  --help               Show this help message

Examples:
  analyze-exports.ts src/api/
  analyze-exports.ts src/ --deps
  analyze-exports.ts src/ --follow-reexports --json
`);
}

async function main(): Promise<void> {
  const args = process.argv.slice(2);

  if (args.includes("--help") || args.length === 0) {
    printUsage();
    process.exit(0);
  }

  const targetPath = args.find(arg => !arg.startsWith("--"));
  if (!targetPath) {
    console.error("Error: No path provided");
    printUsage();
    process.exit(1);
  }

  const options: AnalyzeOptions = {
    followReexports: args.includes("--follow-reexports"),
    showDeps: args.includes("--deps"),
    jsonOutput: args.includes("--json"),
  };

  // Create project
  const project = new Project({
    skipAddingFilesFromTsConfig: true,
  });

  // Add source files
  const absolutePath = path.resolve(targetPath);
  const fs = await import("fs");
  const stats = await fs.promises.stat(absolutePath);

  if (stats.isDirectory()) {
    project.addSourceFilesAtPaths([
      path.join(absolutePath, "**/*.ts"),
      path.join(absolutePath, "**/*.tsx"),
      path.join(absolutePath, "**/*.js"),
      path.join(absolutePath, "**/*.jsx"),
      `!${path.join(absolutePath, "**/node_modules/**")}`,
      `!${path.join(absolutePath, "**/*.d.ts")}`,
    ]);
  } else {
    project.addSourceFileAtPath(absolutePath);
  }

  const sourceFiles = project.getSourceFiles();

  if (sourceFiles.length === 0) {
    console.error("No source files found");
    process.exit(1);
  }

  const modules: ModuleInfo[] = [];

  for (const sourceFile of sourceFiles) {
    const filePath = path.relative(process.cwd(), sourceFile.getFilePath());
    modules.push({
      file: filePath,
      exports: getExportsFromSourceFile(sourceFile),
      imports: getImportsFromSourceFile(sourceFile),
    });
  }

  if (options.jsonOutput) {
    console.log(JSON.stringify(modules, null, 2));
    return;
  }

  // Print exports by file
  console.log("\n=== Module Exports ===\n");

  for (const mod of modules) {
    if (mod.exports.length === 0) continue;

    console.log(`${mod.file}:`);
    for (const exp of mod.exports) {
      const fromStr = exp.fromModule ? ` (from '${exp.fromModule}')` : "";
      console.log(`  [${exp.kind}] ${exp.name}${fromStr}`);
    }
    console.log("");
  }

  // Show dependency graph if requested
  if (options.showDeps) {
    const graph = buildDependencyGraph(modules);
    printDependencyGraph(graph);
  }
}

main().catch(err => {
  console.error("Error:", err.message);
  process.exit(1);
});
