#!/usr/bin/env npx ts-node
// ABOUTME: Detects architectural code smells like circular deps, large functions, missing JSDoc.
// ABOUTME: Quick assessment tool for identifying potential issues before deep code review.

import { Project, SourceFile, FunctionDeclaration, MethodDeclaration, ClassDeclaration, Node, SyntaxKind } from "ts-morph";
import * as path from "path";

interface SmellReport {
  type: string;
  severity: "warning" | "error";
  file: string;
  line?: number;
  message: string;
  details?: string;
}

interface SmellOptions {
  checks: Set<string>;
  exportedOnly: boolean;
  thresholds: {
    maxParams: number;
    maxFunctionLines: number;
    maxFileLines: number;
    maxNestingDepth: number;
    maxComplexity: number;
  };
}

const DEFAULT_THRESHOLDS = {
  maxParams: 5,
  maxFunctionLines: 50,
  maxFileLines: 500,
  maxNestingDepth: 4,
  maxComplexity: 10,
};

const ALL_CHECKS = [
  "circular-deps",
  "large-functions",
  "many-params",
  "missing-jsdoc",
  "deep-nesting",
  "large-files",
  "god-classes",
];

function getJsDoc(node: Node): string | null {
  const jsDocs = (node as any).getJsDocs?.();
  if (!jsDocs || jsDocs.length === 0) return null;
  return jsDocs.map((doc: any) => doc.getText()).join("\n");
}

function isExported(node: Node): boolean {
  if (Node.isExportable(node)) {
    return node.isExported();
  }
  return false;
}

function countLines(node: Node): number {
  const startLine = node.getStartLineNumber();
  const endLine = node.getEndLineNumber();
  return endLine - startLine + 1;
}

function getMaxNestingDepth(node: Node): number {
  let maxDepth = 0;

  function traverse(n: Node, depth: number): void {
    // Control flow statements increase nesting
    if (
      Node.isIfStatement(n) ||
      Node.isForStatement(n) ||
      Node.isForInStatement(n) ||
      Node.isForOfStatement(n) ||
      Node.isWhileStatement(n) ||
      Node.isDoStatement(n) ||
      Node.isTryStatement(n) ||
      Node.isSwitchStatement(n)
    ) {
      maxDepth = Math.max(maxDepth, depth + 1);
      n.forEachChild(child => traverse(child, depth + 1));
    } else {
      n.forEachChild(child => traverse(child, depth));
    }
  }

  traverse(node, 0);
  return maxDepth;
}

function getCyclomaticComplexity(node: FunctionDeclaration | MethodDeclaration): number {
  let complexity = 1; // Base complexity

  node.forEachDescendant(descendant => {
    if (
      Node.isIfStatement(descendant) ||
      Node.isConditionalExpression(descendant) ||
      Node.isForStatement(descendant) ||
      Node.isForInStatement(descendant) ||
      Node.isForOfStatement(descendant) ||
      Node.isWhileStatement(descendant) ||
      Node.isDoStatement(descendant) ||
      Node.isCaseClause(descendant) ||
      Node.isCatchClause(descendant)
    ) {
      complexity++;
    }

    // Count logical operators
    if (Node.isBinaryExpression(descendant)) {
      const op = descendant.getOperatorToken().getKind();
      if (op === SyntaxKind.AmpersandAmpersandToken || op === SyntaxKind.BarBarToken) {
        complexity++;
      }
    }
  });

  return complexity;
}

function checkLargeFunctions(sourceFile: SourceFile, options: SmellOptions): SmellReport[] {
  const reports: SmellReport[] = [];
  const filePath = path.relative(process.cwd(), sourceFile.getFilePath());

  const checkFn = (fn: FunctionDeclaration | MethodDeclaration, className?: string) => {
    if (options.exportedOnly && !isExported(fn)) return;

    const lines = countLines(fn);
    const name = fn.getName() || "(anonymous)";
    const fullName = className ? `${className}.${name}` : name;

    if (lines > options.thresholds.maxFunctionLines) {
      reports.push({
        type: "large-function",
        severity: lines > options.thresholds.maxFunctionLines * 2 ? "error" : "warning",
        file: filePath,
        line: fn.getStartLineNumber(),
        message: `${fullName} is ${lines} lines (max: ${options.thresholds.maxFunctionLines})`,
      });
    }

    // Also check complexity
    const complexity = getCyclomaticComplexity(fn);
    if (complexity > options.thresholds.maxComplexity) {
      reports.push({
        type: "high-complexity",
        severity: complexity > options.thresholds.maxComplexity * 2 ? "error" : "warning",
        file: filePath,
        line: fn.getStartLineNumber(),
        message: `${fullName} has cyclomatic complexity of ${complexity} (max: ${options.thresholds.maxComplexity})`,
      });
    }
  };

  for (const fn of sourceFile.getFunctions()) {
    checkFn(fn);
  }

  for (const cls of sourceFile.getClasses()) {
    for (const method of cls.getMethods()) {
      checkFn(method, cls.getName());
    }
  }

  return reports;
}

function checkManyParams(sourceFile: SourceFile, options: SmellOptions): SmellReport[] {
  const reports: SmellReport[] = [];
  const filePath = path.relative(process.cwd(), sourceFile.getFilePath());

  const checkFn = (fn: FunctionDeclaration | MethodDeclaration, className?: string) => {
    if (options.exportedOnly && !isExported(fn)) return;

    const params = fn.getParameters();
    const name = fn.getName() || "(anonymous)";
    const fullName = className ? `${className}.${name}` : name;

    if (params.length > options.thresholds.maxParams) {
      reports.push({
        type: "many-params",
        severity: params.length > options.thresholds.maxParams + 3 ? "error" : "warning",
        file: filePath,
        line: fn.getStartLineNumber(),
        message: `${fullName} has ${params.length} parameters (max: ${options.thresholds.maxParams})`,
        details: `Consider using an options object instead`,
      });
    }
  };

  for (const fn of sourceFile.getFunctions()) {
    checkFn(fn);
  }

  for (const cls of sourceFile.getClasses()) {
    for (const method of cls.getMethods()) {
      checkFn(method, cls.getName());
    }
  }

  return reports;
}

function checkMissingJsDoc(sourceFile: SourceFile, options: SmellOptions): SmellReport[] {
  const reports: SmellReport[] = [];
  const filePath = path.relative(process.cwd(), sourceFile.getFilePath());

  // Only check exported items (public API)
  for (const fn of sourceFile.getFunctions()) {
    if (!isExported(fn)) continue;
    if (!getJsDoc(fn)) {
      reports.push({
        type: "missing-jsdoc",
        severity: "warning",
        file: filePath,
        line: fn.getStartLineNumber(),
        message: `Exported function '${fn.getName() || "(anonymous)"}' lacks JSDoc`,
      });
    }
  }

  for (const cls of sourceFile.getClasses()) {
    if (!isExported(cls)) continue;
    if (!getJsDoc(cls)) {
      reports.push({
        type: "missing-jsdoc",
        severity: "warning",
        file: filePath,
        line: cls.getStartLineNumber(),
        message: `Exported class '${cls.getName() || "(anonymous)"}' lacks JSDoc`,
      });
    }

    // Check public methods
    for (const method of cls.getMethods()) {
      if (method.getScope() === "private") continue;
      if (!getJsDoc(method)) {
        reports.push({
          type: "missing-jsdoc",
          severity: "warning",
          file: filePath,
          line: method.getStartLineNumber(),
          message: `Public method '${cls.getName()}.${method.getName()}' lacks JSDoc`,
        });
      }
    }
  }

  return reports;
}

function checkDeepNesting(sourceFile: SourceFile, options: SmellOptions): SmellReport[] {
  const reports: SmellReport[] = [];
  const filePath = path.relative(process.cwd(), sourceFile.getFilePath());

  const checkFn = (fn: FunctionDeclaration | MethodDeclaration, className?: string) => {
    const depth = getMaxNestingDepth(fn);
    const name = fn.getName() || "(anonymous)";
    const fullName = className ? `${className}.${name}` : name;

    if (depth > options.thresholds.maxNestingDepth) {
      reports.push({
        type: "deep-nesting",
        severity: depth > options.thresholds.maxNestingDepth + 2 ? "error" : "warning",
        file: filePath,
        line: fn.getStartLineNumber(),
        message: `${fullName} has nesting depth of ${depth} (max: ${options.thresholds.maxNestingDepth})`,
        details: "Consider extracting nested logic into separate functions",
      });
    }
  };

  for (const fn of sourceFile.getFunctions()) {
    checkFn(fn);
  }

  for (const cls of sourceFile.getClasses()) {
    for (const method of cls.getMethods()) {
      checkFn(method, cls.getName());
    }
  }

  return reports;
}

function checkLargeFiles(sourceFile: SourceFile, options: SmellOptions): SmellReport[] {
  const reports: SmellReport[] = [];
  const filePath = path.relative(process.cwd(), sourceFile.getFilePath());
  const lines = sourceFile.getEndLineNumber();

  if (lines > options.thresholds.maxFileLines) {
    reports.push({
      type: "large-file",
      severity: lines > options.thresholds.maxFileLines * 2 ? "error" : "warning",
      file: filePath,
      message: `File is ${lines} lines (max: ${options.thresholds.maxFileLines})`,
      details: "Consider splitting into multiple modules",
    });
  }

  return reports;
}

function checkGodClasses(sourceFile: SourceFile, options: SmellOptions): SmellReport[] {
  const reports: SmellReport[] = [];
  const filePath = path.relative(process.cwd(), sourceFile.getFilePath());

  for (const cls of sourceFile.getClasses()) {
    const methods = cls.getMethods();
    const properties = cls.getProperties();

    if (methods.length > 20) {
      reports.push({
        type: "god-class",
        severity: methods.length > 30 ? "error" : "warning",
        file: filePath,
        line: cls.getStartLineNumber(),
        message: `Class '${cls.getName()}' has ${methods.length} methods`,
        details: "Consider splitting responsibilities into multiple classes",
      });
    }

    if (properties.length > 15) {
      reports.push({
        type: "god-class",
        severity: "warning",
        file: filePath,
        line: cls.getStartLineNumber(),
        message: `Class '${cls.getName()}' has ${properties.length} properties`,
      });
    }
  }

  return reports;
}

function checkCircularDeps(project: Project): SmellReport[] {
  const reports: SmellReport[] = [];
  const graph = new Map<string, Set<string>>();

  for (const sourceFile of project.getSourceFiles()) {
    const filePath = path.relative(process.cwd(), sourceFile.getFilePath());
    const deps = new Set<string>();

    for (const importDecl of sourceFile.getImportDeclarations()) {
      const moduleSpecifier = importDecl.getModuleSpecifierValue();
      if (moduleSpecifier.startsWith(".") || moduleSpecifier.startsWith("/")) {
        const basePath = path.dirname(filePath);
        let resolvedPath = path.normalize(path.join(basePath, moduleSpecifier));

        // Try to resolve to actual file
        for (const ext of [".ts", ".tsx", ".js", ".jsx", "/index.ts", "/index.tsx", "/index.js"]) {
          const tryPath = resolvedPath + ext;
          if (project.getSourceFile(path.resolve(tryPath))) {
            resolvedPath = tryPath;
            break;
          }
        }

        deps.add(resolvedPath);
      }
    }

    graph.set(filePath, deps);
  }

  // Find cycles using DFS
  const visited = new Set<string>();
  const recursionStack = new Set<string>();
  const pathStack: string[] = [];

  function dfs(node: string): void {
    visited.add(node);
    recursionStack.add(node);
    pathStack.push(node);

    const deps = graph.get(node) || new Set();
    for (const dep of deps) {
      if (!visited.has(dep)) {
        dfs(dep);
      } else if (recursionStack.has(dep)) {
        const cycleStart = pathStack.indexOf(dep);
        if (cycleStart !== -1) {
          const cycle = [...pathStack.slice(cycleStart), dep];
          reports.push({
            type: "circular-dependency",
            severity: "error",
            file: cycle[0],
            message: `Circular dependency detected`,
            details: cycle.join(" → "),
          });
        }
      }
    }

    pathStack.pop();
    recursionStack.delete(node);
  }

  for (const node of graph.keys()) {
    if (!visited.has(node)) {
      dfs(node);
    }
  }

  return reports;
}

function printReports(reports: SmellReport[]): void {
  if (reports.length === 0) {
    console.log("\n✅ No code smells detected!\n");
    return;
  }

  // Group by type
  const byType = new Map<string, SmellReport[]>();
  for (const report of reports) {
    const existing = byType.get(report.type) || [];
    existing.push(report);
    byType.set(report.type, existing);
  }

  const errorCount = reports.filter(r => r.severity === "error").length;
  const warningCount = reports.filter(r => r.severity === "warning").length;

  console.log(`\n=== Code Smell Report ===`);
  console.log(`Found ${errorCount} errors, ${warningCount} warnings\n`);

  for (const [type, typeReports] of byType) {
    console.log(`\n## ${type} (${typeReports.length})\n`);

    for (const report of typeReports) {
      const icon = report.severity === "error" ? "❌" : "⚠️";
      const location = report.line ? `${report.file}:${report.line}` : report.file;
      console.log(`${icon} ${location}`);
      console.log(`   ${report.message}`);
      if (report.details) {
        console.log(`   ${report.details}`);
      }
    }
  }

  console.log("");
}

function printUsage(): void {
  console.log(`
Usage: code-smells.ts <path> [options]

Arguments:
  path                 File or directory to analyze

Options:
  --check <type>       Run specific check only (can specify multiple)
                       Types: ${ALL_CHECKS.join(", ")}
  --exported           Only check exported items
  --json               Output as JSON
  --help               Show this help message

Thresholds (can override):
  --max-params N       Max parameters per function (default: ${DEFAULT_THRESHOLDS.maxParams})
  --max-fn-lines N     Max lines per function (default: ${DEFAULT_THRESHOLDS.maxFunctionLines})
  --max-file-lines N   Max lines per file (default: ${DEFAULT_THRESHOLDS.maxFileLines})
  --max-nesting N      Max nesting depth (default: ${DEFAULT_THRESHOLDS.maxNestingDepth})

Examples:
  code-smells.ts src/
  code-smells.ts src/ --check circular-deps
  code-smells.ts src/ --check large-functions --check many-params
  code-smells.ts src/ --exported --max-params 3
`);
}

async function main(): Promise<void> {
  const args = process.argv.slice(2);

  if (args.includes("--help") || args.length === 0) {
    printUsage();
    process.exit(0);
  }

  const targetPath = args.find(arg => !arg.startsWith("--"));
  if (!targetPath) {
    console.error("Error: No path provided");
    printUsage();
    process.exit(1);
  }

  // Parse checks
  const checks = new Set<string>();
  let i = 0;
  while (i < args.length) {
    if (args[i] === "--check" && args[i + 1]) {
      checks.add(args[i + 1]);
      i += 2;
    } else {
      i++;
    }
  }

  // If no specific checks, run all
  if (checks.size === 0) {
    ALL_CHECKS.forEach(c => checks.add(c));
  }

  // Parse thresholds
  const thresholds = { ...DEFAULT_THRESHOLDS };
  const maxParamsIdx = args.indexOf("--max-params");
  if (maxParamsIdx !== -1 && args[maxParamsIdx + 1]) {
    thresholds.maxParams = parseInt(args[maxParamsIdx + 1], 10);
  }
  const maxFnLinesIdx = args.indexOf("--max-fn-lines");
  if (maxFnLinesIdx !== -1 && args[maxFnLinesIdx + 1]) {
    thresholds.maxFunctionLines = parseInt(args[maxFnLinesIdx + 1], 10);
  }
  const maxFileLinesIdx = args.indexOf("--max-file-lines");
  if (maxFileLinesIdx !== -1 && args[maxFileLinesIdx + 1]) {
    thresholds.maxFileLines = parseInt(args[maxFileLinesIdx + 1], 10);
  }
  const maxNestingIdx = args.indexOf("--max-nesting");
  if (maxNestingIdx !== -1 && args[maxNestingIdx + 1]) {
    thresholds.maxNestingDepth = parseInt(args[maxNestingIdx + 1], 10);
  }

  const options: SmellOptions = {
    checks,
    exportedOnly: args.includes("--exported"),
    thresholds,
  };

  const jsonOutput = args.includes("--json");

  // Create project
  const project = new Project({
    skipAddingFilesFromTsConfig: true,
  });

  // Add source files
  const absolutePath = path.resolve(targetPath);
  const fs = await import("fs");
  const stats = await fs.promises.stat(absolutePath);

  if (stats.isDirectory()) {
    project.addSourceFilesAtPaths([
      path.join(absolutePath, "**/*.ts"),
      path.join(absolutePath, "**/*.tsx"),
      path.join(absolutePath, "**/*.js"),
      path.join(absolutePath, "**/*.jsx"),
      `!${path.join(absolutePath, "**/node_modules/**")}`,
      `!${path.join(absolutePath, "**/*.d.ts")}`,
    ]);
  } else {
    project.addSourceFileAtPath(absolutePath);
  }

  const sourceFiles = project.getSourceFiles();

  if (sourceFiles.length === 0) {
    console.error("No source files found");
    process.exit(1);
  }

  const allReports: SmellReport[] = [];

  // Run checks
  if (checks.has("circular-deps")) {
    allReports.push(...checkCircularDeps(project));
  }

  for (const sourceFile of sourceFiles) {
    if (checks.has("large-functions")) {
      allReports.push(...checkLargeFunctions(sourceFile, options));
    }
    if (checks.has("many-params")) {
      allReports.push(...checkManyParams(sourceFile, options));
    }
    if (checks.has("missing-jsdoc")) {
      allReports.push(...checkMissingJsDoc(sourceFile, options));
    }
    if (checks.has("deep-nesting")) {
      allReports.push(...checkDeepNesting(sourceFile, options));
    }
    if (checks.has("large-files")) {
      allReports.push(...checkLargeFiles(sourceFile, options));
    }
    if (checks.has("god-classes")) {
      allReports.push(...checkGodClasses(sourceFile, options));
    }
  }

  if (jsonOutput) {
    console.log(JSON.stringify(allReports, null, 2));
  } else {
    printReports(allReports);
  }

  // Exit with error code if errors found
  const hasErrors = allReports.some(r => r.severity === "error");
  if (hasErrors) {
    process.exit(1);
  }
}

main().catch(err => {
  console.error("Error:", err.message);
  process.exit(1);
});
