#!/usr/bin/env npx ts-node
// ABOUTME: Extracts function, method, and class signatures with JSDoc comments from TypeScript/JavaScript files.
// ABOUTME: Outputs lightweight API surface for architectural analysis without full file reads.

import { Project, SourceFile, FunctionDeclaration, MethodDeclaration, ClassDeclaration, InterfaceDeclaration, TypeAliasDeclaration, Node, SyntaxKind, JSDoc } from "ts-morph";
import * as path from "path";

interface ExtractOptions {
  exportedOnly: boolean;
  includeTypes: boolean;
  jsonOutput: boolean;
  includePrivate: boolean;
}

interface SignatureInfo {
  file: string;
  name: string;
  kind: "function" | "method" | "class" | "interface" | "type";
  signature: string;
  jsdoc: string | null;
  exported: boolean;
  line: number;
  className?: string;
}

function getJsDoc(node: Node): string | null {
  const jsDocs = (node as any).getJsDocs?.() as JSDoc[] | undefined;
  if (!jsDocs || jsDocs.length === 0) return null;
  return jsDocs.map(doc => doc.getText()).join("\n");
}

function getFunctionSignature(fn: FunctionDeclaration | MethodDeclaration): string {
  const name = fn.getName() || "(anonymous)";
  const typeParams = fn.getTypeParameters().map(tp => tp.getText()).join(", ");
  const params = fn.getParameters().map(p => {
    const optional = p.isOptional() ? "?" : "";
    const type = p.getType().getText();
    return `${p.getName()}${optional}: ${type}`;
  }).join(", ");
  const returnType = fn.getReturnType().getText();
  const async = fn.isAsync() ? "async " : "";
  const typeParamStr = typeParams ? `<${typeParams}>` : "";

  return `${async}function ${name}${typeParamStr}(${params}): ${returnType}`;
}

function getMethodSignature(method: MethodDeclaration, className: string): string {
  const name = method.getName();
  const typeParams = method.getTypeParameters().map(tp => tp.getText()).join(", ");
  const params = method.getParameters().map(p => {
    const optional = p.isOptional() ? "?" : "";
    const type = p.getType().getText();
    return `${p.getName()}${optional}: ${type}`;
  }).join(", ");
  const returnType = method.getReturnType().getText();
  const async = method.isAsync() ? "async " : "";
  const staticMod = method.isStatic() ? "static " : "";
  const typeParamStr = typeParams ? `<${typeParams}>` : "";

  return `${staticMod}${async}${name}${typeParamStr}(${params}): ${returnType}`;
}

function getClassSignature(cls: ClassDeclaration): string {
  const name = cls.getName() || "(anonymous)";
  const typeParams = cls.getTypeParameters().map(tp => tp.getText()).join(", ");
  const typeParamStr = typeParams ? `<${typeParams}>` : "";
  const extendsClause = cls.getExtends()?.getText();
  const implementsClauses = cls.getImplements().map(i => i.getText()).join(", ");

  let sig = `class ${name}${typeParamStr}`;
  if (extendsClause) sig += ` extends ${extendsClause}`;
  if (implementsClauses) sig += ` implements ${implementsClauses}`;
  return sig;
}

function getInterfaceSignature(iface: InterfaceDeclaration): string {
  const name = iface.getName();
  const typeParams = iface.getTypeParameters().map(tp => tp.getText()).join(", ");
  const typeParamStr = typeParams ? `<${typeParams}>` : "";
  const extendsClause = iface.getExtends().map(e => e.getText()).join(", ");

  let sig = `interface ${name}${typeParamStr}`;
  if (extendsClause) sig += ` extends ${extendsClause}`;
  return sig;
}

function getTypeSignature(typeAlias: TypeAliasDeclaration): string {
  const name = typeAlias.getName();
  const typeParams = typeAlias.getTypeParameters().map(tp => tp.getText()).join(", ");
  const typeParamStr = typeParams ? `<${typeParams}>` : "";
  const typeText = typeAlias.getType().getText();

  return `type ${name}${typeParamStr} = ${typeText}`;
}

function isExported(node: Node): boolean {
  if (Node.isExportable(node)) {
    return node.isExported();
  }
  return false;
}

function extractFromSourceFile(sourceFile: SourceFile, options: ExtractOptions): SignatureInfo[] {
  const signatures: SignatureInfo[] = [];
  const filePath = sourceFile.getFilePath();
  const relativePath = path.relative(process.cwd(), filePath);

  // Extract functions
  for (const fn of sourceFile.getFunctions()) {
    const exported = isExported(fn);
    if (options.exportedOnly && !exported) continue;

    signatures.push({
      file: relativePath,
      name: fn.getName() || "(anonymous)",
      kind: "function",
      signature: getFunctionSignature(fn),
      jsdoc: getJsDoc(fn),
      exported,
      line: fn.getStartLineNumber(),
    });
  }

  // Extract classes and their methods
  for (const cls of sourceFile.getClasses()) {
    const classExported = isExported(cls);
    if (options.exportedOnly && !classExported) continue;

    const className = cls.getName() || "(anonymous)";

    signatures.push({
      file: relativePath,
      name: className,
      kind: "class",
      signature: getClassSignature(cls),
      jsdoc: getJsDoc(cls),
      exported: classExported,
      line: cls.getStartLineNumber(),
    });

    // Extract methods
    for (const method of cls.getMethods()) {
      const isPrivate = method.getScope() === "private";
      if (!options.includePrivate && isPrivate) continue;

      signatures.push({
        file: relativePath,
        name: method.getName(),
        kind: "method",
        signature: getMethodSignature(method, className),
        jsdoc: getJsDoc(method),
        exported: classExported,
        line: method.getStartLineNumber(),
        className,
      });
    }
  }

  // Extract interfaces and types if requested
  if (options.includeTypes) {
    for (const iface of sourceFile.getInterfaces()) {
      const exported = isExported(iface);
      if (options.exportedOnly && !exported) continue;

      signatures.push({
        file: relativePath,
        name: iface.getName(),
        kind: "interface",
        signature: getInterfaceSignature(iface),
        jsdoc: getJsDoc(iface),
        exported,
        line: iface.getStartLineNumber(),
      });
    }

    for (const typeAlias of sourceFile.getTypeAliases()) {
      const exported = isExported(typeAlias);
      if (options.exportedOnly && !exported) continue;

      signatures.push({
        file: relativePath,
        name: typeAlias.getName(),
        kind: "type",
        signature: getTypeSignature(typeAlias),
        jsdoc: getJsDoc(typeAlias),
        exported,
        line: typeAlias.getStartLineNumber(),
      });
    }
  }

  return signatures;
}

function formatSignature(sig: SignatureInfo): string {
  const lines: string[] = [];

  if (sig.jsdoc) {
    lines.push(sig.jsdoc);
  }

  const exportPrefix = sig.exported ? "export " : "";

  if (sig.kind === "method" && sig.className) {
    lines.push(`  ${sig.signature}`);
  } else {
    lines.push(`${exportPrefix}${sig.signature}`);
  }

  return lines.join("\n");
}

function groupByFile(signatures: SignatureInfo[]): Map<string, SignatureInfo[]> {
  const grouped = new Map<string, SignatureInfo[]>();
  for (const sig of signatures) {
    const existing = grouped.get(sig.file) || [];
    existing.push(sig);
    grouped.set(sig.file, existing);
  }
  return grouped;
}

function printUsage(): void {
  console.log(`
Usage: extract-signatures.ts <path> [options]

Arguments:
  path                 File or directory to analyze

Options:
  --exported           Only include exported items
  --types              Include interfaces and type aliases
  --private            Include private methods
  --json               Output as JSON
  --help               Show this help message

Examples:
  extract-signatures.ts src/api/users.ts
  extract-signatures.ts src/ --exported
  extract-signatures.ts src/ --types --json
`);
}

async function main(): Promise<void> {
  const args = process.argv.slice(2);

  if (args.includes("--help") || args.length === 0) {
    printUsage();
    process.exit(0);
  }

  const targetPath = args.find(arg => !arg.startsWith("--"));
  if (!targetPath) {
    console.error("Error: No path provided");
    printUsage();
    process.exit(1);
  }

  const options: ExtractOptions = {
    exportedOnly: args.includes("--exported"),
    includeTypes: args.includes("--types"),
    jsonOutput: args.includes("--json"),
    includePrivate: args.includes("--private"),
  };

  // Create project - try to find tsconfig.json
  const project = new Project({
    skipAddingFilesFromTsConfig: true,
  });

  // Add source files
  const absolutePath = path.resolve(targetPath);
  const stats = await import("fs").then(fs => fs.promises.stat(absolutePath));

  if (stats.isDirectory()) {
    project.addSourceFilesAtPaths([
      path.join(absolutePath, "**/*.ts"),
      path.join(absolutePath, "**/*.tsx"),
      path.join(absolutePath, "**/*.js"),
      path.join(absolutePath, "**/*.jsx"),
      `!${path.join(absolutePath, "**/node_modules/**")}`,
      `!${path.join(absolutePath, "**/*.d.ts")}`,
    ]);
  } else {
    project.addSourceFileAtPath(absolutePath);
  }

  const sourceFiles = project.getSourceFiles();

  if (sourceFiles.length === 0) {
    console.error("No source files found");
    process.exit(1);
  }

  const allSignatures: SignatureInfo[] = [];
  for (const sourceFile of sourceFiles) {
    const sigs = extractFromSourceFile(sourceFile, options);
    allSignatures.push(...sigs);
  }

  if (options.jsonOutput) {
    console.log(JSON.stringify(allSignatures, null, 2));
  } else {
    const grouped = groupByFile(allSignatures);

    for (const [file, sigs] of grouped) {
      console.log(`\n// ${file}\n`);

      // Group by class for methods
      let currentClass: string | null = null;

      for (const sig of sigs) {
        if (sig.kind === "class") {
          currentClass = sig.name;
          console.log(formatSignature(sig) + " {");
        } else if (sig.kind === "method" && sig.className === currentClass) {
          console.log(formatSignature(sig));
        } else {
          if (currentClass !== null && sig.className !== currentClass) {
            console.log("}\n");
            currentClass = null;
          }
          console.log(formatSignature(sig));
        }
      }

      if (currentClass !== null) {
        console.log("}");
      }
    }
  }
}

main().catch(err => {
  console.error("Error:", err.message);
  process.exit(1);
});
