#!/usr/bin/env npx ts-node
// ABOUTME: Traces call hierarchies up (callers) or down (callees) for a function/method.
// ABOUTME: Helps follow data flow and debug issues by showing the full call chain.

import { Project, SourceFile, Node, SyntaxKind, FunctionDeclaration, MethodDeclaration, CallExpression, Identifier } from "ts-morph";
import * as path from "path";

interface CallNode {
  name: string;
  file: string;
  line: number;
  signature?: string;
  children: CallNode[];
}

interface TraceOptions {
  direction: "up" | "down" | "both";
  maxDepth: number;
  showTree: boolean;
  showUsage: boolean;
}

function findFunctionByName(project: Project, filePath: string, functionName: string): FunctionDeclaration | MethodDeclaration | null {
  const sourceFile = project.getSourceFile(filePath);
  if (!sourceFile) {
    console.error(`File not found: ${filePath}`);
    return null;
  }

  // Try function declarations first
  const fn = sourceFile.getFunction(functionName);
  if (fn) return fn;

  // Try method declarations in classes
  for (const cls of sourceFile.getClasses()) {
    const method = cls.getMethod(functionName);
    if (method) return method;
  }

  // Try to find by traversing and matching name
  let found: FunctionDeclaration | MethodDeclaration | null = null;
  sourceFile.forEachDescendant(node => {
    if (Node.isFunctionDeclaration(node) && node.getName() === functionName) {
      found = node;
      return true;
    }
    if (Node.isMethodDeclaration(node) && node.getName() === functionName) {
      found = node;
      return true;
    }
    return false;
  });

  return found;
}

function getCallers(node: FunctionDeclaration | MethodDeclaration, project: Project, depth: number, maxDepth: number, visited: Set<string>): CallNode[] {
  if (depth >= maxDepth) return [];

  const callers: CallNode[] = [];
  const name = node.getName() || "(anonymous)";
  const nodeKey = `${node.getSourceFile().getFilePath()}:${name}`;

  if (visited.has(nodeKey)) return [];
  visited.add(nodeKey);

  const references = node.findReferencesAsNodes();

  for (const ref of references) {
    // Skip the definition itself
    if (ref === node || ref === node.getNameNode()) continue;

    // Find the containing function/method
    const containingFn = ref.getFirstAncestor(ancestor =>
      Node.isFunctionDeclaration(ancestor) || Node.isMethodDeclaration(ancestor) || Node.isArrowFunction(ancestor)
    );

    if (containingFn && (Node.isFunctionDeclaration(containingFn) || Node.isMethodDeclaration(containingFn))) {
      const callerName = containingFn.getName() || "(anonymous)";
      const callerFile = path.relative(process.cwd(), containingFn.getSourceFile().getFilePath());
      const callerLine = containingFn.getStartLineNumber();
      const callerKey = `${containingFn.getSourceFile().getFilePath()}:${callerName}`;

      if (!visited.has(callerKey)) {
        const children = getCallers(containingFn, project, depth + 1, maxDepth, visited);
        callers.push({
          name: callerName,
          file: callerFile,
          line: callerLine,
          children,
        });
      }
    }
  }

  return callers;
}

function getCallees(node: FunctionDeclaration | MethodDeclaration, project: Project, depth: number, maxDepth: number, visited: Set<string>): CallNode[] {
  if (depth >= maxDepth) return [];

  const callees: CallNode[] = [];
  const name = node.getName() || "(anonymous)";
  const nodeKey = `${node.getSourceFile().getFilePath()}:${name}`;

  if (visited.has(nodeKey)) return [];
  visited.add(nodeKey);

  // Find all call expressions within this function
  const body = node.getBody();
  if (!body) return [];

  body.forEachDescendant(descendant => {
    if (Node.isCallExpression(descendant)) {
      const expression = descendant.getExpression();
      let calledName = "";

      if (Node.isIdentifier(expression)) {
        calledName = expression.getText();
      } else if (Node.isPropertyAccessExpression(expression)) {
        calledName = expression.getName();
      }

      if (calledName) {
        // Try to find the definition
        const definitions = expression.getType().getSymbol()?.getDeclarations() || [];

        for (const def of definitions) {
          if (Node.isFunctionDeclaration(def) || Node.isMethodDeclaration(def)) {
            const calleeFile = path.relative(process.cwd(), def.getSourceFile().getFilePath());
            const calleeLine = def.getStartLineNumber();
            const calleeKey = `${def.getSourceFile().getFilePath()}:${calledName}`;

            if (!visited.has(calleeKey)) {
              const children = getCallees(def, project, depth + 1, maxDepth, new Set(visited));
              callees.push({
                name: calledName,
                file: calleeFile,
                line: calleeLine,
                children,
              });
            }
          }
        }
      }
    }
  });

  return callees;
}

function printTree(nodes: CallNode[], prefix: string = "", isLast: boolean = true, direction: "up" | "down"): void {
  const dirLabel = direction === "up" ? "called by" : "calls";

  for (let i = 0; i < nodes.length; i++) {
    const node = nodes[i];
    const isLastNode = i === nodes.length - 1;
    const connector = isLastNode ? "└── " : "├── ";
    const childPrefix = isLastNode ? "    " : "│   ";

    console.log(`${prefix}${connector}${dirLabel}: ${node.name} (${node.file}:${node.line})`);

    if (node.children.length > 0) {
      printTree(node.children, prefix + childPrefix, isLastNode, direction);
    }
  }
}

function printFlat(nodes: CallNode[], indent: number = 0, direction: "up" | "down"): void {
  const indentStr = "  ".repeat(indent);
  const dirLabel = direction === "up" ? "called by" : "calls";

  for (const node of nodes) {
    console.log(`${indentStr}${dirLabel}: ${node.name} (${node.file}:${node.line})`);
    if (node.children.length > 0) {
      printFlat(node.children, indent + 1, direction);
    }
  }
}

function printUsage(): void {
  console.log(`
Usage: trace-calls.ts <file>:<function> [options]

Arguments:
  file:function        File path and function name (e.g., src/api.ts:getUser)

Options:
  --up                 Show callers (who calls this function?)
  --down               Show callees (what does this function call?)
  --depth N            Maximum trace depth (default: 3)
  --tree               Output as ASCII tree
  --show-usage         Show how return values are used (with --down)
  --help               Show this help message

Examples:
  trace-calls.ts src/api/users.ts:getUser --up
  trace-calls.ts src/api/users.ts:getUser --down --depth 5
  trace-calls.ts src/api/users.ts:getUser --up --tree
`);
}

async function main(): Promise<void> {
  const args = process.argv.slice(2);

  if (args.includes("--help") || args.length === 0) {
    printUsage();
    process.exit(0);
  }

  // Parse target (file:function)
  const target = args.find(arg => !arg.startsWith("--") && arg.includes(":"));
  if (!target) {
    console.error("Error: Must specify file:function (e.g., src/api.ts:getUser)");
    printUsage();
    process.exit(1);
  }

  const [filePath, functionName] = target.split(":");
  if (!filePath || !functionName) {
    console.error("Error: Invalid format. Use file:function (e.g., src/api.ts:getUser)");
    process.exit(1);
  }

  // Parse options
  const options: TraceOptions = {
    direction: args.includes("--down") ? (args.includes("--up") ? "both" : "down") : "up",
    maxDepth: 3,
    showTree: args.includes("--tree"),
    showUsage: args.includes("--show-usage"),
  };

  const depthIndex = args.indexOf("--depth");
  if (depthIndex !== -1 && args[depthIndex + 1]) {
    options.maxDepth = parseInt(args[depthIndex + 1], 10);
  }

  // Create project
  const project = new Project({
    skipAddingFilesFromTsConfig: true,
  });

  // Add all source files in the directory
  const absolutePath = path.resolve(filePath);
  const dirPath = path.dirname(absolutePath);

  project.addSourceFilesAtPaths([
    path.join(dirPath, "**/*.ts"),
    path.join(dirPath, "**/*.tsx"),
    path.join(dirPath, "**/*.js"),
    path.join(dirPath, "**/*.jsx"),
    `!${path.join(dirPath, "**/node_modules/**")}`,
  ]);

  // Also add the specific file
  project.addSourceFileAtPath(absolutePath);

  // Find the function
  const fn = findFunctionByName(project, absolutePath, functionName);
  if (!fn) {
    console.error(`Function '${functionName}' not found in ${filePath}`);
    process.exit(1);
  }

  const fnLine = fn.getStartLineNumber();
  const relPath = path.relative(process.cwd(), absolutePath);

  console.log(`\n${functionName} (${relPath}:${fnLine})`);

  if (options.direction === "up" || options.direction === "both") {
    const callers = getCallers(fn, project, 0, options.maxDepth, new Set());

    if (callers.length === 0) {
      console.log("  No callers found");
    } else if (options.showTree) {
      printTree(callers, "", true, "up");
    } else {
      printFlat(callers, 1, "up");
    }
  }

  if (options.direction === "down" || options.direction === "both") {
    if (options.direction === "both") console.log("");

    const callees = getCallees(fn, project, 0, options.maxDepth, new Set());

    if (callees.length === 0) {
      console.log("  No callees found");
    } else if (options.showTree) {
      printTree(callees, "", true, "down");
    } else {
      printFlat(callees, 1, "down");
    }
  }

  console.log("");
}

main().catch(err => {
  console.error("Error:", err.message);
  process.exit(1);
});
