#!/usr/bin/env python3
"""
API 测试报告生成器
解析 api-registry.md 并生成测试报告
"""

import re
import sys
import argparse
from datetime import datetime
from pathlib import Path
from collections import defaultdict


def parse_registry(content: str) -> dict:
    """解析 API 注册表内容"""
    result = {
        "environment": {},
        "endpoints": []
    }
    
    # 解析环境变量
    env_match = re.search(r'## Environment\s*\n\|[^\n]+\n\|[-|\s]+\n((?:\|[^\n]+\n)+)', content)
    if env_match:
        for line in env_match.group(1).strip().split('\n'):
            parts = [p.strip() for p in line.split('|')[1:-1]]
            if len(parts) >= 2:
                result["environment"][parts[0]] = parts[1]
    
    # 解析接口
    endpoint_pattern = re.compile(
        r'### \[(\w+)\] ([^\n-]+) - ([^\n]+)\n'
        r'((?:- [^\n]+\n)+)'
        r'(?:\n\*\*Request:\*\*\n```[^\n]*\n([\s\S]*?)```)?',
        re.MULTILINE
    )
    
    for match in endpoint_pattern.finditer(content):
        method = match.group(1)
        path = match.group(2).strip()
        name = match.group(3).strip()
        metadata_block = match.group(4)
        request = match.group(5).strip() if match.group(5) else ""
        
        # 解析元数据
        metadata = {}
        for line in metadata_block.strip().split('\n'):
            if ':' in line:
                key, value = line.split(':', 1)
                key = key.replace('-', '').strip().lower()
                metadata[key] = value.strip()
        
        result["endpoints"].append({
            "method": method,
            "path": path,
            "name": name,
            "status": metadata.get("status", "pending"),
            "depends": metadata.get("depends", "none"),
            "last_tested": metadata.get("last tested", ""),
            "notes": metadata.get("notes", ""),
            "request": request
        })
    
    return result


def generate_report(registry: dict, output_format: str = "markdown") -> str:
    """生成测试报告"""
    endpoints = registry["endpoints"]
    
    # 统计
    stats = defaultdict(int)
    for ep in endpoints:
        stats[ep["status"]] += 1
    
    total = len(endpoints)
    tested = stats.get("tested", 0)
    pending = stats.get("pending", 0)
    failed = stats.get("failed", 0)
    blocked = stats.get("blocked", 0)
    
    coverage = (tested / total * 100) if total > 0 else 0
    
    # 生成报告
    lines = [
        "# API 测试报告",
        "",
        f"**生成时间:** {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}",
        "",
        "## 概览",
        "",
        f"| 指标 | 数值 |",
        f"|------|------|",
        f"| 接口总数 | {total} |",
        f"| 已测试 | {tested} |",
        f"| 待测试 | {pending} |",
        f"| 测试失败 | {failed} |",
        f"| 被阻塞 | {blocked} |",
        f"| 测试覆盖率 | {coverage:.1f}% |",
        "",
        "## 测试进度",
        "",
        _generate_progress_bar(coverage),
        "",
    ]
    
    # 按状态分组
    if failed:
        lines.extend([
            "## 失败的接口",
            "",
        ])
        for ep in endpoints:
            if ep["status"] == "failed":
                lines.append(f"- **[{ep['method']}] {ep['path']}** - {ep['name']}")
                if ep["notes"]:
                    lines.append(f"  - {ep['notes']}")
        lines.append("")
    
    if blocked:
        lines.extend([
            "## 被阻塞的接口",
            "",
        ])
        for ep in endpoints:
            if ep["status"] == "blocked":
                lines.append(f"- **[{ep['method']}] {ep['path']}** - {ep['name']}")
                lines.append(f"  - 依赖: {ep['depends']}")
        lines.append("")
    
    if pending:
        lines.extend([
            "## 待测试的接口",
            "",
        ])
        for ep in endpoints:
            if ep["status"] == "pending":
                lines.append(f"- [{ep['method']}] {ep['path']} - {ep['name']}")
        lines.append("")
    
    # 完整接口清单
    lines.extend([
        "## 完整接口清单",
        "",
        "| Method | Path | Name | Status | Last Tested |",
        "|--------|------|------|--------|-------------|",
    ])
    
    status_emoji = {
        "tested": "Pass",
        "pending": "Pending",
        "failed": "Fail",
        "blocked": "Blocked",
        "deprecated": "Deprecated"
    }
    
    for ep in endpoints:
        status = status_emoji.get(ep["status"], ep["status"])
        last_tested = ep["last_tested"] or "-"
        lines.append(f"| {ep['method']} | {ep['path']} | {ep['name']} | {status} | {last_tested} |")
    
    return "\n".join(lines)


def _generate_progress_bar(percentage: float, width: int = 20) -> str:
    """生成文本进度条"""
    filled = int(width * percentage / 100)
    empty = width - filled
    bar = "█" * filled + "░" * empty
    return f"`[{bar}]` {percentage:.1f}%"


def main():
    parser = argparse.ArgumentParser(description="生成 API 测试报告")
    parser.add_argument("registry", help="API 注册表文件路径")
    parser.add_argument("-o", "--output", help="输出文件路径", default=None)
    
    args = parser.parse_args()
    
    # 读取注册表
    registry_path = Path(args.registry)
    if not registry_path.exists():
        print(f"错误: 文件不存在 - {registry_path}", file=sys.stderr)
        sys.exit(1)
    
    content = registry_path.read_text(encoding="utf-8")
    registry = parse_registry(content)
    
    # 生成报告
    report = generate_report(registry)
    
    # 输出
    if args.output:
        output_path = Path(args.output)
        output_path.write_text(report, encoding="utf-8")
        print(f"报告已生成: {output_path}")
    else:
        print(report)


if __name__ == "__main__":
    main()
