#!/bin/bash
# package-solution.sh
# Packages a Power Automate flow JSON into a deployable solution ZIP
#
# Usage: ./package-solution.sh <flow.json> [output-name] [publisher-prefix]
#
# Example:
#   ./package-solution.sh my-flow.json "MyFlowSolution" "rob"
#   -> Creates: MyFlowSolution_1.0.0.zip

set -e

FLOW_JSON="${1:-}"
SOLUTION_NAME="${2:-GeneratedSolution}"
PUBLISHER_PREFIX="${3:-pai}"
VERSION="1.0.0"

# Validate input
if [[ -z "$FLOW_JSON" ]] || [[ ! -f "$FLOW_JSON" ]]; then
    echo "Error: Flow JSON file required"
    echo "Usage: $0 <flow.json> [solution-name] [publisher-prefix]"
    exit 1
fi

# Generate GUIDs
SOLUTION_ID=$(uuidgen | tr '[:upper:]' '[:lower:]')
FLOW_ID=$(uuidgen | tr '[:upper:]' '[:lower:]')
PUBLISHER_ID=$(uuidgen | tr '[:upper:]' '[:lower:]')

# Create temp directory
TEMP_DIR=$(mktemp -d)
WORKFLOW_DIR="$TEMP_DIR/Workflows"
mkdir -p "$WORKFLOW_DIR"

echo "Packaging solution: $SOLUTION_NAME"
echo "  Flow: $FLOW_JSON"
echo "  Solution ID: $SOLUTION_ID"
echo "  Flow ID: $FLOW_ID"

# Extract flow name from JSON or use filename
FLOW_NAME=$(basename "$FLOW_JSON" .json | tr ' ' '_')
FLOW_DISPLAY_NAME=$(jq -r '.properties.displayName // empty' "$FLOW_JSON" 2>/dev/null || echo "$FLOW_NAME")
if [[ -z "$FLOW_DISPLAY_NAME" ]]; then
    FLOW_DISPLAY_NAME="$FLOW_NAME"
fi

# Create [Content_Types].xml
cat > "$TEMP_DIR/[Content_Types].xml" << 'EOF'
<?xml version="1.0" encoding="utf-8"?>
<Types xmlns="http://schemas.openxmlformats.org/package/2006/content-types">
  <Default Extension="xml" ContentType="application/octet-stream" />
  <Default Extension="json" ContentType="application/octet-stream" />
</Types>
EOF

# Create solution.xml
cat > "$TEMP_DIR/solution.xml" << EOF
<?xml version="1.0" encoding="utf-8"?>
<ImportExportXml version="9.2.24014.199" SolutionPackageVersion="9.2" languagecode="1033" generatedBy="PAI" xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance">
  <SolutionManifest>
    <UniqueName>${PUBLISHER_PREFIX}_${SOLUTION_NAME}</UniqueName>
    <LocalizedNames>
      <LocalizedName description="${SOLUTION_NAME}" languagecode="1033" />
    </LocalizedNames>
    <Descriptions>
      <Description description="Generated by PAI PowerPlatform Skill" languagecode="1033" />
    </Descriptions>
    <Version>${VERSION}</Version>
    <Managed>0</Managed>
    <Publisher>
      <UniqueName>${PUBLISHER_PREFIX}</UniqueName>
      <LocalizedNames>
        <LocalizedName description="${PUBLISHER_PREFIX}" languagecode="1033" />
      </LocalizedNames>
      <Descriptions>
        <Description description="PAI Generated" languagecode="1033" />
      </Descriptions>
      <EMailAddress xsi:nil="true" />
      <SupportingWebsiteUrl xsi:nil="true" />
      <CustomizationPrefix>${PUBLISHER_PREFIX}</CustomizationPrefix>
      <CustomizationOptionValuePrefix>10000</CustomizationOptionValuePrefix>
      <Addresses>
        <Address>
          <AddressNumber>1</AddressNumber>
          <AddressTypeCode>1</AddressTypeCode>
        </Address>
        <Address>
          <AddressNumber>2</AddressNumber>
          <AddressTypeCode>1</AddressTypeCode>
        </Address>
      </Addresses>
    </Publisher>
    <RootComponents>
      <RootComponent type="29" id="{${FLOW_ID}}" behavior="0" />
    </RootComponents>
    <MissingDependencies />
  </SolutionManifest>
</ImportExportXml>
EOF

# Create customizations.xml
cat > "$TEMP_DIR/customizations.xml" << EOF
<?xml version="1.0" encoding="utf-8"?>
<ImportExportXml xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance">
  <Workflows>
    <Workflow WorkflowId="{${FLOW_ID}}" Name="${FLOW_DISPLAY_NAME}">
      <JsonFileName>/Workflows/${FLOW_ID}.json</JsonFileName>
      <Type>1</Type>
      <Subprocess>0</Subprocess>
      <Category>5</Category>
      <Mode>0</Mode>
      <Scope>4</Scope>
      <OnDemand>0</OnDemand>
      <TriggerOnCreate>0</TriggerOnCreate>
      <TriggerOnDelete>0</TriggerOnDelete>
      <AsyncAutodelete>0</AsyncAutodelete>
      <SyncWorkflowLogOnFailure>0</SyncWorkflowLogOnFailure>
      <StateCode>1</StateCode>
      <StatusCode>2</StatusCode>
      <RunAs>1</RunAs>
      <IsTransacted>1</IsTransacted>
      <IntroducedVersion>${VERSION}</IntroducedVersion>
      <IsCustomizable>1</IsCustomizable>
      <BusinessProcessType>0</BusinessProcessType>
      <IsCustomProcessingStepAllowedForOtherPublishers>1</IsCustomProcessingStepAllowedForOtherPublishers>
      <PrimaryEntity>none</PrimaryEntity>
      <LocalizedNames>
        <LocalizedName languagecode="1033" description="${FLOW_DISPLAY_NAME}" />
      </LocalizedNames>
      <Descriptions>
        <Description languagecode="1033" description="Generated by PAI" />
      </Descriptions>
    </Workflow>
  </Workflows>
</ImportExportXml>
EOF

# Wrap the flow JSON in proper structure
FLOW_CONTENT=$(cat "$FLOW_JSON")

# Check if flow JSON already has clientData wrapper
if echo "$FLOW_CONTENT" | jq -e '.properties.definition' > /dev/null 2>&1; then
    # Already has proper structure
    WRAPPED_FLOW="$FLOW_CONTENT"
elif echo "$FLOW_CONTENT" | jq -e '.triggers' > /dev/null 2>&1; then
    # Raw definition, needs wrapping
    WRAPPED_FLOW=$(jq -n \
        --argjson def "$FLOW_CONTENT" \
        --arg name "$FLOW_DISPLAY_NAME" \
        '{
            "properties": {
                "definition": $def,
                "displayName": $name,
                "connectionReferences": {},
                "flowFailureAlertSubscribed": false,
                "isManaged": false
            },
            "schemaVersion": "1.0.0.0"
        }')
else
    # Assume it's the full flow structure
    WRAPPED_FLOW="$FLOW_CONTENT"
fi

echo "$WRAPPED_FLOW" > "$WORKFLOW_DIR/${FLOW_ID}.json"

# Create the ZIP
OUTPUT_DIR="${OUTPUT_DIR:-$(dirname "$FLOW_JSON")}"
OUTPUT_FILE="$OUTPUT_DIR/${SOLUTION_NAME}_${VERSION}.zip"

cd "$TEMP_DIR"
zip -r "$OUTPUT_FILE" . -x "*.DS_Store" > /dev/null

# Cleanup
rm -rf "$TEMP_DIR"

echo ""
echo "Solution created: $OUTPUT_FILE"
echo ""
echo "To import:"
echo "  1. Go to make.powerapps.com"
echo "  2. Select your environment"
echo "  3. Solutions -> Import"
echo "  4. Upload: $OUTPUT_FILE"
echo "  5. Map connections when prompted"
