#!/usr/bin/env python3
"""
Imagen - Google Gemini Image Generation Script

Cross-platform image generation using Google Gemini API.
Works on Windows, macOS, and Linux.

Usage:
    python generate_image.py "prompt" [output_path]

Environment variables:
    GEMINI_API_KEY (required) - Your Google Gemini API key
    IMAGE_SIZE (optional) - Image size: "512", "1K" (default), or "2K"
    GEMINI_MODEL (optional) - Model ID (default: gemini-3-pro-image-preview)
"""

import argparse
import base64
import json
import os
import sys
import urllib.request
import urllib.error
from pathlib import Path


# Configuration
DEFAULT_MODEL_ID = "gemini-3-pro-image-preview"
API_BASE_URL = "https://generativelanguage.googleapis.com/v1beta/models"
DEFAULT_IMAGE_SIZE = "1K"
VALID_SIZES = {"512", "1K", "2K"}


def get_api_endpoint(model_id: str) -> str:
    """Build the API endpoint URL for the given model."""
    # Use streamGenerateContent for image generation models
    return f"{API_BASE_URL}/{model_id}:streamGenerateContent"


def get_api_key() -> str:
    """Get the Gemini API key from environment variable."""
    api_key = os.environ.get("GEMINI_API_KEY")
    if not api_key:
        print("Error: GEMINI_API_KEY environment variable not set", file=sys.stderr)
        print("\nTo set it:", file=sys.stderr)
        print("  Windows (PowerShell): $env:GEMINI_API_KEY = 'your-key'", file=sys.stderr)
        print("  Windows (CMD): set GEMINI_API_KEY=your-key", file=sys.stderr)
        print("  macOS/Linux: export GEMINI_API_KEY='your-key'", file=sys.stderr)
        print("\nGet a free key at: https://aistudio.google.com/", file=sys.stderr)
        sys.exit(1)
    return api_key


def validate_image_size(size: str) -> str:
    """Validate and return the image size."""
    if size not in VALID_SIZES:
        print(f"Warning: Invalid IMAGE_SIZE '{size}'. Using default '{DEFAULT_IMAGE_SIZE}'", file=sys.stderr)
        return DEFAULT_IMAGE_SIZE
    return size


def create_output_dir(output_path: Path) -> None:
    """Create output directory if it doesn't exist."""
    output_dir = output_path.parent
    if output_dir and not output_dir.exists():
        output_dir.mkdir(parents=True, exist_ok=True)


def build_request_body(prompt: str, image_size: str) -> bytes:
    """Build the JSON request body for the API."""
    request_data = {
        "contents": [
            {
                "role": "user",
                "parts": [
                    {"text": prompt}
                ]
            }
        ],
        "generationConfig": {
            "responseModalities": ["IMAGE", "TEXT"],
            "imageConfig": {
                "image_size": image_size
            }
        }
    }
    return json.dumps(request_data).encode("utf-8")


def make_api_request(api_key: str, model_id: str, request_body: bytes) -> dict:
    """Make the API request and return the response."""
    endpoint = get_api_endpoint(model_id)
    url = f"{endpoint}?key={api_key}"

    headers = {
        "Content-Type": "application/json"
    }

    req = urllib.request.Request(url, data=request_body, headers=headers, method="POST")

    try:
        with urllib.request.urlopen(req, timeout=120) as response:
            return json.loads(response.read().decode("utf-8"))
    except urllib.error.HTTPError as e:
        error_body = e.read().decode("utf-8") if e.fp else ""
        print(f"Error: API request failed with HTTP status {e.code}", file=sys.stderr)
        if error_body:
            try:
                error_json = json.loads(error_body)
                print(f"Response: {json.dumps(error_json, indent=2)}", file=sys.stderr)
            except json.JSONDecodeError:
                print(f"Response: {error_body}", file=sys.stderr)
        sys.exit(1)
    except urllib.error.URLError as e:
        print(f"Error: Failed to connect to API: {e.reason}", file=sys.stderr)
        sys.exit(1)


def extract_image_data(response: dict) -> str:
    """Extract base64 image data from the API response."""
    try:
        # Handle both streaming array and single object responses
        if isinstance(response, list):
            candidates = response[0].get("candidates", [])
        else:
            candidates = response.get("candidates", [])

        if not candidates:
            raise ValueError("No candidates in response")

        parts = candidates[0].get("content", {}).get("parts", [])

        for part in parts:
            if "inlineData" in part:
                return part["inlineData"].get("data", "")

        raise ValueError("No image data found in response parts")
    except (KeyError, IndexError, TypeError) as e:
        print(f"Error: Failed to parse response: {e}", file=sys.stderr)
        print(f"Response: {json.dumps(response, indent=2)}", file=sys.stderr)
        sys.exit(1)


def save_image(image_data: str, output_path: Path) -> None:
    """Decode and save the base64 image data."""
    try:
        image_bytes = base64.b64decode(image_data)
        output_path.write_bytes(image_bytes)
    except Exception as e:
        print(f"Error: Failed to save image: {e}", file=sys.stderr)
        sys.exit(1)


def get_file_size(path: Path) -> str:
    """Get human-readable file size."""
    size = path.stat().st_size
    for unit in ["B", "KB", "MB", "GB"]:
        if size < 1024:
            return f"{size:.1f} {unit}"
        size /= 1024
    return f"{size:.1f} TB"


def main():
    parser = argparse.ArgumentParser(
        description="Generate images using Google Gemini AI",
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
  python generate_image.py "A sunset over mountains"
  python generate_image.py "An app icon" ./icons/app.png
  python generate_image.py --size 2K "High-res landscape" ./wallpaper.png
  python generate_image.py --model gemini-3-pro-image-preview "A logo" ./logo.png

Environment Variables:
  GEMINI_API_KEY    Your Google Gemini API key (required)
  IMAGE_SIZE        Image size: 512, 1K (default), or 2K
  GEMINI_MODEL      Model ID for image generation
        """
    )
    parser.add_argument("prompt", help="Text description of the image to generate")
    parser.add_argument("output", nargs="?", default="./generated-image.png",
                        help="Output file path (default: ./generated-image.png)")
    parser.add_argument("--size", choices=["512", "1K", "2K"],
                        help="Image size (overrides IMAGE_SIZE env var)")
    parser.add_argument("--model", "-m",
                        help=f"Gemini model ID (default: {DEFAULT_MODEL_ID})")

    args = parser.parse_args()

    # Get configuration
    api_key = get_api_key()
    model_id = args.model or os.environ.get("GEMINI_MODEL", DEFAULT_MODEL_ID)
    image_size = args.size or os.environ.get("IMAGE_SIZE", DEFAULT_IMAGE_SIZE)
    image_size = validate_image_size(image_size)
    output_path = Path(args.output)

    # Create output directory
    create_output_dir(output_path)

    # Display info
    print(f"Generating image with prompt: \"{args.prompt}\"")
    print(f"Model: {model_id}")
    print(f"Image size: {image_size}")
    print(f"Output path: {output_path}")
    print()

    # Build and send request
    request_body = build_request_body(args.prompt, image_size)
    response = make_api_request(api_key, model_id, request_body)

    # Extract and save image
    image_data = extract_image_data(response)
    if not image_data:
        print("Error: No image data received from API", file=sys.stderr)
        sys.exit(1)

    save_image(image_data, output_path)

    # Verify and report success
    if output_path.exists() and output_path.stat().st_size > 0:
        file_size = get_file_size(output_path)
        print("Success! Image generated and saved.")
        print(f"File: {output_path}")
        print(f"Size: {file_size}")
    else:
        print(f"Error: Failed to save image to {output_path}", file=sys.stderr)
        sys.exit(1)


if __name__ == "__main__":
    main()
