---
name: start-task
description: |
  작업 시작 (Supabase DB 기반). Use when (1) "랜드 #123 작업 진행", (2) "#45 시작하자",
  (3) "이슈 작업 시작해줘". 이슈 상태 변경 + 브랜치 생성 + Draft PR.
tools: [Supabase, Bash, Read]
model: inherit
---

> **시스템 메시지**: `[SEMO] Skill: start-task 호출`

# start-task Skill

> 작업 시작 자동화 (Supabase DB 기반)

## 🔴 데이터 소스 변경 (v2.0.0)

| 버전 | 데이터 소스 | 방식 |
|------|------------|------|
| v1.x | GitHub Projects | GraphQL mutation |
| **v2.0** | **Supabase** | `issues.status` UPDATE |

---

## Purpose

이슈 작업을 시작하기 위한 전체 프로세스를 자동화합니다:

1. 이슈 정보 조회 (Supabase)
2. 상태 → "in_progress" 변경
3. Feature 브랜치 생성
4. Draft PR 생성
5. SDD 프로세스 안내

## Input Parsing

```text
"랜드 #123 작업 진행하자"
→ repo: cm-land, issue: 123

"오피스 #45 시작"
→ repo: cm-office, issue: 45

"#123 작업 시작"
→ repo: (현재 디렉토리에서 추론), issue: 123
```

### 레포 별칭

| 별칭 | 실제 레포 |
|------|----------|
| 랜드 | cm-land |
| 오피스 | cm-office |
| 코어 | core-backend |
| 스몰 | cm-small |

## Workflow

### Step 1: 이슈 정보 조회 (Supabase)

```sql
-- 이슈 정보 조회
SELECT
  i.number,
  i.title,
  i.body,
  i.type,
  i.status,
  i.labels,
  ap.name AS assignee_name
FROM issues i
LEFT JOIN agent_personas ap ON i.assignee_id = ap.id
WHERE i.number = 123
  AND i.office_id = '{office_uuid}';
```

```typescript
// Supabase 클라이언트
const { data: issue, error } = await supabase
  .from('issues')
  .select(`
    number, title, body, type, status, labels,
    assignee:agent_personas(name)
  `)
  .eq('number', 123)
  .eq('office_id', officeId)
  .single();
```

### Step 2: 상태 변경 (Supabase)

```sql
-- 상태를 in_progress로 변경
SELECT * FROM update_issue_status(
  123,                    -- issue_number
  '{office_uuid}'::uuid,  -- office_id
  'in_progress',          -- new_status
  '{actor_uuid}'::uuid    -- changed_by
);
```

```typescript
// Supabase 클라이언트
const { data, error } = await supabase
  .from('issues')
  .update({ status: 'in_progress' })
  .eq('number', 123)
  .eq('office_id', officeId)
  .select()
  .single();
```

### Step 3: Feature 브랜치 생성

```bash
# 이슈 제목에서 slug 생성
SLUG=$(echo "로그인 페이지 구현" | tr ' ' '-' | tr '[:upper:]' '[:lower:]')

# 브랜치 생성
git checkout -b feature/123-${SLUG}

# 원격에 푸시
git push -u origin feature/123-${SLUG}
```

### Step 4: Draft PR 생성

```bash
gh pr create \
  --repo semicolon-devteam/cm-land \
  --title "[Draft] #123 로그인 페이지 구현" \
  --body "$(cat <<'EOF'
## 관련 이슈
closes #123

## 변경 사항
- [ ] 작업 진행 중

## 체크리스트
- [ ] Spec 작성 완료
- [ ] 구현 완료
- [ ] 테스트 통과
- [ ] 리뷰 요청

---
Generated by SEMO
EOF
)" \
  --draft
```

### Step 5: 다음 단계 선택지 제시 (NON-NEGOTIABLE)

> **CRITICAL**: 작업 환경 세팅 완료 후 **반드시** 다음 단계 선택지를 제시합니다.

```markdown
## 🚀 작업 환경 준비 완료!

다음 단계를 선택해주세요:

| 옵션 | 적합한 경우 | 키워드 |
|------|------------|--------|
| **A. SDD 전체** (권장) | 새 기능, 복잡한 로직, AC 필요 | "스펙부터 시작" |
| **B. 바로 구현** | 명확한 요구사항, 간단한 기능 | "구현 시작" |
| **C. Fast-track** | 오타/스타일 수정, 3파일 이하 | "패스트트랙" |

**선택하세요**: A, B, 또는 C
```

### 선택지별 라우팅

| 선택 | 라우팅 대상 | 동작 |
|------|------------|------|
| A | `skill:spec` | SDD Phase 1-3 시작 |
| B | `implementation-master` | ADD Phase 4 바로 시작 |
| C | `skill:fast-track` | 간소화된 수정 프로세스 |

## Output Format

```markdown
[SEMO] Skill: start-task 호출

## 작업 시작: #123

### 이슈 정보

| 항목 | 내용 |
|------|------|
| **제목** | 로그인 페이지 구현 |
| **유형** | feature |
| **담당자** | @reus |

### 완료된 작업

✅ 상태 변경: todo → in_progress
✅ 브랜치 생성: `feature/123-로그인-페이지-구현`
✅ Draft PR 생성: #150

### 🚀 다음 단계 선택

| 옵션 | 적합한 경우 | 키워드 |
|------|------------|--------|
| **A. SDD 전체** (권장) | 새 기능, 복잡한 로직 | "스펙부터 시작" |
| **B. 바로 구현** | 명확한 요구사항 | "구현 시작" |
| **C. Fast-track** | 경미한 수정 | "패스트트랙" |

💡 **라벨 기반 권장**: `feature` → **A. SDD 전체** 권장

**선택하세요**: A, B, 또는 C

[SEMO] Skill: start-task 완료
```

## Error Handling

### 이슈를 찾을 수 없는 경우

```markdown
❌ #123 이슈를 찾을 수 없습니다.

확인사항:
- 이슈 번호가 올바른가요?
- Office ID가 올바른가요?
- 이슈가 Open 상태인가요?
```

### 이미 작업중인 경우

```markdown
⚠️ #123은 이미 "in_progress" 상태입니다.

현재 상태:
- 브랜치: feature/123-로그인-페이지-구현 (존재함)
- PR: #150 (Draft)

계속 진행하시겠습니까?
```

## GitHub Projects Fallback

Supabase 연결이 불가능한 경우:

```bash
# Fallback: GitHub Projects GraphQL
gh api graphql -f query='
  mutation {
    updateProjectV2ItemFieldValue(input: {...}) {
      projectV2Item { id }
    }
  }
'
```

## References

- [issues 테이블 마이그레이션](../../../semo-repository/supabase/migrations/20260113003_issues_discussions.sql)
- [git-workflow Skill](../git-workflow/SKILL.md)
- [project-status Skill](../project-status/SKILL.md)
