---
name: supabase-typegen
description: |
  Supabase CLI를 활용한 데이터베이스 타입 생성 및 동기화.
  PROACTIVELY use when (1) DB 스키마 변경 후 타입 업데이트, (2) Repository 구현 전 타입 확인,
  (3) 새 테이블/컬럼 추가 시, (4) 타입 불일치 오류 발생 시.
  Cloud 환경 전용 - On-Premise는 수동 타입 정의 필요.
tools: [Bash, Read, Write, Edit]
---

> **시스템 메시지**: 이 Skill이 호출되면 `[SEMO] Skill: supabase-typegen 호출 - {작업 유형}` 시스템 메시지를 첫 줄에 출력하세요.

# Supabase Typegen Skill

**Purpose**: Supabase CLI를 활용하여 실제 DB 스키마와 TypeScript 타입을 동기화

## When to Use

| 상황 | 트리거 |
|------|--------|
| DB 스키마 변경 후 | "스키마 수정했어", "컬럼 추가했어" |
| Repository 구현 전 | "Repository 만들어줘", "DB 연동해줘" |
| 타입 오류 발생 시 | "타입이 안맞아", "Property does not exist" |
| 새 프로젝트 시작 | "Supabase 연동해줘", "DB 설정해줘" |

## Pre-Check: 환경 확인

```bash
# 1. Supabase CLI 설치 확인
npx supabase --version

# 2. 로그인 상태 확인
npx supabase projects list

# 3. 프로젝트 연결 확인
cat supabase/.temp/project-ref 2>/dev/null || echo "프로젝트 미연결"
```

## 환경별 분기

### Cloud 환경 (기본)

대부분의 서비스는 Supabase Cloud를 사용합니다:

```bash
# 1. 프로젝트 연결 (최초 1회)
npx supabase link --project-ref <project-id>

# 2. 타입 생성
npx supabase gen types typescript --linked > src/lib/supabase/database.types.ts

# 3. 변경사항 확인
git diff src/lib/supabase/database.types.ts
```

### On-Premise 환경

On-Premise Supabase는 CLI 직접 연결이 불가합니다:

```markdown
**수동 타입 정의 필요**

1. SSH로 스키마 조회:
   ```bash
   ssh ubuntu@{host} "docker exec {container} psql -U postgres -d postgres -c '\d {table}'"
   ```

2. 수동으로 database.types.ts 업데이트

3. 또는 로컬 Supabase로 스키마 복제 후 타입 생성
```

## Workflow

### Step 1: 현재 타입 상태 확인

```bash
# database.types.ts 존재 여부
ls -la src/lib/supabase/database.types.ts

# 마지막 수정 시간
stat -f "%Sm" src/lib/supabase/database.types.ts 2>/dev/null || \
stat -c "%y" src/lib/supabase/database.types.ts 2>/dev/null
```

### Step 2: 타입 생성

```bash
# Cloud 환경
npx supabase gen types typescript --linked > src/lib/supabase/database.types.ts
```

### Step 3: 타입 활용 확인

```typescript
// src/lib/supabase/database.types.ts 생성됨
export type Database = {
  public: {
    Tables: {
      posts: {
        Row: { id: string; title: string; ... }
        Insert: { id?: string; title: string; ... }
        Update: { id?: string; title?: string; ... }
      }
      // ...
    }
    Functions: {
      posts_read: { Args: { ... }; Returns: ... }
      // ...
    }
  }
}
```

### Step 4: Repository에서 활용

```typescript
import { Database } from '@/lib/supabase/database.types';

type Post = Database['public']['Tables']['posts']['Row'];
type PostInsert = Database['public']['Tables']['posts']['Insert'];

// RPC 함수 타입
type PostsReadArgs = Database['public']['Functions']['posts_read']['Args'];
type PostsReadReturn = Database['public']['Functions']['posts_read']['Returns'];
```

## Critical Rules

### 1. DB 작업 전 항상 타입 동기화

```markdown
[필수 순서]
1. DB 스키마 변경 (migration 또는 직접)
2. `npx supabase gen types typescript --linked` 실행
3. database.types.ts 변경 확인
4. Repository 코드 작성
```

### 2. 타입 파일 위치 고정

```
src/lib/supabase/
├── database.types.ts  # 자동 생성 (수정 금지)
├── client.ts          # 브라우저 클라이언트
└── server.ts          # 서버 클라이언트
```

### 3. 자동 생성 파일 수정 금지

```typescript
// database.types.ts 상단 주석
/**
 * This file was auto-generated by Supabase CLI.
 * DO NOT MODIFY - regenerate with: npx supabase gen types typescript --linked
 */
```

### 4. Git Commit 포함

타입 파일 변경은 반드시 커밋에 포함:

```bash
git add src/lib/supabase/database.types.ts
git commit -m "chore: sync supabase types"
```

## Troubleshooting

### 프로젝트 연결 안됨

```bash
# 해결: 프로젝트 ID 확인 후 link
npx supabase projects list
npx supabase link --project-ref <project-id>
```

### 인증 오류

```bash
# 해결: 재로그인
npx supabase login
```

### 타입이 반영 안됨

```bash
# 해결: 캐시 클리어 후 재생성
rm -rf supabase/.temp
npx supabase link --project-ref <project-id>
npx supabase gen types typescript --linked > src/lib/supabase/database.types.ts
```

### On-Premise에서 CLI 사용 시도

```markdown
**에러**: Supabase CLI는 Cloud 환경만 지원

**해결 방안**:
1. 로컬 Supabase Docker로 스키마 복제
2. 복제된 환경에서 타입 생성
3. 또는 수동 타입 정의
```

## Output Format

```markdown
## [SEMO] Supabase 타입 동기화 완료

**환경**: Cloud / On-Premise
**프로젝트**: {project-ref}

**생성된 타입**:
- Tables: {table_count}개
- Functions: {function_count}개
- Enums: {enum_count}개

**변경 사항**:
- 추가된 테이블: {added_tables}
- 수정된 컬럼: {modified_columns}

**다음 단계**:
```bash
# 타입 확인
cat src/lib/supabase/database.types.ts | head -50
```
```

## Related

- [database-master Agent](../../agents/database-master.md) - DB 작업 총괄
- [fetch-supabase-example Skill](../fetch-supabase-example/SKILL.md) - 구현 패턴 참조
- [implement Skill](../implement/SKILL.md) - 구현 시 타입 활용
