---
name: conductor-revert
description: Git-aware undo by logical work unit (track, phase, or task)
metadata:
  argument-hint: "[track-id | track-id:phase | track-id:task]"
---

# Revert Track

Revert changes by logical work unit with full git awareness. Supports reverting entire tracks, specific phases, or individual tasks.

## Use this skill when

- Working on revert track tasks or workflows
- Needing guidance, best practices, or checklists for revert track

## Do not use this skill when

- The task is unrelated to revert track
- You need a different domain or tool outside this scope

## Instructions

- Clarify goals, constraints, and required inputs.
- Apply relevant best practices and validate outcomes.
- Provide actionable steps and verification.
- If detailed examples are required, open `resources/implementation-playbook.md`.

## Pre-flight Checks

1. Verify Conductor is initialized:
   - Check `conductor/tracks.md` exists
   - If missing: Display error and suggest running `/conductor:setup` first

2. Verify git repository:
   - Run `git status` to confirm git repo
   - Check for uncommitted changes
   - If uncommitted changes exist:

     ```
     WARNING: Uncommitted changes detected

     Files with changes:
     {list of files}

     Options:
     1. Stash changes and continue
     2. Commit changes first
     3. Cancel revert
     ```

3. Verify git is clean enough to revert:
   - No merge in progress
   - No rebase in progress
   - If issues found: Halt and explain resolution steps

## Target Selection

### If argument provided:

Parse the argument format:

**Full track:** `{trackId}`

- Example: `auth_20250115`
- Reverts all commits for the entire track

**Specific phase:** `{trackId}:phase{N}`

- Example: `auth_20250115:phase2`
- Reverts commits for phase N and all subsequent phases

**Specific task:** `{trackId}:task{X.Y}`

- Example: `auth_20250115:task2.3`
- Reverts commits for task X.Y only

### If no argument:

Display guided selection menu:

```
What would you like to revert?

Currently In Progress:
1. [~] Task 2.3 in dashboard_20250112 (most recent)

Recently Completed:
2. [x] Task 2.2 in dashboard_20250112 (1 hour ago)
3. [x] Phase 1 in dashboard_20250112 (3 hours ago)
4. [x] Full track: auth_20250115 (yesterday)

Options:
5. Enter specific reference (track:phase or track:task)
6. Cancel

Select option:
```

## Commit Discovery

### For Task Revert

1. Search git log for task-specific commits:

   ```bash
   git log --oneline --grep="{trackId}" --grep="Task {X.Y}" --all-match
   ```

2. Also find the plan.md update commit:

   ```bash
   git log --oneline --grep="mark task {X.Y} complete" --grep="{trackId}" --all-match
   ```

3. Collect all matching commit SHAs

### For Phase Revert

1. Determine task range for the phase by reading plan.md
2. Search for all task commits in that phase:

   ```bash
   git log --oneline --grep="{trackId}" | grep -E "Task {N}\.[0-9]"
   ```

3. Find phase verification commit if exists
4. Find all plan.md update commits for phase tasks
5. Collect all matching commit SHAs in chronological order

### For Full Track Revert

1. Find ALL commits mentioning the track:

   ```bash
   git log --oneline --grep="{trackId}"
   ```

2. Find track creation commits:

   ```bash
   git log --oneline -- "conductor/tracks/{trackId}/"
   ```

3. Collect all matching commit SHAs in chronological order

## Execution Plan Display

Before any revert operations, display full plan:

```
================================================================================
                           REVERT EXECUTION PLAN
================================================================================

Target: {description of what's being reverted}

Commits to revert (in reverse chronological order):
  1. abc1234 - feat: add chart rendering (dashboard_20250112)
  2. def5678 - chore: mark task 2.3 complete (dashboard_20250112)
  3. ghi9012 - feat: add data hooks (dashboard_20250112)
  4. jkl3456 - chore: mark task 2.2 complete (dashboard_20250112)

Files that will be affected:
  - src/components/Dashboard.tsx (modified)
  - src/hooks/useData.ts (will be deleted - was created in these commits)
  - conductor/tracks/dashboard_20250112/plan.md (modified)

Plan updates:
  - Task 2.2: [x] -> [ ]
  - Task 2.3: [~] -> [ ]

================================================================================
                              !! WARNING !!
================================================================================

This operation will:
- Create {N} revert commits
- Modify {M} files
- Reset {P} tasks to pending status

This CANNOT be easily undone without manual intervention.

================================================================================

Type 'YES' to proceed, or anything else to cancel:
```

**CRITICAL: Require explicit 'YES' confirmation. Do not proceed on 'y', 'yes', or enter.**

## Revert Execution

Execute reverts in reverse chronological order (newest first):

```
Executing revert plan...

[1/4] Reverting abc1234...
      git revert --no-edit abc1234
      ✓ Success

[2/4] Reverting def5678...
      git revert --no-edit def5678
      ✓ Success

[3/4] Reverting ghi9012...
      git revert --no-edit ghi9012
      ✓ Success

[4/4] Reverting jkl3456...
      git revert --no-edit jkl3456
      ✓ Success
```

### On Merge Conflict

If any revert produces a merge conflict:

```
================================================================================
                           MERGE CONFLICT DETECTED
================================================================================

Conflict occurred while reverting: {sha} - {message}

Conflicted files:
  - src/components/Dashboard.tsx

Options:
1. Show conflict details
2. Abort revert sequence (keeps completed reverts)
3. Open manual resolution guide

IMPORTANT: Reverts 1-{N} have been completed. You may need to manually
resolve this conflict before continuing or fully undo the revert sequence.

Select option:
```

**HALT immediately on any conflict. Do not attempt automatic resolution.**

## Plan.md Updates

After successful git reverts, update plan.md:

1. Read current plan.md
2. For each reverted task, change marker:
   - `[x]` -> `[ ]`
   - `[~]` -> `[ ]`
3. Write updated plan.md
4. Update metadata.json:
   - Decrement `tasks.completed`
   - Update `status` if needed
   - Update `updated` timestamp

**Do NOT commit plan.md changes** - they are part of the revert operation

## Track Status Updates

### If reverting entire track:

- In tracks.md: Change `[x]` or `[~]` to `[ ]`
- Consider offering to delete the track directory entirely

### If reverting to incomplete state:

- In tracks.md: Ensure marked as `[~]` if partially complete, `[ ]` if fully reverted

## Verification

After revert completion:

```
================================================================================
                           REVERT COMPLETE
================================================================================

Summary:
  - Reverted {N} commits
  - Reset {P} tasks to pending
  - {M} files affected

Git log now shows:
  {recent commit history}

Plan.md status:
  - Task 2.2: [ ] Pending
  - Task 2.3: [ ] Pending

================================================================================

Verify the revert was successful:
  1. Run tests: {test command}
  2. Check application: {relevant check}

If issues are found, you may need to:
  - Fix conflicts manually
  - Re-implement the reverted tasks
  - Use 'git revert HEAD~{N}..HEAD' to undo the reverts

================================================================================
```

## Safety Rules

1. **NEVER use `git reset --hard`** - Only use `git revert`
2. **NEVER use `git push --force`** - Only safe push operations
3. **NEVER auto-resolve conflicts** - Always halt for human intervention
4. **ALWAYS show full plan** - User must see exactly what will happen
5. **REQUIRE explicit 'YES'** - Not 'y', not enter, only 'YES'
6. **HALT on ANY error** - Do not attempt to continue past failures
7. **PRESERVE history** - Revert commits are preferred over history rewriting

## Edge Cases

### Track Never Committed

```
No commits found for track: {trackId}

The track exists but has no associated commits. This may mean:
- Implementation never started
- Commits used different format

Options:
1. Delete track directory only
2. Cancel
```

### Commits Already Reverted

```
Some commits appear to already be reverted:
  - abc1234 was reverted by xyz9876

Options:
1. Skip already-reverted commits
2. Cancel and investigate
```

### Remote Already Pushed

```
WARNING: Some commits have been pushed to remote

Commits on remote:
  - abc1234 (origin/main)
  - def5678 (origin/main)

Reverting will create new revert commits that you'll need to push.
This is the safe approach (no force push required).

Continue with revert? (YES/no):
```

## Undo the Revert

If user needs to undo the revert itself:

```
To undo this revert operation:

  git revert HEAD~{N}..HEAD

This will create new commits that restore the reverted changes.

Alternatively, if not yet pushed:
  git reset --soft HEAD~{N}
  git checkout -- .

(Use with caution - this discards the revert commits)
```
