# Dify Tool Plugin Development Workflow

Complete 5-phase workflow for developing Dify tool plugins. Follow phases in sequence - do not skip steps.

---

## Phase 1: Setup & Planning

### Objectives
- Gather complete requirements from user
- Determine tool scope and functionality
- Identify dependencies and permissions
- Verify development environment

### Prerequisites
- Load `references/doc-map.md` first
- Understand documentation structure

### Steps

#### 1. Gather Requirements

**Ask the user these questions:**

1. **Functionality**: "What should this tool do? Please describe the main functionality."

2. **API Access**: "Does this tool need to call external APIs? If yes, which APIs or services?"

3. **Authentication**: "Does it require OAuth authentication? (e.g., GitHub, Google, Slack, etc.)"

4. **Inputs**: "What inputs should users provide? What parameters does the tool need?"

5. **Outputs**: "What should the tool return? What format should the output be in?"

**Record the following:**
- Tool name (lowercase, hyphens only, e.g., `weather-forecast`)
- Author name
- Brief description (1-2 sentences)
- Required permissions (network, storage, etc.)
- OAuth requirement (yes/no + provider)
- Python dependencies (e.g., requests, pandas)
- API endpoints to integrate

#### 2. Verify CLI Installation

**Run:**
```bash
dify version

**If CLI is not installed:**

**Option 1: Automated (Recommended)**

```bash
python scripts/install_cli.py
```

This automatically detects your platform and installs the latest Dify CLI.

**Option 2: Manual**

```

**Expected output:**
```
v0.4.0
```

**If CLI is not installed or returns error:**
1. LOAD: `https://docs.dify.ai/plugin-dev-en/0221-initialize-development-tools.md`
2. Follow installation instructions for user's OS
3. Verify with `dify version`

### Success Criteria
- [ ] All requirements clearly defined
- [ ] Tool name chosen (valid format)
- [ ] Permissions identified
- [ ] OAuth requirement determined
- [ ] Dify CLI installed and verified

---

## Phase 2: Initialize Project

### Objectives
- Create plugin project structure
- Generate required files
- Set up development environment

### Prerequisites
⚠️ **MANDATORY**: Load official documentation BEFORE proceeding

**Fetch:**
```bash
curl -s https://docs.dify.ai/plugin-dev-en/0222-tool-plugin.md
```

**Read sections:**
- "Prerequisites"
- "Creating a New Project"
- "Choosing Plugin Type and Template"

### Steps

#### 1. Prepare init Command

Based on Phase 1 requirements, construct the `dify plugin init` command:

**Basic template:**
```bash
dify plugin init \
  --name "your-tool-name" \
  --author "author-name" \
  --description "Brief tool description" \
  --category tool \
  --language python \
  --min-dify-version "0.6.0" \
  --quick
```

**Add permission flags as needed:**
```bash
--allow-network          # If calling external APIs
--allow-storage          # If needing persistent storage
--storage-size 1048576   # Storage limit in bytes (if --allow-storage)
```

**Example with permissions:**
```bash
dify plugin init \
  --name "github-issue-tracker" \
  --author "your-author-name" \
  --description "Track and manage GitHub issues via Dify" \
  --category tool \
  --language python \
  --min-dify-version "0.6.0" \
  --allow-network \
  --quick
```

#### 2. Execute Initialization

Run the constructed command.

**Expected output:**
```
INFO: plugin github-issue-tracker created successfully
```

#### 3. Verify Project Structure

```bash
ls -la your-tool-name/
```

**Required files:**
```
your-tool-name/
├── manifest.yaml          ✓ Plugin manifest
├── provider/
│   ├── your-tool-name.yaml   ✓ Provider config
│   └── your-tool-name.py     ✓ Provider implementation
├── tools/
│   ├── your-tool-name.yaml   ✓ Tool config
│   └── your-tool-name.py     ✓ Tool implementation
├── main.py                   ✓ Entry point
├── requirements.txt          ✓ Dependencies
├── .env.example              ✓ Environment template
├── .gitignore
├── GUIDE.md
├── PRIVACY.md
└── README.md
```

**If any files are missing:**
- Re-run init command
- Check for error messages
- Verify CLI version

#### 4. Set Up Environment

```bash
cd your-tool-name

# Copy environment template
cp .env.example .env

# Edit .env if credentials needed
# nano .env
```



**Step 4: Note the dify_plugin Version**

Check what version the CLI generated:

```bash
cat your-tool-name/requirements.txt
```

The CLI creates requirements.txt with the latest compatible dify_plugin version.

**DO NOT change this version.** Note it for reference:
```
dify_plugin>=X.X.X    # Generated by CLI - use this version
```

Only add additional dependencies your tool needs (requests, etc.)

### Success Criteria
- [ ] Project directory created
- [ ] All required files present
- [ ] No errors during initialization
- [ ] Environment template copied

---

## Phase 3: Implement Tool

### Objectives
- Configure YAML files with correct structure
- Implement Tool class with exact patterns
- Add proper error handling and logging
- Update manifest and provider configs

### Prerequisites
⚠️ **MANDATORY**: Re-load tool-plugin.md documentation

**Fetch again:**
```bash
curl -s https://docs.dify.ai/plugin-dev-en/0222-tool-plugin.md
```

**Focus on section:** "Developing the Tool Plugin"

**If OAuth needed:**
⚠️ **MANDATORY**: Load OAuth documentation

```bash
curl -s https://docs.dify.ai/plugin-dev-en/0222-tool-oauth.md
```

### Steps

#### 1. Edit tools/*.yaml

**Location:** `tools/your-tool-name.yaml`

**Follow this exact structure:**

```yaml
identity:
  name: "your-tool-name"       # Match filename
  author: "author-name"
  label:
    en_US: "Tool Display Name"
    zh_Hans: "工具显示名称"
    pt_BR: "Nome da Ferramenta"
    ja_JP: "ツール名"

description:
  human:
    en_US: "Description for human users"
    zh_Hans: "人类用户描述"
    pt_BR: "Descrição para usuários"
    ja_JP: "ユーザー向けの説明"
  llm: "Detailed description for AI to understand when to use this tool"

parameters:
  - name: query
    type: string                # string | number | boolean | file
    required: true              # true | false
    label:
      en_US: "Query"
      zh_Hans: "查询"
    human_description:
      en_US: "Search query or input text"
      zh_Hans: "搜索查询或输入文本"
    llm_description: "The search query or input text to process"
    form: llm                   # llm | form

  # Add more parameters as needed

extra:
  python:
    source: tools/your-tool-name.py
```

**Parameter types:**
- `string` - Text input
- `number` - Numeric value
- `boolean` - True/false
- `file` - File upload

**Form types:**
- `llm` - AI extracts from conversation (recommended)
- `form` - User configures in UI

**Validation:**
- Proper YAML indentation (2 spaces)
- All i18n languages present
- `source` path matches Python file

#### 2. Edit tools/*.py

**Location:** `tools/your-tool-name.py`

**Template:**

```python
from collections.abc import Generator
from typing import Any

from dify_plugin import Tool
from dify_plugin.entities.tool import ToolInvokeMessage

class YourToolNameTool(Tool):
    """
    [Brief description of what this tool does]
    """

    def _invoke(
        self, 
        tool_parameters: dict[str, Any]
    ) -> Generator[ToolInvokeMessage]:
        """
        Invoke the tool with given parameters.

        Args:
            tool_parameters: Dictionary containing tool parameters

        Yields:
            ToolInvokeMessage objects to return to user
        """
        try:
            # 1. Extract parameters using .get()
            required_param = tool_parameters.get("param_name", "")
            optional_param = tool_parameters.get("optional_param")

            # 2. Validate required parameters
            if not required_param:
                yield self.create_text_message(
                    "Error: Required parameter 'param_name' is missing."
                )
                return

            # 3. Implement business logic
            result = self._process_request(required_param, optional_param)

            # 4. Return results with appropriate message types
            yield self.create_text_message(f"Result: {result}")

            # Optional: Return structured data
            yield self.create_json_message({
                "status": "success",
                "data": result
            })

            # Optional: Set workflow variable
            yield self.create_variable_message("result", result)

        except Exception as e:
            # Always include error handling
            yield self.create_text_message(f"Error: {str(e)}")

    def _process_request(
        self, 
        required_param: str, 
        optional_param: str = None
    ) -> Any:
        """
        Helper method for processing logic.

        Args:
            required_param: Required input
            optional_param: Optional input

        Returns:
            Processed result
        """
        # Your implementation here
        if optional_param:
            return f"Processed: {required_param} with {optional_param}"
        return f"Processed: {required_param}"
```

**Critical rules:**
- ⚠️ ONE Tool class per file
- ⚠️ Class name format: `[Name]Tool`
- ⚠️ Method signature must be exact: `_invoke(self, tool_parameters: dict[str, Any]) -> Generator[ToolInvokeMessage]`
- ⚠️ Use `.get()` for all parameter access
- ⚠️ Use `yield`, not `return`

**Message types:**
```python
# Text message (always shown to user)
self.create_text_message(str)

# JSON message (structured data)
self.create_json_message(dict)

# Link message (clickable URL)
self.create_link_message(str)

# Variable message (for workflows)
self.create_variable_message(name, value)

# Blob message (files/binary data)
self.create_blob_message(data, name)
```

#### 3. Edit provider/*.yaml

**Location:** `provider/your-tool-name.yaml`

**Add your tool to the tools list:**

```yaml
identity:
  author: "author-name"
  name: "your-tool-name"
  label:
    en_US: "Plugin Name"
    zh_Hans: "插件名称"
  description:
    en_US: "Plugin description"
    zh_Hans: "插件描述"
  icon: icon.svg

# Only include if credentials/API keys needed
credentials_for_provider:
  api_key:
    type: secret-input
    required: true
    label:
      en_US: "API Key"
      zh_Hans: "API 密钥"
    placeholder:
      en_US: "Enter your API key"
      zh_Hans: "输入您的 API 密钥"
    help:
      en_US: "Get your key at https://example.com/keys"
      zh_Hans: "在 https://example.com/keys 获取密钥"
    url: https://example.com/keys

tools:
  - tools/your-tool-name.yaml

extra:
  python:
    source: provider/your-tool-name.py
```

#### 4. Edit provider/*.py (if credentials needed)

**Location:** `provider/your-tool-name.py`

**Template:**

```python
from typing import Any
from dify_plugin.entities.tool import ToolProviderCredentials

class YourToolNameProvider:
    def _validate_credentials(
        self, 
        credentials: dict[str, Any]
    ) -> ToolProviderCredentials:
        """
        Validate provider credentials.

        Args:
            credentials: Dict with credential values

        Returns:
            ToolProviderCredentials if valid

        Raises:
            ToolProviderCredentialValidationError if invalid
        """
        try:
            api_key = credentials.get("api_key")

            if not api_key:
                raise Exception("API key is required")

            # Validate by test API call
            # import requests
            # response = requests.get(
            #     "https://api.example.com/validate",
            #     headers={"Authorization": f"Bearer {api_key}"}
            # )
            # if response.status_code != 200:
            #     raise Exception("Invalid API key")

            return ToolProviderCredentials(credentials=credentials)

        except Exception as e:
            from dify_plugin.errors.tool import ToolProviderCredentialValidationError
            raise ToolProviderCredentialValidationError(str(e))
```

#### 5. Update manifest.yaml

**Location:** `manifest.yaml`

**Complete all i18n fields:**

```yaml
version: 0.0.1
type: plugin
author: author-name
name: your-tool-name

label:
  en_US: "Plugin Display Name"
  zh_Hans: "插件显示名称"
  pt_BR: "Nome do Plugin"
  ja_JP: "プラグイン名"

description:
  en_US: "What this plugin does"
  zh_Hans: "此插件的功能"
  pt_BR: "O que este plugin faz"
  ja_JP: "このプラグインの機能"

icon: icon.svg

resource:
  memory: 268435456  # 256MB
  permission:
    network:
      enabled: true   # Set based on requirements
    storage:
      enabled: false
      size: 0

plugins:
  tools:
    - provider/your-tool-name.yaml

meta:
  version: 0.0.1
  arch:
    - amd64
    - arm64
  runner:
    language: python
    version: "3.12"
    entrypoint: main
  minimum_dify_version: 0.6.0

created_at: 2025-11-04T00:00:00.000000+00:00
privacy: PRIVACY.md
```

#### 6. Update requirements.txt

**Add dependencies:**

```
dify_plugin>=0.0.1
requests>=2.31.0
# Add other packages as needed
```

### Success Criteria
- [ ] All YAML files properly formatted
- [ ] Tool class implements exact pattern
- [ ] One Tool class per Python file
- [ ] Error handling included
- [ ] manifest.yaml completed
- [ ] Dependencies added to requirements.txt

---

## Phase 4: Test & Debug

### Objectives
- Run plugin locally
- Test all functionality
- Debug any errors
- Add logging for troubleshooting

### Prerequisites
**Load debugging documentation:**

```bash
curl -s https://docs.dify.ai/plugin-dev-en/0222-debugging-logs.md
```

### Steps

#### 1. Add Logging

**In your tool file:**

```python
import logging
from dify_plugin.handlers import DifyPluginLogHandler

logger = logging.getLogger(__name__)
logger.addHandler(DifyPluginLogHandler())

class YourTool(Tool):
    def _invoke(self, tool_parameters: dict[str, Any]) -> Generator[ToolInvokeMessage]:
        logger.info(f"Tool invoked with parameters: {tool_parameters}")

        # Your code
        logger.debug(f"Processing step 1...")
        logger.debug(f"Processing step 2...")

        logger.info("Completed successfully")
```

#### 2. Set Up Environment

If credentials needed:

```bash
cd your-tool-directory
cp .env.example .env
# Edit .env and add real credentials for testing
```

#### 3. Run Local Test

```bash
cd your-tool-directory
python -m main
```

**Expected output:**
- Plugin starts without errors
- Logs appear in console
- No import errors
- No syntax errors

#### 4. Debug Common Errors

**If errors occur, load troubleshooting:**

```bash
# See references/troubleshooting.md for solutions
```

**Common issues:**

| Error | Solution |
|-------|----------|
| Multiple Tool subclasses | Keep only one Tool class per .py file |
| ImportError | Check import spelling matches definitions exactly |
| KeyError for parameter | Use `.get()` method instead of direct access |
| Credential validation error | Check API key format, test endpoint accessibility |

#### 5. Test Tool Functionality

Create test scenarios:
- Test with valid inputs
- Test with missing required parameters
- Test with invalid inputs
- Test error handling
- Test API integration (if applicable)

### Success Criteria
- [ ] Plugin runs without errors
- [ ] Logging works correctly
- [ ] Tool processes inputs correctly
- [ ] Error handling works
- [ ] All test scenarios pass

---



---

## Phase 4.5: Remote Testing (MANDATORY Before Packaging)

### Objectives
- Test with real Dify instance
- Verify with remote host and debug key
- Validate all functionality
- Catch integration issues

### Prerequisites
**Must complete Phase 4 (local testing) first**

### Steps

**Step 1: Get Dify Connection Info**

Ask user:
- Remote host URL (e.g., https://dify.example.com or http://localhost)
- Port number (default: 5001)
- Debug key (from Dify developer settings)

**Note:** API credentials for the tool itself should already be in .env from Phase 4.

**Step 2: Configure .env**

```bash
cd your-tool-directory

# Add to existing .env or create new
cat >> .env << EOF
DIFY_REMOTE_HOST=https://your-dify.com:5001
DIFY_DEBUG_KEY=your-debug-key
EOF

# Tool API credentials already in .env from Phase 4
```

**Step 3: Start Plugin**

```bash
python -m main
```

Watch for connection confirmation in logs.

**Step 4: Test from Dify UI**

- Open Dify instance
- Find plugin in Tools
- Invoke with test inputs
- Verify results

**Step 5: Run All Test Scenarios**

Test:
- Valid inputs → Success
- Missing params → Error handling
- Invalid inputs → Validation
- API calls → Correct data
- Errors → Graceful handling

**Step 6: Verify Logs**

Check console shows:
- Tool invocations
- Parameter values
- API calls
- Results
- Any errors

**Step 7: User Confirmation**

Ask: "Plugin tested on your Dify instance. Does everything work as expected?"

Wait for confirmation before packaging.

### Success Criteria
- [ ] Connected to remote host
- [ ] Debug key working
- [ ] All tests passed
- [ ] User confirmed working
- [ ] Ready to package

**Only proceed to Phase 5 if ALL criteria met.**

## Phase 5: Package & Publish

### Objectives
- Create distributable package
- Verify package integrity
- Optionally publish to marketplace

### Prerequisites
**Load packaging documentation:**

```bash
curl -s https://docs.dify.ai/plugin-dev-en/0222-tool-plugin.md
# Read "Packaging the Plugin" section
```

### Steps

#### 1. Final Validation

Before packaging, verify:
- [ ] All YAML files valid
- [ ] All Python files syntax-correct
- [ ] Requirements.txt complete
- [ ] manifest.yaml fully filled out
- [ ] Tests passing
- [ ] No TODO comments in code

#### 2. Package Plugin

```bash
dify plugin package ./your-tool-directory
```

**Expected output:**
```
INFO: plugin packaged successfully, output path: your-tool-name.difypkg
```

#### 3. Verify Package

```bash
ls -lh *.difypkg
```

**Check:**
- File exists
- File size reasonable (typically 10-100 KB)
- No error messages during packaging

#### 4. Test Package (Optional)

```bash
# Install package locally to test
dify plugin install ./your-tool-name.difypkg
```

#### 5. Publish (Optional)

If user wants to publish to Dify marketplace:

**Steps:**
1. Create account at Dify plugin marketplace
2. Upload `.difypkg` file
3. Fill in marketplace metadata:
   - Screenshots
   - Detailed description
   - Usage examples
   - Support information
4. Submit for review
5. Wait for approval

**Guide user through:**
- Marketplace URL
- Account creation
- Upload process
- Metadata requirements
- Review timeline

### Success Criteria
- [ ] Package file created (.difypkg)
- [ ] No packaging errors
- [ ] Package size reasonable
- [ ] Optional: Successfully published

---

## Workflow Checklist

**Complete workflow checklist:**

- [ ] **Phase 1: Setup & Planning**
  - [ ] Requirements gathered
  - [ ] Tool scope defined
  - [ ] CLI verified

- [ ] **Phase 2: Initialize**
  - [ ] Documentation loaded
  - [ ] Project initialized
  - [ ] Structure verified

- [ ] **Phase 3: Implement**
  - [ ] YAML files configured
  - [ ] Tool class implemented
  - [ ] manifest.yaml completed
  - [ ] Dependencies added

- [ ] **Phase 4: Test**
  - [ ] Logging added
  - [ ] Local testing passed
  - [ ] Errors debugged
  - [ ] All scenarios tested

- [ ] **Phase 5: Package**
  - [ ] Final validation done
  - [ ] Package created
  - [ ] Package verified
  - [ ] Optional: Published

---

## Time Estimates

| Phase | Estimated Time |
|-------|----------------|
| Phase 1: Setup & Planning | 15-30 minutes |
| Phase 2: Initialize | 5-10 minutes |
| Phase 3: Implement | 1-3 hours |
| Phase 4: Test & Debug | 30 minutes - 2 hours |
| Phase 5: Package | 10-15 minutes |
| **Total** | **2-6 hours** |

*Times vary based on complexity, API integrations, and OAuth requirements.*

---

## Next Steps After Completion

After successful packaging:

1. **Document the plugin**
   - Update README.md with usage examples
   - Add screenshots/demos
   - Document API requirements

2. **Version control**
   - Initialize git repository
   - Commit all files
   - Tag version

3. **Share with users**
   - Provide installation instructions
   - Share .difypkg file
   - Optionally publish to marketplace

4. **Iterate based on feedback**
   - Gather user feedback
   - Fix bugs
   - Add requested features
   - Release new versions
