# Web Design Guidelines for Zalo Mini Apps

Comprehensive UI/UX guidelines combining Web Interface best practices with Zalo Mini App platform requirements.

---

## Core Principles

1. **Accessibility First** - Semantic HTML, keyboard navigation, screen reader support
2. **Performance-Conscious** - Fast load times (<3s), bundle size <10MB
3. **Mobile-Ready** - Touch interactions, safe areas, responsive layouts
4. **URL Reflects State** - Deep linking, shareable states
5. **Zalo Platform Aligned** - Follow Zalo design tokens, navigation patterns

---

## Accessibility Rules

### Interactive Elements

- Icon-only buttons require `aria-label`
- Form controls need `<label>` or `aria-label`
- Interactive elements need keyboard handlers (`onKeyDown`/`onKeyUp`)
- Use `<button>` for actions, `<a>`/`<Link>` for navigation (never `<div onClick>`)
- Decorative icons need `aria-hidden="true"`

```tsx
// Incorrect
<div onClick={handleClick}>Click me</div>
<Icon icon="zi-home" />

// Correct
<button onClick={handleClick}>Click me</button>
<Icon icon="zi-home" aria-hidden="true" />
<button aria-label="Go home"><Icon icon="zi-home" /></button>
```

### Images & Media

- Images need `alt` attribute (or `alt=""` if decorative)
- `<img>` needs explicit `width` and `height` (prevents CLS)

```tsx
// Correct
<img src={avatar} alt="User avatar" width={48} height={48} />
<img src={decorative} alt="" width={100} height={100} />
```

### Live Regions

- Async updates (toasts, validation errors) need `aria-live="polite"`

```tsx
<div aria-live="polite" role="status">
  {errorMessage}
</div>
```

### Structure

- Use semantic HTML (`<button>`, `<a>`, `<label>`, `<table>`) before ARIA
- Headings must be hierarchical `<h1>`-`<h6>`
- Add `scroll-margin-top` on heading anchors

---

## Focus States

- Interactive elements need visible focus: `focus-visible:ring-*` or equivalent
- Never `outline-none` / `outline: none` without focus replacement
- Use `:focus-visible` over `:focus` (avoids focus ring on click)
- Use `:focus-within` for compound controls

```css
/* Correct */
button:focus-visible {
  outline: 2px solid var(--zmp-primary-color);
  outline-offset: 2px;
}

/* Never */
button:focus {
  outline: none;
}
```

---

## Forms

### Input Configuration

- Inputs need `autocomplete` and meaningful `name`
- Use correct `type` (`email`, `tel`, `url`, `number`) and `inputmode`
- Disable spellcheck on emails, codes, usernames (`spellCheck={false}`)
- Use `autocomplete="off"` on non-auth fields to avoid password manager triggers

```tsx
<Input
  type="tel"
  name="phone"
  inputMode="numeric"
  autoComplete="tel"
  label="Phone Number"
/>

<Input
  type="text"
  name="otp"
  inputMode="numeric"
  autoComplete="one-time-code"
  spellCheck={false}
/>
```

### Labels & Interaction

- Labels must be clickable (`htmlFor` or wrapping control)
- Checkboxes/radios: label + control share single hit target (no dead zones)
- Never block paste (`onPaste` + `preventDefault`)

### Submit & Validation

- Submit button stays enabled until request starts; show spinner during request
- Display errors inline next to fields
- Focus first error on submit
- Warn before navigation with unsaved changes

```tsx
// Zalo Mini App pattern with ZaUI
<Button loading={isSubmitting} onClick={handleSubmit}>
  {isSubmitting ? "Saving..." : "Save"}
</Button>
```

### Placeholders

- End with `...` and show example pattern: `"Enter phone number..."`

---

## Animation

### Performance

- Honor `prefers-reduced-motion` (provide reduced variant or disable)
- Animate `transform`/`opacity` only (compositor-friendly)
- Never `transition: all`-list properties explicitly
- Set correct `transform-origin`

```css
/* Incorrect */
.card {
  transition: all 0.3s ease;
}

/* Correct */
.card {
  transition: transform 0.3s ease, opacity 0.3s ease;
}

@media (prefers-reduced-motion: reduce) {
  .card {
    transition: none;
  }
}
```

### SVG Animation

- Transforms on `<g>` wrapper with `transform-box: fill-box; transform-origin: center`
- Wrap SVG in div for hardware acceleration

```tsx
// Incorrect: no hardware acceleration
<svg className="animate-spin">
  <circle cx="12" cy="12" r="10" />
</svg>

// Correct: hardware accelerated
<div className="animate-spin">
  <svg>
    <circle cx="12" cy="12" r="10" />
  </svg>
</div>
```

### Interruptibility

- Animations must be interruptible-respond to user input mid-animation

---

## Typography

### Character Usage

- Use `...` not `...` (ellipsis character)
- Use curly quotes `"` `"` not straight `"`
- Non-breaking spaces: `10&nbsp;MB`, brand names

### Loading States

- End with `...`: `"Loading..."`, `"Saving..."`

### Number Formatting

- Use `font-variant-numeric: tabular-nums` for number columns/comparisons
- Use `Intl.NumberFormat` not hardcoded formats

```tsx
// Correct
const formatter = new Intl.NumberFormat('vi-VN', {
  style: 'currency',
  currency: 'VND'
});
formatter.format(1000000); // "1.000.000 ₫"
```

### Zalo Typography Scale

| Type | Size | Weight |
|------|------|--------|
| Title | 20px | Bold |
| Subtitle | 16px | Medium |
| Body | 14px | Regular |
| Caption | 12px | Regular |

### Text Wrapping

- Use `text-wrap: balance` or `text-pretty` on headings (prevents widows)

---

## Content Handling

- Text containers must handle long content: `truncate`, `line-clamp-*`, or `break-words`
- Flex children need `min-w-0` to allow text truncation
- Handle empty states-never render broken UI for empty strings/arrays
- User-generated content: anticipate short, average, and very long inputs

```css
.truncate {
  overflow: hidden;
  text-overflow: ellipsis;
  white-space: nowrap;
}

.line-clamp-2 {
  display: -webkit-box;
  -webkit-line-clamp: 2;
  -webkit-box-orient: vertical;
  overflow: hidden;
}
```

---

## Images

- `<img>` needs explicit `width` and `height` (prevents CLS)
- Below-fold images: `loading="lazy"`
- Above-fold critical images: `fetchpriority="high"`

```tsx
// Hero image
<img src={hero} alt="Banner" width={375} height={200} fetchpriority="high" />

// Below fold
<img src={product} alt="Product" width={120} height={120} loading="lazy" />
```

---

## Performance

### Large Lists

- Lists >50 items: virtualize (`virtua`, `content-visibility: auto`)

```css
.message-item {
  content-visibility: auto;
  contain-intrinsic-size: 0 80px;
}
```

### DOM Operations

- No layout reads in render (`getBoundingClientRect`, `offsetHeight`, `offsetWidth`, `scrollTop`)
- Batch DOM reads/writes; avoid interleaving

### Inputs

- Prefer uncontrolled inputs
- Controlled inputs must be cheap per keystroke

### Resource Loading

- Add `<link rel="preconnect">` for CDN/asset domains
- Critical fonts: `<link rel="preload" as="font">` with `font-display: swap`

### Zalo Mini App Limits

- Bundle size: <10MB
- First load: <3 seconds
- Optimize images, lazy load non-critical assets

---

## Navigation & State

- URL reflects state-filters, tabs, pagination, expanded panels in query params
- Links use `<a>`/`<Link>` (enables Cmd/Ctrl+click, middle-click)
- Deep-link all stateful UI
- Destructive actions need confirmation modal or undo window-never immediate

### Zalo Navigation Patterns

- Fixed header with back/home button on left
- Menu button (fixed by Zalo) on top right-don't place elements under it
- Bottom navigation: max 4 tabs, main sections only

```tsx
<App>
  <Page>
    <Header title="Products" showBackIcon />
    <Box className="content">{/* content */}</Box>
  </Page>
  <BottomNavigation fixed>
    <BottomNavigation.Item key="home" label="Home" icon={<Icon icon="zi-home" />} />
    <BottomNavigation.Item key="cart" label="Cart" icon={<Icon icon="zi-cart" />} />
  </BottomNavigation>
</App>
```

---

## Touch & Interaction

- Use `touch-action: manipulation` (prevents double-tap zoom delay)
- Set `-webkit-tap-highlight-color` intentionally
- Use `overscroll-behavior: contain` in modals/drawers/sheets
- During drag: disable text selection, add `inert` on dragged elements
- Use `autoFocus` sparingly-desktop only, single primary input; avoid on mobile

```css
/* Zalo Mini App touch optimization */
button, a, [role="button"] {
  touch-action: manipulation;
  -webkit-tap-highlight-color: rgba(0, 106, 245, 0.1);
}

.modal-content {
  overscroll-behavior: contain;
}
```

---

## Safe Areas & Layout

- Full-bleed layouts need `env(safe-area-inset-*)` for notches
- Avoid unwanted scrollbars: `overflow-x-hidden` on containers
- Prefer flex/grid over JS measurement for layout

```css
.bottom-bar {
  padding-bottom: calc(16px + env(safe-area-inset-bottom));
}

.page-container {
  padding-left: env(safe-area-inset-left);
  padding-right: env(safe-area-inset-right);
}
```

---

## Dark Mode & Theming

- Add `color-scheme: dark` on `<html>` for dark themes
- `<meta name="theme-color">` should match page background
- Native `<select>`: set explicit `background-color` and `color`

### Zalo Color Tokens

| Type | Light | Usage |
|------|-------|-------|
| Primary | #006AF5 | Actions, links |
| Success | #00C853 | Confirmations |
| Warning | #FFA000 | Cautions |
| Error | #FF3B30 | Errors |
| Text Primary | #141415 | Main text |
| Text Secondary | #767A7F | Secondary text |
| Background | #F4F5F6 | Page background |

---

## Locale & i18n

- Dates/times: use `Intl.DateTimeFormat` not hardcoded formats
- Numbers/currency: use `Intl.NumberFormat` not hardcoded formats
- Detect language via `navigator.languages`, not IP

```tsx
// Vietnamese date formatting
const dateFormatter = new Intl.DateTimeFormat('vi-VN', {
  year: 'numeric',
  month: 'long',
  day: 'numeric'
});

// Currency
const currencyFormatter = new Intl.NumberFormat('vi-VN', {
  style: 'currency',
  currency: 'VND'
});
```

---

## Hydration Safety

- Inputs with `value` need `onChange` (or use `defaultValue` for uncontrolled)
- Date/time rendering: guard against hydration mismatch (server vs client)
- Use `suppressHydrationWarning` only where truly needed

---

## Hover & Interactive States

- Buttons/links need `hover:` state (visual feedback)
- Interactive states increase contrast: hover/active/focus more prominent than rest

```css
.btn-primary {
  background: var(--zmp-primary-color);
}

.btn-primary:hover {
  background: #0055cc;
}

.btn-primary:active {
  background: #004099;
}
```

---

## Content & Copy

### Voice & Style

- Active voice: "Install the app" not "The app will be installed"
- Title Case for headings/buttons
- Numerals for counts: "8 items" not "eight items"
- Second person; avoid first person

### Specificity

- Specific button labels: "Save Order" not "Continue"
- Error messages include fix/next step, not just problem

---

## Loading & Feedback States

### Zalo Loading Patterns

1. **Splash Loading** - Initial app load (auto by Zalo)
2. **Progress Bar** - Below action bar for page loads
3. **Modal Loading** - Full overlay for operations

### Feedback

- Show toast for success/error states
- Provide empty states for no content
- Show error states with retry options

```tsx
import { showToast } from 'zmp-sdk/apis';

// Success feedback
showToast({ message: 'Order placed successfully' });

// Error with retry
<Box className="empty-state">
  <Text>Failed to load</Text>
  <Button onClick={retry}>Try Again</Button>
</Box>
```

---

## Zalo Design Tokens

### Spacing

Base unit: 4px (U1)
- Component padding: 12-16px
- Section spacing: 24px
- Screen margin: 16px

### Corner Radius

- Buttons: 8px
- Cards: 12px
- Modals: 16px
- Avatar: 20% of width

---

## Anti-patterns (Flag These)

### Critical

- `user-scalable=no` or `maximum-scale=1` disabling zoom
- `onPaste` with `preventDefault`
- `transition: all`
- `outline-none` without focus-visible replacement
- Inline `onClick` navigation without `<a>`
- `<div>` or `<span>` with click handlers (should be `<button>`)

### Images & Performance

- Images without dimensions
- Large arrays `.map()` without virtualization
- Layout reads in render (`getBoundingClientRect`)

### Accessibility

- Form inputs without labels
- Icon buttons without `aria-label`
- Missing `prefers-reduced-motion` handling

### Formatting

- Hardcoded date/number formats (use `Intl.*`)
- `autoFocus` without clear justification

---

## Review Checklist

### Accessibility
- [ ] All interactive elements keyboard accessible
- [ ] Icon buttons have `aria-label`
- [ ] Form inputs have labels
- [ ] Images have alt text
- [ ] Focus states visible

### Performance
- [ ] Bundle size <10MB
- [ ] Images have width/height
- [ ] Large lists virtualized
- [ ] No `transition: all`

### Touch & Mobile
- [ ] Touch targets minimum 44x44px
- [ ] Safe areas handled
- [ ] `touch-action: manipulation` on buttons

### Zalo Platform
- [ ] Header follows Zalo patterns
- [ ] Bottom nav max 4 items
- [ ] Colors match Zalo tokens
- [ ] Loading states implemented

---

## Output Format (For Code Reviews)

Group findings by file using `file:line` format. Be terse.

```text
## src/components/Button.tsx

src/components/Button.tsx:42 - icon button missing aria-label
src/components/Button.tsx:55 - animation missing prefers-reduced-motion

## src/pages/Home.tsx

src/pages/Home.tsx:12 - image missing width/height
src/pages/Home.tsx:34 - "..." -> "..."

## src/components/Card.tsx

Pass
```
