"""
memory/scripts/memory.py - Memory Skill Commands

Modernized:
- @skill_command with autowire=True for clean dependency injection
- Uses PRJ_DATA_HOME/memory for persistent storage (not PRJ_CACHE)
- Uses ConfigPaths for semantic path resolution
- Refactored to use omni.foundation for embedding and vector_store.
"""

import json
import time
import uuid
from pathlib import Path
from typing import Any

from omni.foundation.api.decorators import skill_command
from omni.foundation.config.logging import get_logger
from omni.foundation.config.paths import ConfigPaths
from omni.foundation.config.skills import SKILLS_DIR
from omni.foundation.services import vector as vector_service
from omni.foundation.services.embedding import get_embedding_service
from omni.rag.retrieval import run_recall_semantic_rows

logger = get_logger("skill.memory")

# =============================================================================
# Vector Store Initialization (Foundation Layer)
# =============================================================================


def _get_memory_path() -> Path:
    """Get memory root path using PRJ_DATA_HOME."""
    # Use ConfigPaths for semantic path resolution (Layer 1)
    paths = ConfigPaths()
    # Memory data goes in $PRJ_DATA_HOME/memory (persistent data, not cache)
    return paths.get_data_dir("memory")


MEMORY_ROOT = _get_memory_path()
DEFAULT_TABLE = "memory"


def _get_embedding(text: str) -> list[float]:
    """
    Get embedding using Foundation embedding service.

    Uses Qwen/Qwen3-Embedding-4B (2560 dimensions) with automatic model download.
    """
    service = get_embedding_service()
    return service.embed(text)[0]


def _load_skill_manifest(skill_name: str) -> tuple[dict[str, Any] | None, str | None]:
    """Load a skill's manifest and prompts."""
    skill_path = SKILLS_DIR(skill_name)

    if not skill_path.exists():
        return None, None

    # Load manifest from SKILL.md
    skill_md = skill_path / "SKILL.md"
    prompts = None

    manifest = None
    if skill_md.exists():
        try:
            import yaml

            content = skill_md.read_text(encoding="utf-8")
            if content.startswith("---"):
                parts = content.split("---", 3)
                if len(parts) >= 2:
                    manifest = yaml.safe_load(parts[1])
        except Exception:
            pass

    # Load prompts.md
    prompts_path = skill_path / "prompts.md"
    if prompts_path.exists():
        prompts = prompts_path.read_text(encoding="utf-8")

    return manifest, prompts


# =============================================================================
# Memory Commands
# =============================================================================


@skill_command(
    name="save_memory",
    category="write",
    description="""
    Store a key insight into long-term memory (LanceDB).

    Use this when you've learned something reusable:
    - "Use scope 'nix' for flake changes"
    - "The project uses Conventional Commits"
    - "Use 'just test' for normal development; 'just validate' only for release pre-release"

    Args:
        - content: str - The insight to store (what you learned) (required)
        - metadata: Optional[Dict[str, Any]] - Dictionary of metadata (tags, domain, etc.)

    Returns:
        Confirmation message with stored content preview.
    """,
    autowire=True,
)
async def save_memory(
    content: str,
    metadata: dict[str, Any] | str | None = None,
    paths: ConfigPaths | None = None,
) -> str:
    """
    [Long-term Memory] Store a key insight, decision, or learning into LanceDB.

    Use this when you've learned something reusable:
    - "Use scope 'nix' for flake changes"
    - "The project uses Conventional Commits"
    - "Use 'just test' for normal development; 'just validate' only for release pre-release"

    Args:
        content: The insight to store (what you learned)
        metadata: Optional metadata dict (tags, domain, etc.)

    Returns:
        Confirmation message with stored content preview
    """
    client = vector_service.get_vector_store()
    store = client.store
    if not store:
        raise RuntimeError("VectorStore not available. Cannot store memory.")

    try:
        doc_id = str(uuid.uuid4())

        # [FIX] Robust metadata handling - handle str, None, or dict
        if metadata is None:
            metadata = {}
        elif isinstance(metadata, str):
            try:
                # LLM sometimes passes JSON string instead of dict
                metadata = json.loads(metadata)
            except (json.JSONDecodeError, TypeError):
                metadata = {"raw_metadata": metadata}
        elif not isinstance(metadata, dict):
            # Handle other unexpected types
            metadata = {"raw_metadata": str(metadata)}

        # Type narrowing: ensure metadata is dict before modification
        assert isinstance(metadata, dict), "metadata should be dict after handling"

        # Add timestamp to metadata (after ensuring it's a dict)
        metadata["timestamp"] = time.time()

        success = await client.add(content, metadata, collection=DEFAULT_TABLE)
        if success:
            return f"Saved memory [{doc_id[:8]}]: {content[:80]}..."
        raise RuntimeError("Failed to store memory.")

    except Exception as e:
        logger.error("save_memory failed", error=str(e))
        raise


@skill_command(
    name="search_memory",
    category="read",
    description="""
    Semantically search memory for relevant past experiences or rules.

    Examples:
    - search_memory("git commit message format")
    - search_memory("nixfmt error solution")
    - search_memory("how to add a new skill")

    Args:
        - query: str - What you're looking for (required)
        - limit: int = 5 - Number of results to return

    Returns:
        Relevant memories found, or "No relevant memories found".
    """,
    autowire=True,
)
async def search_memory(
    query: str,
    limit: int = 5,
    paths: ConfigPaths | None = None,
) -> str:
    """
    [Retrieval] Semantically search memory for relevant past experiences or rules.

    Examples:
    - search_memory("git commit message format")
    - search_memory("nixfmt error solution")
    - search_memory("how to add a new skill")

    Args:
        query: What you're looking for
        limit: Number of results to return (default: 5)

    Returns:
        Relevant memories found, or "No relevant memories found"
    """
    try:
        rows = await run_recall_semantic_rows(
            vector_store=vector_service.get_vector_store(),
            query=query,
            collection=DEFAULT_TABLE,
            fetch_limit=limit,
            use_cache=True,
        )

        if not rows:
            return "No matching memories found."

        output = [f"Found {len(rows)} matches for '{query}':"]
        for row in rows:
            # Format output for LLM consumption with normalized row contract.
            score = float(row.get("score", 0.0))
            content = str(row.get("content", ""))
            source = str(row.get("source", "")).strip()
            line = f"- [Score: {score:.4f}] {content[:100]}"
            if source:
                line += f" (Source: {source})"
            output.append(line)

        return "\n".join(output)

    except Exception as e:
        error_msg = str(e).lower()
        if "table" in error_msg and "not found" in error_msg:
            raise RuntimeError(
                "No memories stored yet. Use save_memory() to store insights first."
            ) from e
        logger.error("search_memory failed", error=str(e))
        raise


@skill_command(
    name="index_memory",
    category="write",
    description="""
    Optimize memory index for faster search using IVF-FLAT algorithm.

    Call this after bulk imports to improve search performance.

    Args:
        - None

    Returns:
        Confirmation of index creation.
    """,
    autowire=True,
)
async def index_memory(
    paths: ConfigPaths | None = None,
) -> str:
    """
    [Optimization] Create/optimize vector index for faster search.

    Call this after bulk imports to improve search performance.
    Uses IVF-FLAT algorithm for ANN search.

    Returns:
        Confirmation of index creation
    """
    success = await vector_service.get_vector_store().create_index(collection=DEFAULT_TABLE)
    if success:
        return "Index creation/optimization complete. Search performance improved."
    raise RuntimeError("Failed to create index.")


@skill_command(
    name="get_memory_stats",
    category="view",
    description="""
    Get statistics about stored memories.

    Args:
        - None

    Returns:
        Count of stored memories.
    """,
    autowire=True,
)
async def get_memory_stats(
    paths: ConfigPaths | None = None,
) -> str:
    """
    [Diagnostics] Get statistics about stored memories.

    Returns:
        Count of stored memories
    """
    count = await vector_service.get_vector_store().count(collection=DEFAULT_TABLE)
    return f"Stored memories: {count}"


@skill_command(
    name="load_skill",
    category="write",
    description="""
    Load a skill's manifest into semantic memory for LLM recall.

    Usage:
    - load_skill("git") - Load git skill
    - load_skill("terminal") - Load terminal skill

    Args:
        - skill_name: str - Name of the skill to load (e.g., git, terminal) (required)

    Returns:
        Confirmation message with skill details.
    """,
    autowire=True,
)
async def load_skill(
    skill_name: str,
    paths: ConfigPaths | None = None,
) -> str:
    """
    [Skill Loader] Load a single skill's manifest into semantic memory.

    Usage:
    - load_skill("git") - Load git skill
    - load_skill("terminal") - Load terminal skill

    This enables LLM to recall skill capabilities via semantic search.

    Returns:
        Confirmation message with skill details
    """
    client = vector_service.get_vector_store()
    store = client.store
    if not store:
        raise RuntimeError("VectorStore not available. Cannot load skill.")

    manifest, prompts = _load_skill_manifest(skill_name)
    if not manifest:
        raise RuntimeError(f"Skill '{skill_name}' not found or invalid manifest.")

    # Build document from manifest
    routing_kw = manifest.get("routing_keywords", [])
    intents = manifest.get("intents", [])
    deps = manifest.get("dependencies", [])

    document = f"""# {manifest.get("name", skill_name)}

{manifest.get("description", "No description.")}

**Version:** {manifest.get("version", "unknown")}
**Routing Keywords:** {", ".join(routing_kw)}
**Intents:** {", ".join(intents)}
**Dependencies:** {", ".join(deps) if deps else "None"}
"""

    # Append prompts.md content if available
    if prompts:
        document += f"\n---\n\n## System Prompts\n{prompts[:2000]}"

    try:
        success = await client.add(
            document,
            metadata={
                "type": "skill_manifest",
                "skill_name": skill_name,
                "version": manifest.get("version", "unknown"),
            },
            collection=DEFAULT_TABLE,
        )
        if success:
            return f"Skill '{skill_name}' loaded into semantic memory."
        raise RuntimeError(f"Failed to load skill '{skill_name}'.")
    except Exception as e:
        logger.error("load_skill failed", error=str(e))
        raise


__all__ = [
    "DEFAULT_TABLE",
    "MEMORY_ROOT",
    "get_memory_stats",
    "index_memory",
    "load_skill",
    "save_memory",
    "search_memory",
]
