---
name: bee-daily-journal
description: "Generate daily journal from Bee conversations + Google Calendar. Use when user says 'daily journal', 'journal for [date]', 'summarize my day', 'что я делал [дата]', 'дневник за [дату]'. Creates structured summary with Calendar vs Bee comparison, timeline Gantt, mind map by meetings/people, and action items."
---

# Bee Daily Journal Generator

Creates comprehensive daily summaries combining **Bee AI conversations** with **Google Calendar** events.

## When to Use

- User says "daily journal", "journal for [date]", "summarize my day"
- User says "что я делал [дата]", "дневник за [дату]", "день [дата]"
- User wants to understand what happened on a specific day
- End of day reflection / weekly review

## Output Location

`data_sources/obsidian/vault/Daniel Personal/Daniel Journal/{YYYY-MM-DD}/{YYYY-MM-DD} Daily Summary.md`

## Data Sources

### 1. Bee CLI (conversations, utterances)
```bash
# Get all conversations for date range
bee conversations list --limit 50

# Get specific conversation with full utterances
bee conversations get <id>

# Get last 10 hours real-time context
bee now

# Search conversations by topic
bee search conversations --query "topic"
```

### 2. Google Calendar API
```python
from data_sources.google_calendar.calendar_client import GoogleCalendarClient
from datetime import datetime
import pytz

client = GoogleCalendarClient()
client.authenticate()

pst = pytz.timezone('America/Los_Angeles')
start = pst.localize(datetime(YYYY, MM, DD, 0, 0, 0))
end = pst.localize(datetime(YYYY, MM, DD, 23, 59, 59))

events = client.calendar_service.events().list(
    calendarId='primary',
    timeMin=start.isoformat(),
    timeMax=end.isoformat(),
    maxResults=50,
    singleEvents=True,
    orderBy='startTime'
).execute().get('items', [])
```

## Document Structure

### 1. Header
```markdown
# Daily Summary — {YYYY-MM-DD} ({Day of Week})

> {One-line summary from Bee daily or generated}

**Calendar Events:** {N} | **Bee Conversations:** {N} | **Duration:** {start} – {end} PST
```

### 2. Highlights (MANDATORY)
Extract the most vivid, funny, or touching moments from family/personal conversations.
- **Read full utterances** (`bee conversations get <id>`) for personal/family conversations
- Look for: funny quotes, kids' wordplay, games, emotional moments, unexpected observations
- Format: compact blockquotes, 3–6 highlights per day
- Bold the punchline or key phrase
- Include context in 1 sentence before the quote
```markdown
## Highlights

> **"Quote here"** — Brief context. "**Punchline.**"

> **"Another quote"** — What happened. "Response quote."

> **Tender moment:** "Sweet quote" / "Another sweet quote"
```
**What makes a good highlight:**
- Kids' imaginative games and dialogue (they create storylines!)
- Misheard words / funny interpretations
- Parent-child negotiations (bedtime, food, toys)
- Tender expressions of love
- Unexpected work thoughts during family time (AI observations, etc.)

### 3. Calendar vs Reality Table
Compare scheduled meetings with actual Bee recordings:
```markdown
| Time | 📅 Calendar | 🐝 Bee (actual) |
|------|------------|-----------------|
| 09:00 | Meeting Name | (Bee: HH:MM–HH:MM) Topic |
```

### 4. Timeline Gantt
Two sections: Calendar (Planned) and Bee (Actual)
```mermaid
gantt
    title Day — {Date}
    dateFormat HH:mm

    section Calendar (Planned)
    Meeting 1    :09:00, 60m

    section Bee (Actual)
    Conversation 1    :09:15, 45m
```

### 5. Mind Map — By Meetings & People
Structure by **time + person**, not by abstract categories:
```mermaid
mindmap
  root((Date))
    09:00 Person Name
      Topic 1
        Detail
      Topic 2
    10:00 Another Person
      Topic discussed
```

**Key principle:** Show WHO said WHAT, not just topics.

### 6. Conversations Detail
For each Bee conversation:
- Time range
- Who (from calendar or detected)
- Summary from Bee
- Key takeaways

### 7. Action Items
Collected from all conversations, grouped by priority.

## Workflow

1. **Get target date** from user (or use yesterday/today)
2. **Fetch Calendar events** for that date
3. **Fetch Bee conversations** overlapping that date (summaries first via `list`)
4. **Read full utterances** for family/personal conversations (`bee conversations get <id>`)
5. **Extract highlights** — scan utterances for funny, touching, or vivid moments
6. **Match conversations to calendar** events by time
7. **Generate document** — Highlights section goes right after header, before Calendar table
8. **Save to Obsidian vault** in Journal folder

## Example Usage

```
User: "journal for yesterday"
User: "дневник за 5 февраля"
User: "summarize my day"
User: "что я делал вчера"
```

## Notes

- Timezone: America/Los_Angeles (PST)
- Calendar token: `data_sources/gmail/credentials/extended_token.pickle`
- Bee auth: `bee status` to verify
- All conversations are confidential — output only to personal Obsidian vault
