---
name: claude-task-from-logseq
description: Execute autonomous tasks received from Logseq automation and write results back using PAGE_UUID-safe updates. Use when user says "execute Logseq task", "выполни задачу из логсек", or prompt contains AUTONOMOUS MODE with PAGE_UUID.
---

# Claude Task from Logseq

Use when agent receives message containing `[[task_name]]` from Logseq automation, or when user says "execute Logseq task", "выполни задачу из логсек".

## CRITICAL: UUID-Based Operations

⚠️ **The prompt contains PAGE_UUID!** When you receive a task message like:
```
AUTONOMOUS MODE... Read [[task name]] (PAGE_UUID: 69abc123...). Execute task, write **Done:** result to PAGE_UUID 69abc123...
```

**ALWAYS use the PAGE_UUID for writing** - not the task name! This prevents writing to wrong page if name has typos.

## CRITICAL: Complete Task Lifecycle

When you receive a task from Logseq (message like "AUTONOMOUS MODE... Read [[task_name]]"), you MUST follow this complete lifecycle:

### Phase 1: Read Task

```bash
# Get task content from Logseq using lsq CLI
$PROJECT_ROOT/data_sources/logseq/bin/lsq get "task_name"
```

### Phase 2: Execute Task

- Understand what the task requires
- Use appropriate tools to complete it
- Keep track of what you did

### Phase 3: Write Result (MANDATORY!)

After completing the task, you MUST update the Logseq task page using **PAGE_UUID from the prompt**:

```bash
# Extract PAGE_UUID from the prompt message (e.g., "PAGE_UUID: 69abc123-def4-...")
PAGE_UUID="69abc123-def4-5678-90ab-cdef12345678"  # Replace with actual UUID from prompt!

# 1. Add result block using UUID (ensures correct page even with typos)
$PROJECT_ROOT/claude_venv/bin/python << EOF
import requests

token = open('/tmp/logseq_token.txt').read().strip()
headers = {"Content-Type": "application/json", "Authorization": f"Bearer {token}"}
api = "http://127.0.0.1:12315/api"

page_uuid = "$PAGE_UUID"

# Add result block
requests.post(api, headers=headers, json={
    "method": "logseq.Editor.appendBlockInPage",
    "args": [page_uuid, "**Done:** Brief summary of what was accomplished"]
}, timeout=10)

# Set status to In Review (73)
requests.post(api, headers=headers, json={
    "method": "logseq.Editor.upsertBlockProperty",
    "args": [page_uuid, ":logseq.property/status", 73]
}, timeout=5)
print(f"Task completed: {page_uuid}")
EOF
```

⚠️ **Why UUID matters:** If task name has typo (e.g., "sentnece" vs "sentence"), using name could write to wrong page. UUID is exact!

## Quick Template (UUID-based)

When task is complete, run this (replace PAGE_UUID and RESULT with values from prompt):

```bash
# Use PAGE_UUID from prompt - NEVER use task name for writing!
PAGE_UUID="69abc123-def4-5678-90ab-cdef12345678"
RESULT="Successfully completed X, created Y, fixed Z"

$PROJECT_ROOT/claude_venv/bin/python << EOF
import requests
token = open('/tmp/logseq_token.txt').read().strip()
headers = {"Content-Type": "application/json", "Authorization": f"Bearer {token}"}
api = "http://127.0.0.1:12315/api"

# Add result block using UUID
requests.post(api, headers=headers, json={
    "method": "logseq.Editor.appendBlockInPage",
    "args": ["$PAGE_UUID", "**Done:** $RESULT"]
}, timeout=10)

# Set status to In Review (73)
requests.post(api, headers=headers, json={
    "method": "logseq.Editor.upsertBlockProperty",
    "args": ["$PAGE_UUID", ":logseq.property/status", 73]
}, timeout=5)
print("Task set to In Review!")
EOF
```

## Status Flow

| Status | ID | When | Who Sets |
|--------|-----|------|----------|
| Todo | 71 | Task created | User |
| Doing | 72 | Agent launched | Server |
| In Review | 73 | Task completed | **Agent (YOU!)** |
| Done | 74 | User approved | User (manual) |

⚠️ **IMPORTANT:** Agent sets `In Review`, NOT `Done`! User reviews and sets `Done` manually.

## Checklist Before Finishing

Before you consider the task complete, verify:

- [ ] Task requirements fulfilled
- [ ] `**Done:** summary` added to task page
- [ ] Status changed to `In Review` (73)
- [ ] Stay interactive for follow-up questions (if user present)

## Example Flow

```
1. Receive: "AUTONOMOUS MODE... Read [[describe repo in 1 sentnece]] (PAGE_UUID: 69abc-123...).
   Execute task, write **Done:** result to PAGE_UUID 69abc-123..."

2. Extract PAGE_UUID from message: "69abc-123..."

3. Read task:
   lsq get "describe repo in 1 sentnece"  # Read using name is OK
   → "Describe the chrome-extension-tcs repository in one sentence"

4. Execute:
   - Analyze repository structure
   - Write description

5. Complete using UUID (CRITICAL!):
   # Use PAGE_UUID for writing - NOT the task name!
   python: appendBlockInPage("69abc-123...", "**Done:** Chrome extension for TCS...")
   python: upsertBlockProperty("69abc-123...", ":logseq.property/status", 73)

6. Task goes to In Review → User reviews → User sets Done
```

⚠️ **Note:** Even if name has typo ("sentnece"), UUID ensures we write to correct page!

## Related Skills

- `/logseq` - Read/write Logseq pages
- `/chat_sum` - Generate session summary

---
Created: 2025-12-21 | Session: Logseq Task Automation
Updated: 2025-12-21 | Changed Done → In Review (user reviews before Done)
Updated: 2025-12-21 | Added UUID-based operations to prevent writing to wrong page
