---
name: company-intelligence
description: "Unified company intelligence pipeline that auto-detects stage (prospect/active/dual) from SFDC, runs stage-appropriate prompts, and outputs to Obsidian vault. Use when user says 'company intelligence for [domain]', 'deal brief for [prospect]', 'analyze [company]', 'what do we know about [company]'. Merges Deal Intelligence Brief (prospects) and Client Intelligence (active customers) into one skill."
version: "1.0.0"
---

# Company Intelligence Skill

**Core Purpose:** Auto-detect company stage from SFDC (prospect / active customer / both), load all available context (Gong, Jira, Gmail, SFDC, client files), run stage-appropriate Gemini prompts, and write intelligence report to Obsidian vault.

## Quick Reference

**Use this skill when:**
- "Company intelligence for 898marketing.com"
- "What do we know about [company]?"
- "Deal brief for [prospect domain]"
- "Analyze [company] — prospect or customer?"
- "Generate intelligence for [company]"
- "Update report for [company]"

**Workflow:**
```mermaid
graph LR
    Input[Domain or Company Name] --> Stage[Detect Stage from SFDC]
    Stage --> Context[Load Context - Gong Jira Gmail SFDC]
    Context --> Route{Prospect or Customer?}
    Route -->|Prospect| Brief[Deal Intelligence Brief - 14 sections]
    Route -->|Active| Pipeline[3-Phase Project Intelligence]
    Route -->|Both| Both[Run Both Pipelines]
    Brief --> Obsidian[Write to Obsidian Vault]
    Pipeline --> Obsidian
    Both --> Obsidian
```

## Prerequisites

**Pipeline follows Miras Data Pipeline Philosophy — 2 files, 0 magic:**
- `algorithms/A8_G&A_div/company_intelligence/pipeline.py` — Algorithm (WHAT we do, ~50 lines)
- `algorithms/A8_G&A_div/company_intelligence/impl.py` — Implementation (HOW we do it)
- `algorithms/A8_G&A_div/company_intelligence/templates/` — Stage-specific prompt templates

**Environment:**
- `GOOGLE_API_KEY` or `GEMINI_API_KEY` in `.env`
- ClickHouse access for internal analytics

## Step-by-Step Execution

### Step 1: Run the Unified Pipeline

**One-command execution:**
```bash
claude_venv/bin/python "algorithms/A8_G&A_div/company_intelligence/pipeline.py" \
  DOMAIN_OR_NAME
```

**With flags:**
```bash
# Force prospect-only (even if active customer)
claude_venv/bin/python "algorithms/A8_G&A_div/company_intelligence/pipeline.py" \
  898marketing.com --prospect-only

# Force customer-only
claude_venv/bin/python "algorithms/A8_G&A_div/company_intelligence/pipeline.py" \
  898marketing.com --customer-only

# Full refresh (ignore existing reports)
claude_venv/bin/python "algorithms/A8_G&A_div/company_intelligence/pipeline.py" \
  898marketing.com --force-refresh

# With client folder for extra file context
claude_venv/bin/python "algorithms/A8_G&A_div/company_intelligence/pipeline.py" \
  t-mobile.com --client-folder client_cases/im_XXXX___TMobile
```

### Step 2: Review Pipeline Output

The pipeline will print:
1. **Stage detection** — prospect, active_customer, or dual_status
2. **Context stats** — how many tokens loaded, from which sources
3. **Mode** — first_run or update
4. **Generation progress** — which phases/sections are being generated
5. **Output path** — Obsidian vault location

### Step 3: Verify Obsidian Output

**Output locations:**
- `vault/Improvado/01-Revenue/11-Sales/{Company Name}/` — all stages

**File names by stage:**
- Prospect: `{Company} - Deal Intelligence Brief.md`
- Active Customer: `{Company} - Client Intelligence Report.md`
- Dual Status: Both files generated

**Frontmatter includes:** `session_id`, `stage`, `company`, `domain`, `generated_at`, `mode`, `agency_id`, tags, `related_to`

### Step 4: Open in Obsidian

```bash
# Open the generated report (example for 898 Marketing)
open "obsidian://open?vault=vault&file=Improvado%2F01-Revenue%2F11-Sales%2F898%20Marketing%2F898%20Marketing%20-%20Deal%20Intelligence%20Brief"
```

## Stage Detection Logic

**Two SFDC views queried:**

| View | What It Finds | Key Fields |
|------|---------------|------------|
| `biz_active_customers` | Active customers | `dts_agency_id`, `sfdc_primary_csm_name`, `sfdc_arr` |
| `biz_active_opportunities` | Open deals | `opportunity_stage`, `opportunity_amount`, `opportunity_close_date` |

**Stage outcomes:**

| Result | Meaning | Pipeline |
|--------|---------|----------|
| `prospect` | Open opportunity, not a customer | Deal Intelligence Brief (14 sections) |
| `active_customer` | Active customer, no open opportunities | Client Intelligence (3-phase) |
| `dual_status` | BOTH — customer with open upsell/expansion | Both pipelines run |
| `unknown` | Not found in SFDC | Error — check domain spelling |

**Test stage detection in Python REPL:**
```python
# Debug mode — run pipeline step by step
from impl import *
stage = detect_stage("898marketing.com")
context = build_context("898marketing.com", stage)
print(f"Gong: {len(context['gong_calls'])}, Jira: {len(context['jira_tickets'])}")
```

## Pipeline Outputs by Stage

### Prospect: Deal Intelligence Brief (14 Sections)

Based on Peter Cloutier's proven workflow. Single Gemini call produces:

| # | Section | Content |
|---|---------|---------|
| 1 | Executive Summary | Deal snapshot, health, key blockers |
| 2 | Opportunity Details | SFDC stage, amount, close date, competitors |
| 3 | Stakeholder Map | Contacts, roles, engagement level |
| 4 | Business Case | Pain points, desired outcomes (verbatim quotes) |
| 5 | Technical Use Case | Data sources, requirements, POC status |
| 6 | Critical Open Issues | Per-issue status, root cause, resolution |
| 7 | Commercial Blockers | Contract terms, pricing concerns |
| 8 | Evaluation Criteria | Customer's checklist with status |
| 9 | Competitive Intelligence | Named competitors, comparison |
| 10 | Engagement Timeline | Chronological interaction table |
| 11 | JIRA Tickets | Existing + recommended to create |
| 12 | Next Steps | SE priorities + AE priorities |
| 13 | Risk Assessment | Probability x impact matrix |
| 14 | Key Quotes | Verbatim customer quotes |

### Active Customer: Client Intelligence (3 Phases)

| Phase | Purpose | Output |
|-------|---------|--------|
| Phase 1 | Identify 3-9 MECE projects | JSON project list |
| Phase 2 | Generate per-project report (12 sections each) | Markdown reports |
| Phase 3 | Executive portfolio summary | Combined document |

## Update Mode

When re-running on a company that already has a report:
- **First-run mode:** Full pipeline — all phases from scratch
- **Update mode:** Reuses existing project list, only regenerates reports
- **Force refresh:** `--force-refresh` flag ignores existing reports

## Key Database Tables

| Table | Shard | Content | Lookback |
|-------|-------|---------|----------|
| `biz_active_customers` | Palantir | Customer + SFDC data | Current |
| `biz_active_opportunities` | Palantir | Open deals + SFDC | Current |
| `dim_gong_calls` | Palantir | Call metadata + transcripts | 365 days |
| `mrt_jira_tickets` | Palantir | Tickets with comments | 365 days |
| `stg_gmail_cs_messages` | Palantir | CS emails | 180 days |

## Citation Format

**All reports enforce source attribution:**
- `[SFDC: field_name]` — Salesforce data
- `[CALL: YYYY-MM-DD | Call Title]` — Gong call
- `[EMAIL: YYYY-MM-DD | Subject]` — Email thread
- `[JIRA: TICKET-ID]` — Jira ticket
- `[FILE: filename]` — Client file

## Troubleshooting

¶1 **"Company not found in SFDC"**
- Check domain spelling (use `.com` suffix)
- Try company name fragment instead of domain
- Verify company exists: `ch internal "SELECT * FROM biz_active_customers WHERE lower(company_domain_id) LIKE '%name%'"`

¶2 **Rate limits (429 from Gemini)**
- Pipeline uses `gemini-2.5-flash` (fast, cheap)
- If rate-limited, wait 60 seconds and retry
- Reduce context with lower `--max-tokens`

¶3 **JSON parse errors in Phase 1**
- Pipeline has built-in JSON recovery for truncated responses
- Check `/tmp/gemini_response_*.txt` for raw output

¶4 **Missing Gong/Jira/Email data**
- Domain pattern matching uses first part of domain (e.g., "898marketing" from "898marketing.com")
- Try alternative search terms if no data found

## Success Criteria

- [ ] Stage detected correctly (prospect/customer/dual)
- [ ] Context loaded from all available sources
- [ ] Report generated with proper structure for stage
- [ ] All claims have source citations
- [ ] Obsidian frontmatter includes session_id, stage, company
- [ ] Report written to correct vault folder
- [ ] Report opens correctly in Obsidian

## Related Skills

| Skill | Relationship |
|-------|-------------|
| `/client-intelligence` | **Predecessor** — active customer only, no stage detection |
| `/gong-call-analyzer` | Shared dependency — Gong call extraction |
| `/gmail-operations` | Shared dependency — email search |
| `/jira` | Shared dependency — ticket lookup |
| `/improvado-mcp` | ClickHouse queries for all data |

## Architecture (Miras Pipeline Philosophy)

```
algorithms/A8_G&A_div/company_intelligence/
├── pipeline.py     — Algorithm (WHAT we do, ~50 lines)
├── impl.py         — Implementation (HOW we do it)
└── templates/
    ├── prospect_prompt.md          — 14-section Deal Brief prompt
    └── active_customer_prompt.md   — 3-phase Client Intelligence prompt
```

**Philosophy:** `pipeline.py` reads like documentation. `impl.py` has all the details.
See: `vault/Miras Knowledge Platform/.../Miras Data Pipeline Philosophy.md`

---

**Primary source:** `data_sources/obsidian/vault/Improvado/01-Revenue/11-Sales/Customer Intelligence Lifecycle - Prospect to Active.md`
**Created:** 2026-02-08
**Author:** Claude Code
**Session:** 72cb0617-0c02-460e-b750-199d63378704
**Version:** 1.0.0 — Initial unified pipeline merging Deal Brief + Client Intelligence
