---
name: download-session-from-drive
description: Download and import Claude Code sessions from Google Drive. Browse available sessions with metadata, select desired session, and automatically install to current project for immediate resume. Use when recovering archived sessions, importing shared sessions from colleagues, or resuming work from different machines.
---

# Download Session from Drive Skill

This skill downloads Claude Code session files from Google Drive and automatically installs them to your current project for immediate resume.

## ⚠️ Prerequisites

**REQUIRED:** Google Drive API credentials file must be present before using this skill.

**Location:** `data_sources/google_drive/credentials/credentials.json`

**Setup instructions:** See [data_sources/google_drive/00_README.md](../../../data_sources/google_drive/00_README.md) for:
- Creating OAuth2 credentials in Google Cloud Console
- Enabling Google Drive API
- Downloading and placing credentials.json file

**First-time authentication:**
- On first use, browser will open for Google OAuth authorization
- After authorization, token is cached in `credentials/token.pickle`
- Subsequent uses are fully automatic (no browser)

`★ Insight ─────────────────────────────────────`
**How It Works:**
1. **Lists sessions** - Shows all .jsonl files on Drive with metadata
2. **Shows metadata** - CWD, branch, modification date for each session
3. **User selects** - Choose session from interactive list
4. **Auto-install** - Downloads and places in correct ~/.claude/projects/ folder
5. **Ready to resume** - Provides exact resume command
`─────────────────────────────────────────────────`

## Quick Start: How This Skill Works

### 🔐 First Time Use
**What happens:**
1. User invokes skill: "Download my session from Drive"
2. Skill checks if token exists
3. If no token → Opens browser for Google OAuth
4. User authorizes Google Drive access
5. Token saved for future use
6. Sessions listed from Drive
7. User selects, session downloaded and installed

**Note:** Browser will open ONLY on first use. After that, fully automatic!

### ⚡ Subsequent Uses
**What happens:**
1. User invokes skill: "Download session from Drive"
2. Skill uses cached token (no browser!)
3. Lists available sessions
4. User selects session
5. Downloads and installs automatically
6. Shows resume command

### 📥 Usage Examples

**Example 1: Download to current project**
```
User: "Download my dashboard session from Drive"

Claude: [Executes skill]

📂 Current project: $PROJECT_ROOT

🔐 Authenticating with Google Drive...
✅ Authentication successful

📋 Searching for session files on Google Drive...

Found 3 session file(s)

================================================================================
Available Sessions on Google Drive
================================================================================

⭐ 1. f7a79beb-138f-4858-8edc-f18ffa076bf0.jsonl
      Session ID: f7a79beb-138f-4858-8edc-f18ffa076bf0
      Created in: $PROJECT_ROOT
                  👆 Same as current project!
      Branch:     main
      Modified:   2025-11-11T18:34:18.207Z
      Size:       789.0 KB

   2. 14b475a3-c196-4870-9755-dcbcf393e543.jsonl
      Session ID: 14b475a3-c196-4870-9755-dcbcf393e543
      Created in: $HOME/projects/ai-agent-improvado
      Branch:     feature/dashboard
      Modified:   2025-11-10T15:22:10.123Z
      Size:       234.5 KB

Select session to download (1-2, or 'q' to quit): 1

================================================================================
📥 Downloading: f7a79beb-138f-4858-8edc-f18ffa076bf0.jsonl
================================================================================

Downloading to: /tmp/claude-sessions/f7a79beb-138f-4858-8edc-f18ffa076bf0.jsonl
✅ Download complete

================================================================================
📁 Installing to Current Project
================================================================================

Current directory:  $PROJECT_ROOT
Encoded path:       -Users-as-Desktop-Impro-tcs-chrome-extension
Target directory:   $HOME/.claude/projects/-Users-as-Desktop-Impro-tcs-chrome-extension
Target file:        f7a79beb-138f-4858-8edc-f18ffa076bf0.jsonl

✅ Session installed successfully!

================================================================================
🚀 Ready to Resume
================================================================================

To resume this session, run:

   claude --resume f7a79beb-138f-4858-8edc-f18ffa076bf0

Or use shortened ID:

   claude --resume f7a79beb

✅ All done!
```

**Example 2: Download session created in different directory**
```
User: "Download the ai-agent session from Drive"

Claude: [User selects session #2 from list]

================================================================================
ℹ️  Note: Session CWD Difference
================================================================================

Session was created in: $HOME/projects/ai-agent-improvado
Installing to:          $PROJECT_ROOT

This is OK! Claude Code will use the current directory when resuming.

================================================================================
🚀 Ready to Resume
================================================================================

To resume this session, run:

   claude --resume 14b475a3-c196-4870-9755-dcbcf393e543
```

**Example 3: Session already exists (overwrite prompt)**
```
⚠️  Session already exists: f7a79beb-138f-4858-8edc-f18ffa076bf0.jsonl

Overwrite existing session? (y/n): y

✅ Session installed successfully!
```

## Implementation

When this skill is invoked:

### 1. **Authenticate Google Drive:**
```python
from data_sources.google_drive.drive_client import GoogleDriveClient

client = GoogleDriveClient()

# Check if first time
token_path = f"{client.credentials_path}/token.pickle"
if not os.path.exists(token_path):
    print("🔐 First time: Opening browser for Google Drive authentication...")
    print("Token will be saved for future automatic use.\n")

# Authenticate (opens browser if needed, or reuses token)
client.authenticate()
```

### 2. **List sessions from Drive:**
```python
# Search for .jsonl session files
query = "name contains '.jsonl' and trashed=false"
files = client.list_files(query=query, limit=50)

# For each file, fetch metadata (first line of .jsonl)
for file in files:
    # Download only first 2KB to get metadata
    metadata = get_session_metadata_from_drive(client, file['id'])
    # metadata contains: sessionId, cwd, gitBranch, timestamp
```

### 3. **Display sessions with metadata:**
```python
# Show sessions sorted by modified time (newest first)
# Highlight sessions from current project with ⭐
for i, session in enumerate(sessions_sorted, 1):
    metadata = session['metadata']
    from_current_project = (metadata['cwd'] == current_cwd)
    marker = "⭐" if from_current_project else "  "

    print(f"{marker} {i}. {session['file_name']}")
    print(f"      Session ID: {metadata['sessionId']}")
    print(f"      Created in: {metadata['cwd']}")
    if from_current_project:
        print(f"                  👆 Same as current project!")
    print(f"      Branch:     {metadata['gitBranch']}")
    print(f"      Modified:   {session['modified']}")
```

### 4. **Download selected session:**
```python
# Download to temporary location
temp_dir = Path('/tmp/claude-sessions')
downloaded_path = client.download_file(
    file_id=file_id,
    output_path=str(temp_dir / file_name)
)
```

### 5. **Install to current project:**
```python
# Determine current project path
current_cwd = Path.cwd().absolute()

# Encode path for Claude projects folder
# Example: $HOME/projects/foo → -Users-as-Desktop-Impro-foo
encoded_path = str(current_cwd).replace('/', '-')

# Target directory: ~/.claude/projects/{encoded-path}/
target_dir = Path.home() / '.claude' / 'projects' / encoded_path
target_dir.mkdir(parents=True, exist_ok=True)

# Copy session file with correct name: {session_id}.jsonl
target_file = target_dir / f"{session_id}.jsonl"
shutil.copy2(downloaded_path, target_file)
```

### 6. **Show resume instructions:**
```python
print(f"To resume this session, run:")
print(f"   claude --resume {session_id}")
```

## Key Features

### 📊 Metadata Preview
Before downloading, see:
- **Session ID** - Full UUID for precise identification
- **CWD** - Original directory where session was created
- **Branch** - Git branch at time of session
- **Modified date** - When session was last uploaded
- **File size** - How large the session file is

### ⭐ Current Project Highlighting
Sessions matching your current directory are marked with ⭐ to help you quickly identify relevant sessions.

### 🔄 CWD Flexibility
**Important discovery:** Session CWD doesn't prevent resume!
- Sessions can be imported to any project directory
- Claude Code adapts to current directory when resuming
- Original CWD shown for reference only

### ⚠️ Overwrite Protection
If session already exists in current project:
- Skill prompts for confirmation before overwriting
- Shows which file would be replaced
- Allows cancellation without data loss

### 🧹 Automatic Cleanup
- Downloads to temporary `/tmp/claude-sessions/` directory
- Automatically removes temp file after installation
- Keeps Drive and local storage clean

## File Locations

### Credentials
```
data_sources/google_drive/credentials/
├── credentials.json    # OAuth2 client credentials (from Google Cloud Console)
└── token.pickle        # Cached authentication token (auto-generated)
```

### Session Storage
```
~/.claude/projects/
├── -Users-as-Desktop-Impro-tcs-chrome-extension/
│   └── f7a79beb-138f-4858-8edc-f18ffa076bf0.jsonl    # Installed here
├── -Users-as-Desktop-Impro-ai-agent-improvado/
│   └── 14b475a3-c196-4870-9755-dcbcf393e543.jsonl
└── ...
```

### Temporary Downloads
```
/tmp/claude-sessions/
└── f7a79beb-138f-4858-8edc-f18ffa076bf0.jsonl    # Downloaded here, then moved
```

## How Claude Code Finds Sessions

When you run `claude --resume {session_id}`:

1. **Encodes current directory:**
   - Current: `$PROJECT_ROOT`
   - Encoded: `-Users-as-Desktop-Impro-tcs-chrome-extension`

2. **Searches in encoded folder:**
   - Looks in: `~/.claude/projects/-Users-as-Desktop-Impro-tcs-chrome-extension/`
   - Finds: `{session_id}.jsonl`

3. **Resumes session:**
   - Loads entire message history
   - Restores tool state
   - Continues from where you left off

**This is why installation to current project folder is critical!**

## Troubleshooting

### Problem: "No session files found on Google Drive"

**Cause:** No .jsonl files uploaded to Drive yet

**Solution:**
```bash
# Upload a session first
User: "Upload this session to Drive"
```

### Problem: "Authentication failed"

**Cause:** Missing or invalid credentials

**Solution:**
1. Check credentials exist:
   ```bash
   ls data_sources/google_drive/credentials/credentials.json
   ```

2. If missing, follow setup in `data_sources/google_drive/00_README.md`

3. If token expired, delete and re-authenticate:
   ```bash
   rm data_sources/google_drive/credentials/token.pickle
   # Next skill run will re-authenticate
   ```

### Problem: "Session already exists"

**Cause:** Session with same ID already in current project

**Options:**
1. **Overwrite:** Enter 'y' to replace existing session
2. **Cancel:** Enter 'n' to keep existing session
3. **Different directory:** `cd` to different project and retry

### Problem: Can't find session after installation

**Cause:** Running `claude --resume` from wrong directory

**Solution:**
```bash
# Navigate to directory where you installed
cd $PROJECT_ROOT

# Then resume
claude --resume {session_id}
```

**Or use universal resume (if configured):**
```bash
# Works from any directory
rc {session_id}
```

## Related Skills

- **upload-session-to-drive** - Upload current session to Drive
- **claude-code-sessions** - Search and manage local sessions

## Security & Privacy

### What gets uploaded to Drive:
- Complete session file (.jsonl) with full conversation history
- All user messages and assistant responses
- Tool usage and results
- File contents that were read during session

### Credentials storage:
- `credentials.json` - OAuth2 client credentials (not secret, project-specific)
- `token.pickle` - Cached access token (secret, user-specific)
- Both in `data_sources/google_drive/credentials/` (git-ignored)

### Sharing considerations:
- Sessions contain full conversation context
- May include sensitive code, data, or information
- Only share with trusted colleagues
- Review session content before sharing

## Advanced Usage

### Download session from different machine

**Scenario:** You uploaded a session on Machine A, want to resume on Machine B

**Workflow:**
```bash
# On Machine B:
cd ~/your-project-directory
claude  # Start Claude Code

User: "Download session {session_id} from Drive"

# Skill downloads and installs to current directory
# Resume immediately:
claude --resume {session_id}
```

**Note:** CWD in session file may differ - this is OK! Claude Code will use Machine B's directory.

### Import session to multiple projects

**Scenario:** Want same session available in two different project directories

**Workflow:**
```bash
# Install to first project
cd ~/project-1
User: "Download session from Drive"  # Select session
claude --resume {session_id}  # Works!

# Install to second project
cd ~/project-2
User: "Download session from Drive"  # Select same session
claude --resume {session_id}  # Also works!
```

### Filter sessions by current project

**Tip:** Sessions matching current directory show ⭐ marker

```
⭐ 1. f7a79beb...  # Created in current project
   2. 14b475a3...  # Created elsewhere
```

Look for ⭐ to quickly find relevant sessions!

## Next Steps

After downloading and installing a session:

1. **Resume session:**
   ```bash
   claude --resume {session_id}
   ```

2. **Verify context restored:**
   - Check conversation history loads
   - Verify file access works
   - Test tool state preserved

3. **Continue work:**
   - Session continues exactly where it left off
   - All previous context available
   - Ready for new tasks

## Technical Notes

### Path Encoding
Claude Code encodes project paths by replacing `/` with `-`:
- `$PROJECT_ROOT`
- → `-Users-as-Desktop-Impro-tcs-chrome-extension`

This creates isolated session storage per project directory.

### Session File Format
Sessions are `.jsonl` (JSON Lines) files:
- Each line is a complete JSON object
- First line contains metadata (sessionId, cwd, gitBranch, timestamp)
- Subsequent lines are conversation messages
- Can be large (100KB - 10MB+)

### Metadata Extraction Optimization
Skill reads only first 2KB of each file to extract metadata:
- Avoids downloading entire session just to show list
- Uses Drive API `Range` header for efficient partial download
- Metadata line typically <1KB, so 2KB buffer is safe

### Installation Location
Must install to correct encoded path for `claude --resume` to work:
- ✅ Correct: `~/.claude/projects/{current-dir-encoded}/{session-id}.jsonl`
- ❌ Wrong: Any other location won't be found by Claude Code

---

**Created by:** Claude Code
**Session ID:** [current_session_id]
**Date:** 2025-11-11
**Related:** upload-session-to-drive, GoogleDriveClient, session management
