#!/usr/bin/env python3
"""
Download Claude Code Session from Google Drive

Created by: Claude Code
Session ID: [current_session_id]
Date: 2025-11-11
Purpose: Download and install Claude Code sessions from Google Drive

This skill:
1. Authenticates with Google Drive (reuses cached token)
2. Lists available session files on Drive
3. Shows metadata for each session (CWD, branch, date)
4. Downloads selected session
5. Installs to current project for immediate resume
"""

import sys
import os
import json
import shutil
from pathlib import Path
from typing import Dict, List, Optional, Any

# Add project root to path for imports
project_root = Path(__file__).resolve().parents[3]
sys.path.insert(0, str(project_root))

from data_sources.google_drive.drive_client import GoogleDriveClient


def encode_path_for_claude(path: str) -> str:
    """
    Encode filesystem path to Claude Code projects folder format.

    Example: $HOME/projects/foo → -Users-as-Desktop-Impro-foo
    """
    return path.replace('/', '-')


def get_session_metadata_from_content(content: str) -> Optional[Dict[str, Any]]:
    """
    Extract session metadata from first line of session file.

    Args:
        content: First line(s) of .jsonl file

    Returns:
        Dict with sessionId, cwd, gitBranch, timestamp, or None if parsing fails
    """
    try:
        first_line = content.split('\n')[0]
        data = json.loads(first_line)

        return {
            'sessionId': data.get('sessionId', 'UNKNOWN'),
            'cwd': data.get('cwd', 'UNKNOWN'),
            'gitBranch': data.get('gitBranch', 'UNKNOWN'),
            'version': data.get('version', 'UNKNOWN'),
            'timestamp': data.get('timestamp', 'UNKNOWN')
        }
    except (json.JSONDecodeError, IndexError) as e:
        print(f"⚠️  Warning: Could not parse session metadata: {e}")
        return None


def get_session_metadata_from_drive(
    client: GoogleDriveClient,
    file_id: str
) -> Optional[Dict[str, Any]]:
    """
    Get session metadata by reading first 2KB of file from Drive.

    This avoids downloading entire file just to show metadata.
    """
    try:
        import io
        from googleapiclient.http import MediaIoBaseDownload

        # Get first 2KB of file (enough for metadata line)
        request = client.drive_service.files().get_media(fileId=file_id)
        fh = io.BytesIO()

        # Download with chunk size limit
        downloader = MediaIoBaseDownload(fh, request, chunksize=2048)
        done = False

        # Get first chunk only
        status, done = downloader.next_chunk()

        # Read content
        fh.seek(0)
        content = fh.read().decode('utf-8')

        return get_session_metadata_from_content(content)

    except Exception as e:
        print(f"⚠️  Warning: Could not fetch metadata for file {file_id}: {e}")
        return None


def list_sessions_from_drive(client: GoogleDriveClient) -> List[Dict[str, Any]]:
    """
    List all session files from Google Drive with metadata.

    Returns:
        List of dicts with file_id, file_name, modified, size, metadata
    """
    print("📋 Searching for session files on Google Drive...")
    print()

    # Search for .jsonl files (session files)
    query = "name contains '.jsonl' and trashed=false"
    files = client.list_files(query=query, limit=50)

    if not files:
        print("❌ No session files found on Google Drive")
        print()
        print("Tip: Upload a session first using upload-session-to-drive skill")
        return []

    print(f"Found {len(files)} session file(s)")
    print()

    # Fetch metadata for each session
    sessions = []
    for file in files:
        print(f"📖 Reading metadata: {file['name'][:50]}...")

        metadata = get_session_metadata_from_drive(client, file['id'])

        if metadata:
            sessions.append({
                'file_id': file['id'],
                'file_name': file['name'],
                'modified': file.get('modifiedTime', 'UNKNOWN'),
                'size': file.get('size', 0),
                'metadata': metadata
            })

    print()
    return sessions


def display_sessions(sessions: List[Dict[str, Any]], current_cwd: str) -> None:
    """Display sessions with formatting and metadata."""

    print("=" * 80)
    print("Available Sessions on Google Drive")
    print("=" * 80)
    print()

    # Sort by modified time (newest first)
    sessions_sorted = sorted(
        sessions,
        key=lambda s: s['modified'],
        reverse=True
    )

    for i, session in enumerate(sessions_sorted, 1):
        metadata = session['metadata']

        # Highlight sessions from current project
        from_current_project = (metadata['cwd'] == current_cwd)
        marker = "⭐" if from_current_project else "  "

        print(f"{marker} {i}. {session['file_name']}")
        print(f"      Session ID: {metadata['sessionId']}")
        print(f"      Created in: {metadata['cwd']}")

        if from_current_project:
            print(f"                  👆 Same as current project!")

        print(f"      Branch:     {metadata['gitBranch']}")
        print(f"      Modified:   {session['modified']}")

        size_kb = int(session['size']) / 1024 if session['size'] else 0
        print(f"      Size:       {size_kb:.1f} KB")
        print()


def select_session(sessions: List[Dict[str, Any]]) -> Optional[Dict[str, Any]]:
    """
    Prompt user to select a session.

    Returns:
        Selected session dict, or None if cancelled
    """
    while True:
        try:
            choice = input("Select session to download (1-N, or 'q' to quit): ").strip()

            if choice.lower() == 'q':
                print("❌ Download cancelled")
                return None

            index = int(choice) - 1

            if 0 <= index < len(sessions):
                # Sort by modified time (match display order)
                sessions_sorted = sorted(
                    sessions,
                    key=lambda s: s['modified'],
                    reverse=True
                )
                return sessions_sorted[index]
            else:
                print(f"⚠️  Invalid choice. Please enter 1-{len(sessions)}")

        except ValueError:
            print("⚠️  Invalid input. Please enter a number or 'q'")
        except KeyboardInterrupt:
            print("\n❌ Download cancelled")
            return None


def download_and_install_session(
    client: GoogleDriveClient,
    session: Dict[str, Any],
    current_cwd: Path,
    force_overwrite: bool = False
) -> bool:
    """
    Download session from Drive and install to current project.

    Args:
        client: Authenticated GoogleDriveClient
        session: Session metadata dict
        current_cwd: Current working directory
        force_overwrite: If True, skip overwrite confirmation

    Returns:
        True if successful, False otherwise
    """
    file_id = session['file_id']
    file_name = session['file_name']
    metadata = session['metadata']
    session_id = metadata['sessionId']

    print()
    print("=" * 80)
    print(f"📥 Downloading: {file_name}")
    print("=" * 80)
    print()

    # Download to temporary location
    temp_dir = Path('/tmp/claude-sessions')
    temp_dir.mkdir(exist_ok=True)

    print(f"Downloading to: {temp_dir / file_name}")

    try:
        downloaded_path = client.download_file(
            file_id=file_id,
            output_path=str(temp_dir / file_name)
        )

        if not downloaded_path:
            print("❌ Download failed")
            return False

        print()
        print("✅ Download complete")
        print()

    except Exception as e:
        print(f"❌ Download error: {e}")
        return False

    # Install to current project
    print("=" * 80)
    print("📁 Installing to Current Project")
    print("=" * 80)
    print()

    # Encode current path for Claude projects directory
    encoded_path = encode_path_for_claude(str(current_cwd))
    target_dir = Path.home() / '.claude' / 'projects' / encoded_path
    target_file = target_dir / f"{session_id}.jsonl"

    print(f"Current directory:  {current_cwd}")
    print(f"Encoded path:       {encoded_path}")
    print(f"Target directory:   {target_dir}")
    print(f"Target file:        {target_file.name}")
    print()

    # Check if session already exists
    if target_file.exists():
        print(f"⚠️  Session already exists: {target_file.name}")
        print()

        if force_overwrite:
            print("🔧 Force mode enabled - overwriting without confirmation")
            print()
        else:
            while True:
                overwrite = input("Overwrite existing session? (y/n): ").strip().lower()
                if overwrite in ['y', 'yes']:
                    break
                elif overwrite in ['n', 'no']:
                    print("❌ Installation cancelled")
                    # Clean up temp file
                    Path(downloaded_path).unlink(missing_ok=True)
                    return False
                else:
                    print("Please enter 'y' or 'n'")

    # Create target directory
    target_dir.mkdir(parents=True, exist_ok=True)

    # Copy session file
    try:
        shutil.copy2(downloaded_path, target_file)
        print("✅ Session installed successfully!")
        print()

        # Clean up temp file
        Path(downloaded_path).unlink(missing_ok=True)

    except Exception as e:
        print(f"❌ Installation error: {e}")
        return False

    # Show CWD mismatch warning if applicable
    original_cwd = metadata['cwd']
    if original_cwd != str(current_cwd):
        print("=" * 80)
        print("ℹ️  Note: Session CWD Difference")
        print("=" * 80)
        print()
        print(f"Session was created in: {original_cwd}")
        print(f"Installing to:          {current_cwd}")
        print()
        print("This is OK! Claude Code will use the current directory when resuming.")
        print()

    # Show resume instructions
    print("=" * 80)
    print("🚀 Ready to Resume")
    print("=" * 80)
    print()
    print("To resume this session, run:")
    print()
    print(f"   claude --resume {session_id}")
    print()
    print("Or use shortened ID:")
    print()
    print(f"   claude --resume {session_id[:8]}")
    print()

    if os.path.exists(os.path.expanduser('~/.zshrc')):
        # Check if universal resume function exists
        with open(os.path.expanduser('~/.zshrc'), 'r') as f:
            if 'resume_claude' in f.read():
                print("Or use universal resume alias:")
                print()
                print(f"   rc {session_id[:10]}")
                print()

    return True


def main():
    """Main workflow: authenticate → list → select → download → install"""

    import argparse

    # Parse command line arguments
    parser = argparse.ArgumentParser(
        description='Download Claude Code session from Google Drive',
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
  # Interactive mode (shows list, prompts for selection)
  python3 download_session.py

  # Auto-download by number (after seeing list)
  python3 download_session.py --select 1

  # Auto-download by session ID
  python3 download_session.py --session-id f7a79beb-138f-4858-8edc-f18ffa076bf0

  # Skip overwrite confirmation
  python3 download_session.py --select 1 --force
        """
    )
    parser.add_argument('--select', '-s', type=int,
                       help='Auto-select session by number (1-N) without prompting')
    parser.add_argument('--session-id', '-i',
                       help='Download specific session by ID')
    parser.add_argument('--force', '-f', action='store_true',
                       help='Skip overwrite confirmation (auto-overwrite)')

    args = parser.parse_args()

    print()
    print("=" * 80)
    print("Download Claude Code Session from Google Drive")
    print("=" * 80)
    print()

    # Get current working directory
    current_cwd = Path.cwd().absolute()
    print(f"📂 Current project: {current_cwd}")
    print()

    # Step 1: Authenticate with Google Drive
    print("🔐 Authenticating with Google Drive...")
    print()

    try:
        client = GoogleDriveClient().authenticate()
        print("✅ Authentication successful")
        print()

    except Exception as e:
        print(f"❌ Authentication failed: {e}")
        print()
        print("Make sure credentials are configured:")
        print("   data_sources/google_drive/credentials/credentials.json")
        print()
        print("See: data_sources/google_drive/00_README.md")
        return 1

    # Step 2: List sessions from Drive
    sessions = list_sessions_from_drive(client)

    if not sessions:
        return 1

    # Step 3: Display sessions
    display_sessions(sessions, str(current_cwd))

    # Step 4: Select session (interactive or auto)
    selected_session = None

    # Sort sessions by modified time (match display order)
    sessions_sorted = sorted(
        sessions,
        key=lambda s: s['modified'],
        reverse=True
    )

    if args.session_id:
        # Select by session ID
        print(f"🔍 Looking for session ID: {args.session_id}")
        for session in sessions_sorted:
            if session['metadata']['sessionId'].startswith(args.session_id):
                selected_session = session
                print(f"✅ Found: {session['file_name']}")
                break

        if not selected_session:
            print(f"❌ Session not found: {args.session_id}")
            return 1

    elif args.select:
        # Auto-select by number
        if 1 <= args.select <= len(sessions_sorted):
            selected_session = sessions_sorted[args.select - 1]
            print(f"✅ Auto-selected #{args.select}: {selected_session['file_name']}")
        else:
            print(f"❌ Invalid selection: {args.select} (must be 1-{len(sessions_sorted)})")
            return 1

    else:
        # Interactive selection
        selected_session = select_session(sessions)

        if not selected_session:
            return 1

    # Step 5: Download and install (pass force flag)
    success = download_and_install_session(
        client,
        selected_session,
        current_cwd,
        force_overwrite=args.force
    )

    if success:
        print("✅ All done!")
        return 0
    else:
        return 1


if __name__ == '__main__':
    try:
        exit_code = main()
        sys.exit(exit_code)
    except KeyboardInterrupt:
        print("\n\n❌ Interrupted by user")
        sys.exit(1)
    except Exception as e:
        print(f"\n\n❌ Unexpected error: {e}")
        import traceback
        traceback.print_exc()
        sys.exit(1)
