# Gmail Operations Skill - Summary

## 📋 Что это?

**Gmail Operations Skill** - полнофункциональный skill для работы с Gmail через API.

## ✅ Что умеет (без хардкода):

### 🔍 Поиск писем
- ❌ **НЕ работает** с Gmail URL ID (типа `FMfcgzQcqbWzsKXlVrtllTxwjvvKdPsz`)
- ✅ **Работает** через search по критериям:
  - От кого: `from:sender@example.com`
  - Тема: `subject:"keyword"`
  - Дата: `after:2025/11/10 before:2025/11/15`
  - Вложения: `has:attachment filename:html`

### 📥 Скачивание писем
- ✅ Автоматически создает папки по дате: `YYYY-MM-DD`
- ✅ Сохраняет email в `00_email_content.md`
- ✅ Скачивает все attachments с **оригинальными именами**
- ✅ Сохраняет ссылки на Mermaid диаграммы (если есть в письме)

### 📁 Куда сохраняет (по умолчанию):

```
algorithms/A8_G&A_div/Daniel Personal/Daniel_communications/emails/YYYY-MM-DD/
├── 00_email_content.md          # Содержимое письма
├── original_filename.html       # Вложения (имена не меняются!)
├── diagram.png                  # Если были изображения
└── mermaid_diagram.md           # Ссылки на Mermaid (если нашлись)
```

**Важно:** Папка создается автоматически по текущей дате!

### ✉️ Создание и отправка
- ✅ Создание черновиков (plain text + HTML)
- ✅ Отправка писем
- ✅ Ответы на письма (с правильным threading)
- ✅ Пересылка писем

## 🚨 Что НЕ хардкожено:

| Что | Как работает |
|-----|--------------|
| Email addresses | Передаются как параметры |
| Message IDs | Находятся через search |
| Folder paths | Создаются динамически по дате |
| Attachment names | Сохраняются оригинальные |
| Thread IDs | Получаются из API автоматически |

## 🎯 Основные примеры использования:

### 1. Найти и скачать письмо с вложением

```python
from data_sources.gmail.gmail_client import GmailClient
import base64, os
from datetime import datetime

# Authenticate
client = GmailClient('data_sources/gmail/credentials').authenticate()

# Search email
messages = client.search_messages(
    "from:efim@example.com has:attachment after:2025/11/10",
    max_results=5
)

# Get first message
message = client.get_message_content(messages[0]['id'], format='full')

# Download attachments
today = datetime.now().strftime('%Y-%m-%d')
output_dir = f"algorithms/A8_G&A_div/Daniel Personal/Daniel_communications/emails/{today}"
os.makedirs(output_dir, exist_ok=True)

attachments = client.extract_message_attachments(message)
for att in attachments:
    if att['attachmentId']:
        att_data = client.service.users().messages().attachments().get(
            userId='me', messageId=messages[0]['id'], id=att['attachmentId']
        ).execute()

        file_data = base64.urlsafe_b64decode(att_data['data'])
        with open(f"{output_dir}/{att['filename']}", 'wb') as f:
            f.write(file_data)
```

### 2. Поиск через CLI

```bash
# Поиск по отправителю
python data_sources/gmail/gmail_client.py --method search_messages \
  --query "from:sender@example.com" --limit 5

# Поиск с вложениями за сегодня
python data_sources/gmail/gmail_client.py --method search_messages \
  --query "has:attachment after:2025/11/10" --limit 10

# Поиск по теме
python data_sources/gmail/gmail_client.py --method search_messages \
  --query "subject:\"Marketing Report\"" --limit 5
```

### 3. Создать черновик

```bash
python data_sources/gmail/gmail_client.py --method create_draft \
  --to "recipient@email.com" \
  --subject "Follow-up from meeting" \
  --body "Hi, here are the materials we discussed..."
```

## 🔧 Как работает с Gmail URL:

**Проблема:**
```
URL: https://mail.google.com/mail/u/0/#inbox/FMfcgzQcqbWzsKXlVrtllTxwjvvKdPsz
                                         ↑ Этот ID НЕ работает с Gmail API!
```

**Решения:**

1. **Поиск по критериям (ЛУЧШИЙ способ):**
   - Ищем по отправителю, дате, теме
   - Gmail API вернет правильный message ID

2. **Извлечение из браузера:**
   ```javascript
   // DevTools Console
   document.querySelector('[data-legacy-thread-id]').getAttribute('data-legacy-thread-id')
   ```

3. **Список последних писем:**
   ```bash
   python data_sources/gmail/gmail_client.py --method list_messages --limit 20
   ```

## 📖 Полная документация:

Файл: `.claude/skills/gmail-operations/SKILL.md`

Содержит:
- Все методы Gmail API
- Примеры кода для каждой операции
- Workflow для common tasks
- Troubleshooting guide
- Best practices

## 🎓 Использование через Skill:

```
/gmail-operations
```

После вызова skill, можно просто сказать:
- "Найди письма от Ефима за сегодня"
- "Скачай последнее письмо с HTML вложением"
- "Создай черновик ответа"

Skill автоматически:
- ✅ Использует правильные search queries
- ✅ Создает папки по дате
- ✅ Сохраняет все с правильными именами
- ✅ Извлекает Mermaid ссылки
- ✅ Формирует markdown с метаданными

## ⚡ Key Points:

1. **Gmail URL ID ≠ API message ID** - всегда ищем через search
2. **Папки по дате** - автоматическое создание `YYYY-MM-DD`
3. **Оригинальные имена** - attachments сохраняются как есть
4. **Без хардкода** - все параметры динамические
5. **OAuth автоматический** - token.pickle обновляется сам
