---
name: gong-call-analyzer
description: Use when user says "analyze Gong call", "extract call transcript", "summarize customer call", "find calls with [client]". Automatically searches Gong calls in ClickHouse, extracts transcripts, creates summaries, and organizes in proper client folder structure.
version: "1.0.0"
---

# Gong Call Analyzer Skill

**Core Purpose:** Query Gong calls from ClickHouse, extract transcripts, create summaries, and organize call materials in client folders.

## Quick Reference

**Use this skill when:**
- "Get the last call with [client] and create a summary"
- "Analyze Gong call from [date] with [customer]"
- "Extract all calls with [company] from last month"
- "Find calls about [topic] and summarize key points"
- "Create summary for call [URL]"

**Workflow:**
```mermaid
graph LR
    Search[Search Gong Calls] --> Verify[Verify Call Exists]
    Verify --> Extract[Extract Transcript]
    Extract --> Summarize[Create Summary]
    Summarize --> Organize[Organize in Client Folder]
```

## Core Operations

### 1. Search for Calls (ALWAYS FIRST)

**CRITICAL:** ALWAYS search for calls using terminal commands BEFORE any extraction.

```bash
# Search recent calls for client
ch internal "SELECT gong_call_url, gong_call_started, gong_call_workspace_id
                     FROM internal_analytics.dim_gong_calls
                     WHERE gong_call_workspace_id ILIKE '%ClientName%'
                     ORDER BY gong_call_started DESC LIMIT 5" --palantir
```

### 2. Extract Transcript

**After confirming call exists:**

```bash
# Use existing Gong module
python -m data_sources.gong.get_transcript "[gong_call_url]" [ClientName]
```

### 3. Create Summary

**Summary structure:**
- **Overview:** 2-3 sentences about call purpose and outcome
- **Key Points:** Bulleted list of main discussion topics
- **Action Items:** Concrete next steps with owners
- **Participants:** List of attendees and their roles
- **Metrics/Commitments:** Any numbers, dates, or promises mentioned

### 4. Organize in Client Folder

**Folder structure:**
```
client_cases/[ClientName]/customer_communication/calls/
├── 15_nov_2025-kickoff_transcript.txt
├── 15_nov_2025-kickoff_SUMMARY.md
└── 20_nov_2025-followup_transcript.txt
```

## Key Database Columns

**Table:** `internal_analytics.dim_gong_calls` (Palantir shard, NON-DEFAULT)

- `gong_call_url` - Unique call URL
- `gong_call_started` - Call start timestamp
- `gong_call_workspace_id` - Client/workspace identifier
- `gong_call_transcript_string` - Full transcript text
- `gong_call_improvado_emails_array` - Improvado participants
- `gong_call_external_participants_array` - Customer participants
- `gong_call_duration_seconds` - Call length

## Python Usage

```python
# CORRECT - Use existing gong_db.py functions
from data_sources.gong.gong_db import search_calls, get_recent_calls

# Search by client name
calls = search_calls("ClientName", limit=10)

# Get recent calls for workspace
recent = get_recent_calls(workspace_id="client_workspace", days=30)
```

## Critical Rules

1. **Search First:** ALWAYS use `ch internal` in terminal to search before extraction
2. **Verify Before Extract:** Confirm call exists before writing extraction code
3. **Use Existing Module:** Never write custom DB queries, use `data_sources.gong.gong_db`
4. **Client Isolation:** NEVER mix data between clients - verify correct client folder
5. **Numbered Prefixes:** Follow `DD_month_YYYY-title` naming convention
6. **Document Queries:** Include search query in code comments

## Common Triggers

| Trigger | Action |
|---------|--------|
| "Get last call with [client]" | Search recent calls, extract latest, create summary |
| "Analyze call about [topic]" | Search by topic keywords, extract matching calls |
| "Summarize call [URL]" | Extract transcript from specific URL, create summary |
| "Find all calls from last week" | Query date range, list available calls |
| "Extract calls and emails for [client]" | Mass extraction across communication types |

## Examples

**Example 1: Single Call Summary**
```
User: "Get the last call with ExampleClient and create a summary"

Workflow:
1. Search: ch internal for ExampleClient calls
2. Verify: Confirm call exists, get URL
3. Extract: python -m data_sources.gong.get_transcript
4. Summarize: Create SUMMARY.md with key points
5. Organize: Save to client_cases/MB2_Dental/customer_communication/calls/
```

**Example 2: Mass Extraction**
```
User: "Extract all calls, emails and Jira tickets for Kaiser from last month"

Workflow:
1. Search: Query all communication types for Kaiser
2. Extract: Batch process calls, emails, tickets
3. Organize: Create dated summaries in correct folders
4. Aggregate: Cross-reference related communications
```

## Troubleshooting

**Problem:** "Call not found"
- **Solution:** Verify workspace_id spelling, check date range, search by URL directly

**Problem:** "Empty transcript"
- **Solution:** Call may be too recent (processing delay), verify call completed

**Problem:** "Wrong client folder"
- **Solution:** Double-check workspace_id matches client, verify folder naming convention

---

**Primary source:** `.claude/agents/gong-call-analyzer.md`
**Created:** 2025-11-16
**Author:** Claude Code
**Version:** 1.0.0 - Initial skill creation from subagent
