# Gong Call → Jira/Notion Automation Skill

**Thesis:** Automated skill that downloads Gong call transcript, analyzes it with Claude Code + Gemini 3, synthesizes action items, creates/updates Jira tickets with comments, and generates a fully-linked Notion event page - replacing 2+ hours of manual work with a single command.

---

## Overview

This document defines the architecture for an end-to-end automation skill that transforms raw Gong call recordings into actionable Jira tickets and structured Notion documentation. The skill follows a **hybrid pattern**: deterministic Python code for API operations + agentic Claude Code for analysis/synthesis.

### Code vs Agent Split

```mermaid
graph TB
    subgraph PYTHON["🐍 PYTHON CODE (Deterministic)"]
        P1[Gong API - fetch transcript]
        P2[ClickHouse - find agency_id]
        P3[Jira API - search/create/comment]
        P4[Notion API - create page/blocks]
        P5[File I/O - save markdown/JSON]
        P6[Orchestration - call sequence]
    end

    subgraph AGENT["🤖 CLAUDE CODE AGENT (Agentic)"]
        A1[Analyze transcript]
        A2[Map topics to tickets]
        A3[Generate call summary]
        A4[Synthesize ACTION_ITEMS.md]
        A5[Decide new vs update]
    end

    P1 --> A1
    P2 --> P3
    A1 --> A2
    A2 --> A3
    A3 --> A4
    A4 --> A5
    A5 --> P3
    P3 --> P4

    style PYTHON fill:#e8f5e9,stroke:#4caf50,stroke-width:2px
    style AGENT fill:#e3f2fd,stroke:#2196f3,stroke-width:2px
```

**Rationale:**
- Python code = fast, predictable, debuggable, no token cost
- Agent = flexible, understands context, makes intelligent decisions

```mermaid
graph TD
    subgraph INPUT["📥 Input"]
        A[Gong Call URL/ID]
        B[Client Name]
    end

    subgraph INGEST["1️⃣ Ingest Phase"]
        C[Fetch Gong Transcript]
        D[Parse Speakers & Content]
        E[Find Client Epic & Tickets]
    end

    subgraph ANALYZE["2️⃣ Analyze Phase"]
        F[Claude Code Analysis]
        G[Gemini 3 Pro Analysis]
        H[Cross-Validation]
    end

    subgraph SYNTHESIZE["3️⃣ Synthesize Phase"]
        I[Merge Analyses]
        J[Create Action Items Doc]
        K[Map to Jira Tickets]
    end

    subgraph EXECUTE["4️⃣ Execute Phase"]
        L[Create New Jira Tickets]
        M[Add Comments to Existing]
        N[Link Everything]
    end

    subgraph PUBLISH["5️⃣ Publish Phase"]
        O[Create Notion Event]
        P[Add Full Document]
        Q[Open & Verify]
    end

    A --> C
    B --> E
    C --> D
    D --> F
    D --> G
    E --> K
    F --> H
    G --> H
    H --> I
    I --> J
    J --> K
    K --> L
    K --> M
    L --> N
    M --> N
    N --> O
    O --> P
    P --> Q

    style INPUT fill:#e1f5fe
    style INGEST fill:#fff3e0
    style ANALYZE fill:#f3e5f5
    style SYNTHESIZE fill:#e8f5e9
    style EXECUTE fill:#fce4ec
    style PUBLISH fill:#e0f2f1
```

---

## 1.0 Ingest Phase

**Purpose:** Acquire raw data from Gong and Jira to establish the working context.

```mermaid
graph LR
    subgraph GONG["Gong API"]
        G1[Get Call Metadata]
        G2[Get Transcript]
        G3[Get Participants]
    end

    subgraph JIRA["Jira API"]
        J1[Find Client Epic]
        J2[List Child Tickets]
        J3[Get Ticket Details]
    end

    subgraph OUTPUT["Local Files"]
        O1[raw_transcript.md]
        O2[tickets_snapshot.json]
    end

    G1 --> G2 --> G3 --> O1
    J1 --> J2 --> J3 --> O2
```

### 1.1 Gong Transcript Fetch (🐍 Python)

¶1 **Input:** Gong call URL or call ID
```
https://us-22694.app.gong.io/call?id=3492099935620041057
```

¶2 **Process:**
- Extract call ID from URL
- Fetch call metadata (title, date, duration)
- Fetch transcript with speaker IDs
- Fetch participants to map speaker IDs to names
- Parse and format transcript

¶3 **Output:** `{client_folder}/customer_communication/calls/{date}_{title}_TRANSCRIPT.md`

### 1.2 Agency Discovery (🐍 Python)

¶1 **Input:** Gong call title (auto-extracted)
```
"ExampleClient | Improvado - Working Session (Dec 3, 2025)"
```

¶2 **Process:**
```python
# Step 1: Parse client name from Gong title
title = "ExampleClient | Improvado - Working Session (Dec 3, 2025)"
client_name = title.split("|")[0].strip()  # "ExampleClient"

# Step 2: Find agency_id in ClickHouse
query = f"""
    SELECT rtbm_agency_id, agency_name, agency_schema_name
    FROM internal_analytics.src_dts_dsas_extraction_agency
    WHERE agency_name ILIKE '%{client_name}%'
"""
result = internal.query(query)  # NON-DEFAULT shard
agency_id = result[0]['rtbm_agency_id']

# Step 3: Find client folder
# Pattern: im_{agency_id}_{hash}___ClientName
client_folder = glob(f"client_cases/im_{agency_id}_*")

# Step 4: Load client config (if exists)
config_path = f"{client_folder}/client_config.json"
```

¶3 **Output:**
```python
{
    "client_name": "ExampleClient",
    "agency_id": "XXXX",
    "folder": "client_cases/im_XXXX_XXX___ExampleClient",
    "jira_epic": "PS-2944",  # from config or search
    "notion_events_db": "fe49ffa8-3d47-4d7b-a06c-f9fd9ac04a7b"
}
```

### 1.3 Jira Tickets Discovery - Smart Sync (🐍 Python)

¶1 **Problem:** Downloading all 100+ tickets every time is slow and wasteful.

¶2 **Solution:** Simple incremental sync - re-fetch only updated tickets.

```python
def smart_ticket_sync(epic_key: str, client_folder: str) -> dict:
    """
    Simple smart sync:
    - First run: fetch all tickets
    - Next runs: only fetch tickets updated since last sync
    - If ticket updated → re-extract whole ticket (simple!)
    """

    sync_file = f"{client_folder}/jira_sync_state.json"

    if exists(sync_file):
        last_sync = load_json(sync_file)
        last_time = last_sync['last_sync_at']
        cached = last_sync['tickets']

        # Only fetch updated tickets (Jira JQL handles this)
        updated = jira.search(
            f'parent = {epic_key} AND updated >= "{last_time}"'
        )

        # Re-extract whole ticket if it was updated
        for t in updated:
            cached[t['key']] = extract_full_ticket(t)
            cached[t['key']]['updated_since_last_sync'] = True

        # Mark unchanged
        for key in cached:
            if key not in [t['key'] for t in updated]:
                cached[key]['updated_since_last_sync'] = False

        tickets = cached

    else:
        # First sync - get all
        all_tickets = jira.search(f'parent = {epic_key}')
        tickets = {t['key']: extract_full_ticket(t) for t in all_tickets}

    # Save state
    save_json(sync_file, {
        'last_sync_at': now_iso(),
        'epic_key': epic_key,
        'tickets': tickets
    })

    return tickets
```

¶3 **Console Output:**
```
📥 JIRA SYNC
   Last sync: 2025-12-01 10:00 (2 days ago)
   ─────────────────────────────────
   📊 Epic PS-2944: 45 tickets cached

   🔄 Re-fetched (updated): 7 tickets
      PS-4461, PS-3856, PS-3690, PS-3683, PS-3667, PS-4505, PS-4504

   ⏭️ Unchanged (from cache): 38 tickets
   ─────────────────────────────────
   ✅ Sync: 1.2s (vs ~15s full)
```

---

## 2.0 Analyze Phase (🤖 Claude Code Agent)

**Purpose:** Extract actionable insights from transcript using Claude Code agent intelligence. Optionally cross-validate with Gemini 3 Pro.

```mermaid
graph LR
    subgraph INPUT["Input to Agent"]
        T[Transcript]
        K[Existing Tickets]
    end

    subgraph AGENT["🤖 Claude Code Agent"]
        A1[Read & Understand]
        A2[Extract Topics]
        A3[Map to Tickets]
        A4[Select Quotes]
        A5[Identify New Tickets]
    end

    subgraph OPTIONAL["Optional: Gemini Cross-Check"]
        G1[Gemini 3 Pro API]
        G2[Compare Results]
    end

    subgraph OUTPUT["Output"]
        O1[analysis.json]
    end

    T --> A1
    K --> A1
    A1 --> A2 --> A3 --> A4 --> A5
    A5 --> O1
    A5 -.-> G1
    G1 -.-> G2
    G2 -.-> O1

    style AGENT fill:#e3f2fd,stroke:#2196f3,stroke-width:2px
    style OPTIONAL fill:#fff3e0,stroke:#ff9800,stroke-dasharray:5
```

### 2.1 Agent Analysis (🤖 Claude Code)

¶1 **Agent receives context:**
- Full transcript with timestamps
- List of existing Jira tickets (key, summary, status)
- Client folder path for saving files

¶2 **Agent task (natural language - NOT rigid prompt):**
```
The Claude Code agent reads the transcript and:
1. Identifies all topics discussed
2. For each topic:
   - Extracts timestamp range (e.g., "02:33-08:00")
   - Selects best quote from participants
   - Maps to existing ticket (if applicable)
   - Marks as "new ticket needed" (if no match)
3. Writes structured output to file
```

¶3 **Output:** `{client_folder}/.../{date}_ANALYSIS.json`
```json
{
  "call_summary": "3 paragraph summary...",
  "topics": [
    {
      "title": "CivicShout Mapping Table",
      "timestamp_start": 153,
      "timestamp_end": 480,
      "timestamp_display": "02:33-08:00",
      "quote": "Devon: \"We need a mapping table...\"",
      "existing_ticket": "PS-4461",
      "action_items": ["Devon creating Google Sheet today", "Live by Dec 4"]
    },
    {
      "title": "Global Date Range Filter",
      "timestamp_start": 1143,
      "timestamp_end": 1169,
      "timestamp_display": "19:03-19:29",
      "quote": "Nermin: \"There is no straightforward solution...\"",
      "existing_ticket": null,
      "new_ticket": {
        "title": "EJ - Tableau Date Range Filter Across All Tabs",
        "priority": "Medium",
        "assignee": "user@improvado.io"
      }
    }
  ]
}
```

### 2.2 Optional: Gemini Cross-Validation (🐍 Python API call)

¶1 **When to use:**
- High-stakes calls (executive reviews, QBRs)
- Complex multi-topic discussions
- User requests `--verify` flag

¶2 **Process (Python):**
```python
# Only if --verify flag or high-stakes
if verify_mode:
    gemini_result = call_gemini_api(
        model="gemini-3-pro-preview",
        prompt=f"Analyze transcript and list topics with ticket mappings:\n{transcript}",
    )
    # Merge: union of topics, union of tickets
    merged = merge_analyses(claude_analysis, gemini_result)
```

¶3 **Output:** Updated analysis with confidence scores

---

## 3.0 Synthesize Phase (🤖 Claude Code Agent)

**Purpose:** Create unified action items document ready for Jira/Notion. This is the core agentic part - Claude Code writes the document using its understanding of the analysis.

```mermaid
graph TD
    subgraph INPUT["Merged Analysis"]
        I1[Topics + Timestamps]
        I2[Ticket Mappings]
        I3[New Ticket Recommendations]
        I4[Quotes]
    end

    subgraph PROCESS["Document Generation"]
        P1[Call Summary - 3 paragraphs]
        P2[Action Items - per ticket]
        P3[New Tickets - with details]
        P4[Next Steps - by owner]
    end

    subgraph OUTPUT["Final Document"]
        O1[ACTION_ITEMS.md]
    end

    I1 --> P1
    I2 --> P2
    I3 --> P3
    I4 --> P2
    P1 --> O1
    P2 --> O1
    P3 --> O1
    P4 --> O1
```

### 3.1 Document Structure

```markdown
# {Client} Call - Action Items
**Date:** {date}
**Recording:** [Gong Link]({url})
**Participants:** {list}

---

## Call Summary
{3 paragraphs summarizing call}

---

## Create New Tickets
### NEW 1: {Title}
- **Call Reference:** [timestamp](gong_url&t=seconds)
- **Quote:** "..."
- **Description:** ...
- **Priority:** ... | **Assignee:** ...

---

## Update Existing Tickets
### 1. {Topic}
- **Ticket:** [PS-XXXX](jira_url) - {summary}
- **Call Reference:** [timestamp](gong_url&t=seconds)
- **Quote:** "..."
- **What to Update:**
  - Bullet 1
  - Bullet 2

---

## Next Steps
| Owner | Task | Deadline |
|-------|------|----------|
| ... | ... | ... |
```

---

## 4.0 Execute Phase (🐍 Python) — Task List Contract Pattern

**Purpose:** Create/update Jira tickets with proper fields and comments. **v2.7.0: Uses Task List Contract pattern for guaranteed completion.**

> ⚠️ **Key Insight:** Agent cannot "forget" or "skip" tasks when they are explicitly listed and validated. The Task List Contract enforces completion by making progress visible and dependencies explicit.

```mermaid
graph TD
    subgraph GENERATE["📋 Generate Task List"]
        G1[Read 04_SYNTHESIS.json]
        G2[Create Task for each operation]
        G3[Set dependencies between tasks]
        G4[Save 06_TASK_LIST.json]
    end

    subgraph EXECUTE["⚡ Execute Tasks"]
        E1[Get next pending task]
        E2{Dependencies met?}
        E3[Execute task]
        E4[Validate output]
        E5{Passed?}
        E6[Mark COMPLETED]
        E7[Mark FAILED]
        E8{More tasks?}
    end

    subgraph TASKS["Task Types"]
        T1[CREATE_NOTION_PAGE]
        T2[CREATE_JIRA_TICKET x N]
        T3[ADD_JIRA_COMMENT x M]
        T4[COMPLETE_NOTION_PAGE]
    end

    G1 --> G2 --> G3 --> G4
    G4 --> E1
    E1 --> E2
    E2 -->|No| E1
    E2 -->|Yes| E3
    E3 --> E4
    E4 --> E5
    E5 -->|Yes| E6
    E5 -->|No| E7
    E6 --> E8
    E7 --> E8
    E8 -->|Yes| E1
    E8 -->|No| DONE[✅ ALL COMPLETE]

    T1 --> T2
    T2 --> T3
    T1 --> T4
    T2 --> T4
    T3 --> T4

    style GENERATE fill:#e8f5e9
    style EXECUTE fill:#fff3e0
    style TASKS fill:#e3f2fd
```

### 4.1 Task List Contract (v2.7.0)

¶1 **Core Classes** (`lib/task_list.py`):

| Class | Purpose |
|-------|---------|
| `Task` | Single task with id, type, status, dependencies, output |
| `TaskList` | Container with dependency tracking and completion check |
| `TaskListGenerator` | Creates task list from synthesis data |
| `TaskListRenderer` | Console output with progress visualization |
| `ValidationGate` | Validates each task's output before completion |
| `TaskListPersistence` | Save/load for resume capability |

¶2 **Task Types:**
```python
class TaskType(str, Enum):
    CREATE_NOTION_PAGE = "CREATE_NOTION_PAGE"    # Step 1 (no deps)
    CREATE_JIRA_TICKET = "CREATE_JIRA_TICKET"    # Step 2 (depends on Notion)
    ADD_JIRA_COMMENT = "ADD_JIRA_COMMENT"        # Step 2 (depends on Notion)
    COMPLETE_NOTION_PAGE = "COMPLETE_NOTION_PAGE"# Step 3 (depends on ALL)
```

¶3 **Dependency Chain (enforces order):**
```
Task 1: CREATE_NOTION_PAGE     ← No dependencies
Task 2: CREATE_JIRA_TICKET     ← Depends on Task 1 (needs notion_url)
Task 3: CREATE_JIRA_TICKET     ← Depends on Task 1
...
Task N: ADD_JIRA_COMMENT       ← Depends on Task 1
Task N+1: COMPLETE_NOTION_PAGE ← Depends on Tasks 1..N (needs all ticket_keys)
```

¶4 **Validation Gates:**
```python
# Each task type has a validator
ValidationGate.validate_notion_created(output)   # notion_url exists?
ValidationGate.validate_ticket_created(output)   # ticket_key starts with PS-?
ValidationGate.validate_comment_added(output)    # success=True?
ValidationGate.validate_notion_completed(output) # blocks_added >= 5?
```

### 4.2 Console Output Example

```
═══════════════════════════════════════════════════════════════
📋 TASK LIST CONTRACT
═══════════════════════════════════════════════════════════════
   Call ID: 7832716595142003667
   Total tasks: 14
   Completed: 0
   Failed: 0
   Pending: 14

────────────────────────────────────────
STEP 1: Create Notion Page
────────────────────────────────────────
   ⏳ [1] Create empty Notion page for meeting notes
       Expected: notion_url, notion_page_id
       Depends on: []

────────────────────────────────────────
STEP 2: Jira Operations (12 tasks)
────────────────────────────────────────
   ⏳ [2] Create ticket: EJ - CivicShout Mapping Table...
       Expected: ticket_key, ticket_url
       Depends on: [1]
   ⏳ [3] Create ticket: EJ - Tableau Date Range Filter...
       ...
   ⏳ [13] Add comment to: PS-4461
       Expected: comment_added
       Depends on: [1]

────────────────────────────────────────
STEP 3: Complete Notion Page
────────────────────────────────────────
   ⏳ [14] Complete Notion page with Jira ticket links
       Expected: blocks_added > 0
       Depends on: [1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13]

═══════════════════════════════════════════════════════════════
⏳ 14 tasks remaining
═══════════════════════════════════════════════════════════════
```

### 4.3 New Ticket Creation

¶1 **Required Fields (from existing tickets analysis):**
```python
{
    "project": {"key": "PS"},
    "summary": "EJ - {title}",
    "description": create_adf_description(),  # ADF format
    "issuetype": {"name": "Task"},
    "priority": {"name": "Medium"},  # or from analysis
    "parent": {"key": "{epic_key}"},
    "duedate": "{calculated_date}",
    "customfield_10201": {"id": "10877"},  # Non Billable
    "assignee": {"accountId": "{assignee_id}"}
}
```

¶2 **Description Format (ADF):**
```python
{
    "type": "doc",
    "version": 1,
    "content": [
        {"type": "paragraph", "content": [
            {"type": "text", "text": "Task: ..."},
            {"type": "hardBreak"},
            {"type": "text", "text": "Background: "},
            {"type": "text", "text": "Gong Link", "marks": [{"type": "link", "attrs": {"href": gong_url}}]},
            ...
        ]},
        {"type": "bulletList", "content": [...]}
    ]
}
```

### 4.2 Comment Addition

¶1 **Comment Format (ADF):**
```python
{
    "type": "doc",
    "version": 1,
    "content": [
        {"type": "paragraph", "content": [
            {"type": "text", "text": "Update from {date} call", "marks": [{"type": "strong"}]},
            {"type": "text", "text": " ("},
            {"type": "text", "text": "Gong {timestamp}", "marks": [{"type": "link", "attrs": {"href": gong_url}}]},
            ...
        ]},
        {"type": "bulletList", "content": [
            {"type": "listItem", "content": [{"type": "paragraph", "content": [{"type": "text", "text": "✅ ..."}]}]},
            ...
        ]}
    ]
}
```

---

## 5.0 Publish Phase (🐍 Python)

**Purpose:** Create Notion event page with full documentation. Deterministic Python code using ACTION_ITEMS.md as input.

```mermaid
graph TD
    subgraph NOTION["Notion API"]
        N1[Create Page in events_calls_meetings_db]
        N2[Set Properties: Name, Date, URLs, Type]
        N3[Append Blocks: Full document]
        N4[Update links to new tickets]
    end

    subgraph VERIFY["Verification"]
        V1[Open in browser]
        V2[Log final URLs]
    end

    N1 --> N2 --> N3 --> N4 --> V1 --> V2
```

### 5.1 Notion Event Properties

```python
{
    "Name": {"title": [{"text": {"content": "{Client} | Improvado - {Call Type} ({date})"}}]},
    "Meeting date": {"date": {"start": "{date}"}},
    "Recording URL": {"url": "{gong_url}"},
    "Meeting URL": {"url": "{meet_url}"},  # if available
    "Meeting type": {"multi_select": [{"name": "Occasional meeting"}]},
    "Done?": {"checkbox": True},
    "Description": {"rich_text": [{"text": {"content": "{short_summary}"}}]}
}
```

### 5.2 Notion Content Blocks

¶1 **Block Types Used:**
- `heading_1`, `heading_2`, `heading_3` - structure
- `paragraph` with `rich_text` - content with links
- `bulleted_list_item` - action items
- `quote` - call quotes
- `divider` - section separators

¶2 **Link Patterns:**
- Gong timestamp: `https://us-22694.app.gong.io/call?id={call_id}&t={seconds}`
- Jira ticket: `https://improvado.atlassian.net/browse/{ticket_key}`

---

## 6.0 Progress Tracking & Resume

**Purpose:** Track execution progress with checkpoints so skill can resume from interruption point.

### 6.1 State Machine

```mermaid
stateDiagram-v2
    [*] --> INIT: /gong-to-jira-notion {url}

    INIT --> GONG_FETCH: Start
    GONG_FETCH --> GONG_DONE: ✓ Transcript saved
    GONG_DONE --> AGENCY_DISCOVER: Continue
    AGENCY_DISCOVER --> AGENCY_DONE: ✓ Agency found
    AGENCY_DONE --> JIRA_DISCOVER: Continue
    JIRA_DISCOVER --> JIRA_DONE: ✓ Tickets listed
    JIRA_DONE --> ANALYZE: Continue
    ANALYZE --> ANALYZE_DONE: ✓ analysis.json
    ANALYZE_DONE --> SYNTHESIZE: Continue
    SYNTHESIZE --> SYNTHESIZE_DONE: ✓ ACTION_ITEMS.md
    SYNTHESIZE_DONE --> JIRA_EXECUTE: Continue
    JIRA_EXECUTE --> JIRA_EXECUTE_DONE: ✓ Tickets created/commented
    JIRA_EXECUTE_DONE --> NOTION_PUBLISH: Continue
    NOTION_PUBLISH --> COMPLETE: ✓ Notion page created

    GONG_FETCH --> ERROR: ✗ API fail
    AGENCY_DISCOVER --> ERROR: ✗ Not found
    ANALYZE --> ERROR: ✗ Agent error
    ERROR --> [*]: Abort with state

    COMPLETE --> [*]: Success!
```

### 6.2 Progress File (🐍 Python)

```python
# File: {client_folder}/gong_processing/{call_id}/progress.json
{
    "call_id": "3492099935620041057",
    "gong_url": "https://us-22694.app.gong.io/call?id=3492099935620041057",
    "started_at": "2025-12-03T14:30:00Z",
    "updated_at": "2025-12-03T14:32:45Z",

    "state": "JIRA_EXECUTE",  # Current state
    "progress_pct": 75,

    "checkpoints": {
        "GONG_FETCH": {"status": "done", "file": "transcript.md", "at": "14:30:15"},
        "AGENCY_DISCOVER": {"status": "done", "agency_id": "XXXX", "at": "14:30:18"},
        "JIRA_DISCOVER": {"status": "done", "epic": "PS-2944", "tickets": 45, "at": "14:30:25"},
        "ANALYZE": {"status": "done", "file": "analysis.json", "at": "14:31:30"},
        "SYNTHESIZE": {"status": "done", "file": "ACTION_ITEMS.md", "at": "14:32:00"},
        "JIRA_EXECUTE": {
            "status": "in_progress",
            "new_tickets": {"total": 2, "done": 1, "created": ["PS-4506"]},
            "comments": {"total": 10, "done": 3, "added": ["PS-4461", "PS-3693", "PS-3856"]}
        },
        "NOTION_PUBLISH": {"status": "pending"}
    },

    "error": null  # or {"stage": "JIRA_EXECUTE", "message": "API timeout", "at": "..."}
}
```

### 6.3 Console Progress Display

```
🎯 Gong → Jira/Notion Automation
━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

📥 INGEST
   [████████████████████] 100%
   ✅ Transcript: 27,002 chars, 5 speakers
   ✅ Agency: ExampleClient (10717)
   ✅ Epic: PS-2944 (45 tickets)

🔍 ANALYZE
   [████████████████████] 100%
   ✅ Topics: 11 extracted
   ✅ Mapped: 9 to existing tickets
   ✅ New: 2 tickets needed

📝 SYNTHESIZE
   [████████████████████] 100%
   ✅ ACTION_ITEMS.md: 279 lines

⚡ EXECUTE
   [████████░░░░░░░░░░░░] 40%
   ✅ Created: PS-4506
   🔄 Creating: PS-4507...
   ⏳ Comments: 0/10

📤 PUBLISH
   [░░░░░░░░░░░░░░░░░░░░] 0%
   ⏳ Waiting...

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
⏱️ Elapsed: 2m 15s | State: JIRA_EXECUTE
```

### 6.4 Resume Command

```bash
# If interrupted, resume from last checkpoint:
/gong-to-jira-notion --resume

# Or resume specific call:
/gong-to-jira-notion --resume 3492099935620041057
```

**Resume Logic (🐍 Python):**
```python
def resume_processing(call_id: str = None):
    # Find latest progress file
    if call_id:
        progress_file = f"{client_folder}/gong_processing/{call_id}/progress.json"
    else:
        # Find most recent incomplete
        progress_file = find_latest_incomplete_progress()

    progress = load_json(progress_file)

    # Skip completed stages
    if progress['checkpoints']['GONG_FETCH']['status'] == 'done':
        transcript = read_file(progress['checkpoints']['GONG_FETCH']['file'])

    # Resume from current state
    current_state = progress['state']

    if current_state == 'JIRA_EXECUTE':
        # Resume from where we left off
        remaining_new = [t for t in analysis['new_tickets']
                        if t['key'] not in progress['checkpoints']['JIRA_EXECUTE']['created']]
        remaining_comments = [t for t in analysis['existing_tickets']
                             if t not in progress['checkpoints']['JIRA_EXECUTE']['added']]

        execute_jira(remaining_new, remaining_comments)

    # Continue to next stages...
```

### 6.5 Checkpoint After Each Operation

```python
def create_jira_ticket(ticket_data, progress):
    """Create ticket and update checkpoint."""
    result = jira.create_issue(...)

    # Update checkpoint immediately after success
    progress['checkpoints']['JIRA_EXECUTE']['created'].append(result['key'])
    progress['checkpoints']['JIRA_EXECUTE']['done'] += 1
    progress['updated_at'] = now()
    save_progress(progress)

    # Update console
    print(f"   ✅ Created: {result['key']}")

    return result
```

---

## 7.0 Execution Flow

```mermaid
sequenceDiagram
    participant U as User
    participant P as 🐍 Python (main.py)
    participant A as 🤖 Claude Code Agent
    participant G as Gong API
    participant CH as ClickHouse
    participant J as Jira API
    participant N as Notion API

    U->>P: /gong-to-jira-notion {gong_url}

    rect rgb(232, 245, 233)
        Note over P,CH: 1. Ingest Phase (🐍 Python)
        P->>G: Fetch transcript + metadata
        G-->>P: transcript, title, participants
        P->>P: Parse client name from title
        P->>CH: Find agency_id
        CH-->>P: agency_id, folder
        P->>J: Search Epic & tickets
        J-->>P: epic_key, tickets[]
        P->>P: Save context.json
    end

    rect rgb(227, 242, 253)
        Note over P,A: 2-3. Analyze + Synthesize (🤖 Agent)
        P->>A: Load skill, pass context
        A->>A: Read transcript
        A->>A: Extract topics & quotes
        A->>A: Map to tickets
        A->>A: Write analysis.json
        A->>A: Write ACTION_ITEMS.md
        A-->>P: Done signal
    end

    rect rgb(232, 245, 233)
        Note over P,N: 4-5. Execute + Publish (🐍 Python)
        P->>P: Read analysis.json
        loop For each new_ticket
            P->>J: Create ticket (ADF)
            J-->>P: ticket_key
        end
        loop For each existing_ticket
            P->>J: Add comment (ADF)
        end
        P->>N: Create event page
        P->>N: Append blocks from ACTION_ITEMS.md
        P->>N: Update ticket links
        N-->>P: page_url
    end

    P->>U: ✅ Complete! URLs: jira, notion
```

### Key Insight: Hybrid Handoff

```
┌──────────────────────────────────────────────────────────────────┐
│                     EXECUTION TIMELINE                           │
├──────────────────────────────────────────────────────────────────┤
│                                                                  │
│  🐍 PYTHON          🤖 AGENT           🐍 PYTHON                 │
│  ──────────         ────────           ──────────                │
│  │                  │                  │                         │
│  │ 1. Fetch Gong    │                  │                         │
│  │ 2. Find agency   │                  │                         │
│  │ 3. Get tickets   │                  │                         │
│  │                  │                  │                         │
│  │ 4. Save context ─┼─► 5. Analyze     │                         │
│  │                  │   6. Synthesize  │                         │
│  │                  │   7. Write files │                         │
│  │                  │                  │                         │
│  │ 8. Read files ◄──┼──────────────────┤                         │
│  │ 9. Create Jira   │                  │                         │
│  │ 10. Create Notion│                  │                         │
│  │ 11. Open browser │                  │                         │
│  ▼                  ▼                  ▼                         │
│                                                                  │
│  TIME ─────────────────────────────────────────────────────────► │
└──────────────────────────────────────────────────────────────────┘
```

---

## 7.0 File Structure

```
.claude/skills/gong-to-jira-notion/
│
├── 00_SKILL_ARCHITECTURE.md      # This document (architecture)
├── SKILL.md                      # 🤖 Skill prompt (agent instructions)
├── SKILL_RELEASE_LOG.md          # Version history
│
├── lib/                          # 🐍 Python modules
│   ├── __init__.py
│   ├── base.py                   # Centralized clients (ClickHouse, Jira, Notion)
│   ├── gong_fetcher.py           # Gong API - fetch transcript
│   ├── agency_discovery.py       # ClickHouse - find client + SFDC data
│   ├── jira_sync.py              # Jira API - smart incremental sync
│   ├── jira_executor.py          # Jira API - create/comment
│   ├── notion_publisher.py       # Notion API - create event
│   ├── gemini_analyzer.py        # Gemini 3 Pro analysis (⚠️ locked model)
│   ├── synthesis.py              # Two-agent merge with confidence scoring
│   ├── task_list.py              # v2.7.0: Task List Contract (explicit task tracking)
│   └── progress_tracker.py       # Checkpoints and resume
│
└── main.py                       # 🐍 Orchestrator (calls lib + triggers agent)

Client folder structure:
```
client_cases/{im_XXXX_hash___ClientName}/
│
├── customer_communication/
│   ├── calls/                    # Gong transcripts
│   │   └── {date}_{title}_TRANSCRIPT.md
│   └── emails/
│
├── jira_tickets/                 # Jira ticket cache (in client root)
│   ├── jira_sync_state.json
│   └── tickets/*.json
│
└── gong_processing/              # Skill working directory
    └── {call_id}/
        ├── progress.json              # State machine
        ├── context.json               # Discovered context
        ├── 03_gemini_analysis.json    # Gemini analysis output
        ├── 03_claude_analysis.json    # Claude analysis output
        ├── 04_SYNTHESIS.json          # Two-agent merged result
        ├── 05_EXECUTION_RESULTS.json  # Jira/Notion results
        └── 06_TASK_LIST.json          # v2.7.0: Task List Contract state
```

### What each file does:

| File | Type | Purpose |
|------|------|---------|
| `SKILL.md` | 🤖 Agent | Instructions for Claude Code agent (analyze + synthesize) |
| `lib/base.py` | 🐍 Python | Centralized clients (ClickHouse, Jira, Notion) |
| `lib/gong_fetcher.py` | 🐍 Python | Fetch transcript from Gong API |
| `lib/agency_discovery.py` | 🐍 Python | Parse title → find agency_id → find folder → SFDC data |
| `lib/jira_sync.py` | 🐍 Python | Smart incremental sync of tickets |
| `lib/jira_executor.py` | 🐍 Python | Create tickets with ADF, add comments, priority mapping |
| `lib/notion_publisher.py` | 🐍 Python | Create Notion page, append blocks, attendees |
| `lib/gemini_analyzer.py` | 🐍 Python | Gemini 3 Pro analysis (⚠️ model locked!) |
| `lib/synthesis.py` | 🐍 Python | Two-agent merge with confidence scoring |
| `lib/task_list.py` | 🐍 Python | **v2.7.0:** Task List Contract (explicit task tracking) |
| `main.py` | 🐍 Python | Orchestrates: Python → Agent → Python |
| `execute_synthesis.py` | 🐍 Python | **v2.7.0:** Execute phase with TaskExecutor |

---

## 8.0 Usage

### 8.1 Invocation

```bash
# Via skill
/gong-to-jira-notion

# User provides:
# - Gong URL: https://us-22694.app.gong.io/call?id=...
# - Client: ExampleClient
```

### 8.2 Expected Output

```
🎯 Gong → Jira/Notion Automation
━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

📥 INGEST
   ✅ Fetched transcript: 27,002 chars, 5 speakers
   ✅ Found Epic: PS-2944 (ExampleClient Dashboard)
   ✅ Found 45 child tickets

🔍 ANALYZE
   ✅ Claude analysis: 10 topics, 8 ticket mappings, 2 new tickets
   ✅ Gemini analysis: 9 topics, 6 ticket mappings, 2 new tickets
   ✅ Cross-validation: 11 unique topics, 9 ticket mappings, 2 new tickets

📝 SYNTHESIZE
   ✅ Created: ACTION_ITEMS.md (279 lines)

⚡ EXECUTE
   ✅ Created: PS-4506 - EJ - Tableau Date Range Filter
   ✅ Created: PS-4507 - EJ - Add QA Tab
   ✅ Commented: PS-4461, PS-3693, PS-3856, PS-3690, PS-3830, PS-3834, PS-3683, PS-3667, PS-3775, PS-3586

📤 PUBLISH
   ✅ Created Notion event: ExampleClient | Improvado - Working Session (Dec 3, 2025)
   ✅ Added 142 content blocks
   ✅ Updated 2 blocks with new ticket links

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
✅ COMPLETE

📄 Local doc: client_cases/.../ACTION_ITEMS.md
🎫 Jira: 2 created, 10 commented
📓 Notion: https://notion.so/...

Time: 2m 34s
```

---

## 9.0 Configuration

### 9.1 Required Environment Variables

```bash
# Gong
GONG_ACCESS_KEY=...
GONG_ACCESS_KEY_SECRET=...

# Jira
JIRA_TOKEN=...
JIRA_USERNAME=...

# Notion
NOTION_TOKEN=...

# Gemini
GOOGLE_API_KEY=...
```

### 9.2 Client Auto-Discovery (NO hardcoded config!)

**Key principle:** Skill discovers EVERYTHING from Gong call - no client-specific configuration needed.

```python
# AUTO-DISCOVERY FLOW:
#
# 1. Gong title: "ExampleClient | Improvado - Working Session"
#                 ↓ parse
# 2. Client name: "ExampleClient"
#                 ↓ ClickHouse query
# 3. Agency info: {agency_id: XXXX, agency_name: "ExampleClient"}
#                 ↓ glob pattern
# 4. Client folder: "client_cases/im_XXXX_XXX___ExampleClient"
#                 ↓ Jira search
# 5. Epic: PS-2944 (from: project=PS AND summary~"ExampleClient" AND type=Epic)
#                 ↓ inspect existing ticket
# 6. Defaults: {engagement_type: "10877", assignee: from Epic children}

def auto_discover_client(gong_title: str) -> dict:
    """
    Auto-discover all client context from Gong call title.
    Works for ANY client - no hardcoded config.
    """

    # 1. Parse client name from Gong title
    # Patterns: "Client | Improvado - ..." or "Client - Improvado ..."
    if "|" in gong_title:
        client_name = gong_title.split("|")[0].strip()
    elif " - Improvado" in gong_title:
        client_name = gong_title.split(" - Improvado")[0].strip()
    else:
        client_name = gong_title.split(" - ")[0].strip()

    # 2. Find agency in ClickHouse (Palantir shard)
    agency = internal.query(f"""
        SELECT rtbm_agency_id, agency_name, agency_schema_name
        FROM internal_analytics.src_dts_dsas_extraction_agency
        WHERE agency_name ILIKE '%{client_name}%'
        LIMIT 1
    """, palantir=True)

    if not agency:
        raise ClientNotFoundError(f"No agency found for '{client_name}'")

    agency_id = agency[0]['rtbm_agency_id']

    # 3. Find client folder (glob pattern)
    folders = glob(f"client_cases/im_{agency_id}_*")
    if folders:
        client_folder = folders[0]
    else:
        # Create new folder if doesn't exist
        client_folder = create_client_folder(agency_id, client_name)

    # 4. Find Jira Epic
    epics = jira.search(f'project = PS AND summary ~ "{client_name}" AND type = Epic')
    if epics:
        epic_key = epics[0]['key']
    else:
        epic_key = None  # Will prompt user or create new

    # 5. Get defaults from existing tickets (if Epic found)
    defaults = {"engagement_type_id": "10877", "assignee": None}  # Non Billable default

    if epic_key:
        children = jira.search(f'parent = {epic_key} ORDER BY updated DESC', max=5)
        if children:
            # Copy settings from most recent ticket
            recent = children[0]['fields']
            if recent.get('customfield_10201'):
                defaults['engagement_type_id'] = recent['customfield_10201']['id']
            if recent.get('assignee'):
                defaults['assignee'] = recent['assignee']['accountId']

    return {
        "client_name": client_name,
        "agency_id": agency_id,
        "folder": client_folder,
        "jira_project": "PS",
        "epic_key": epic_key,
        "defaults": defaults,
        "notion_events_db": "fe49ffa8-3d47-4d7b-a06c-f9fd9ac04a7b"  # Universal events DB
    }
```

### 9.3 Handling Edge Cases

| Scenario | Handling |
|----------|----------|
| Client not in ClickHouse | Prompt user for client name, search again |
| Multiple agencies match | Show list, let user pick |
| No client folder | Create new folder with correct naming |
| No Jira Epic | Search by other patterns, or prompt user |
| No existing tickets | Use defaults (Non Billable, no assignee) |
| Unusual Gong title format | Fall back to prompt user for client name |

---

## 10.0 Error Handling

| Error | Handling |
|-------|----------|
| Gong API fails | Retry 3x, then abort with clear message |
| No Epic found | Prompt user for Epic key or create new |
| Gemini timeout | Continue with Claude-only analysis |
| Jira create fails | Log error, continue with others, report at end |
| Notion fails | Save local doc, provide manual instructions |

---

## Ground Truth

**Source Files Referenced:**
- `data_sources/gong/get_transcript.py` - Existing Gong fetcher
- `data_sources/jira/jira_client.py` - Jira client with ADF support
- `data_sources/notion/notion_client.py` - Notion client
- `algorithms/CLAUDE.md` - Internal operations guide
- `client_cases/im_XXXX_XXX___ExampleClient/01_PROJECTS/jira_tickets/` - Today's implementation

**Implementation Derived From:**
- Session: Dec 3, 2025 - ExampleClient call → Jira/Notion workflow
- Files: `05_fetch_gong_transcript.py` through `15_create_qa_tab_ticket.py`

---

## Author Checklist (Knowledge Framework Compliance)

- [x] **Thesis:** Single sentence stating specific outcome
- [x] **Overview:** Introduces ALL sections with ordering principle
- [x] **Mermaid diagrams:** TD for structure, LR for flow, sequence for interaction
- [x] **MECE sections:** 10 numbered sections covering full workflow
- [x] **Paragraph numbering:** ¶1, ¶2, ¶3 within sections
- [x] **Code examples:** Python, JSON, Markdown as appropriate
- [x] **Ground Truth:** Source files and session referenced

---

*Document created: 2025-12-03*
*Updated: 2025-12-04 (v2.7.0 - Task List Contract)*
*Author: Claude Code*
*Status: CURRENT*
