"""
Created by: Claude Code
Session ID: gong-to-jira-notion-mermaid
Date: 2025-12-03
Purpose: Add Mermaid diagram and GitHub link to Notion page for Gong-to-Jira-Notion skill documentation
"""

import os
import sys
import requests
from dotenv import load_dotenv

# Add parent directory to path for imports
sys.path.insert(0, os.path.dirname(os.path.dirname(os.path.dirname(os.path.dirname(os.path.abspath(__file__))))))
from data_sources.notion.notion_client import NotionClient

# Load environment variables
load_dotenv()


def create_mermaid_diagram_blocks() -> list:
    """
    Create Notion blocks for Mermaid diagram and GitHub link.

    Returns:
        List of Notion block objects
    """

    # GitHub repository URL
    github_url = "https://github.com/tekliner/tcs-chrome-extension/tree/main/.claude/skills/gong-to-jira-notion"

    # Mermaid diagram code (5-phase pipeline)
    mermaid_code = """graph LR
    subgraph INPUT["📥 Input"]
        A[Gong Call URL]
    end

    subgraph INGEST["1️⃣ Ingest"]
        B[Fetch Transcript]
        C[Find Client & Epic]
    end

    subgraph ANALYZE["2️⃣ Analyze"]
        D[🤖 Claude + Gemini]
        E[Extract Topics]
    end

    subgraph SYNTHESIZE["3️⃣ Synthesize"]
        F[ACTION_ITEMS.md]
    end

    subgraph EXECUTE["4️⃣ Execute"]
        G[Create/Update Jira]
    end

    subgraph PUBLISH["5️⃣ Publish"]
        H[Notion Event Page]
    end

    A --> B --> C --> D --> E --> F --> G --> H

    style INPUT fill:#e1f5fe
    style INGEST fill:#fff3e0
    style ANALYZE fill:#f3e5f5
    style SYNTHESIZE fill:#e8f5e9
    style EXECUTE fill:#fce4ec
    style PUBLISH fill:#e0f2f1"""

    blocks = [
        # Divider before the section
        {
            "object": "block",
            "type": "divider",
            "divider": {}
        },
        # Heading for the section
        {
            "object": "block",
            "type": "heading_2",
            "heading_2": {
                "rich_text": [
                    {
                        "type": "text",
                        "text": {"content": "🏗️ Skill Architecture"}
                    }
                ]
            }
        },
        # Callout with GitHub link
        {
            "object": "block",
            "type": "callout",
            "callout": {
                "rich_text": [
                    {
                        "type": "text",
                        "text": {"content": "📂 Source Code: "}
                    },
                    {
                        "type": "text",
                        "text": {
                            "content": "GitHub Repository",
                            "link": {"url": github_url}
                        },
                        "annotations": {
                            "bold": True,
                            "color": "blue"
                        }
                    }
                ],
                "icon": {
                    "type": "emoji",
                    "emoji": "💻"
                },
                "color": "gray_background"
            }
        },
        # Mermaid code block
        {
            "object": "block",
            "type": "code",
            "code": {
                "rich_text": [
                    {
                        "type": "text",
                        "text": {"content": mermaid_code}
                    }
                ],
                "language": "mermaid",
                "caption": [
                    {
                        "type": "text",
                        "text": {"content": "Gong → Jira/Notion Automation Pipeline"}
                    }
                ]
            }
        },
        # Additional paragraph explaining the flow
        {
            "object": "block",
            "type": "paragraph",
            "paragraph": {
                "rich_text": [
                    {
                        "type": "text",
                        "text": {"content": "The skill follows a "},
                        "annotations": {"color": "default"}
                    },
                    {
                        "type": "text",
                        "text": {"content": "hybrid pattern"},
                        "annotations": {"bold": True}
                    },
                    {
                        "type": "text",
                        "text": {"content": ": deterministic Python code for API operations (Gong, Jira, Notion) + agentic Claude Code for analysis and synthesis."}
                    }
                ]
            }
        }
    ]

    return blocks


def append_blocks_after(page_id: str, after_block_id: str, blocks: list) -> dict:
    """
    Append blocks after a specific block on a Notion page.

    Args:
        page_id: The page ID where blocks will be added
        after_block_id: The block ID after which to insert new blocks
        blocks: List of block objects to insert

    Returns:
        API response
    """
    token = os.getenv('NOTION_TOKEN')
    if not token:
        raise ValueError("NOTION_TOKEN not found in environment")

    headers = {
        "Authorization": f"Bearer {token}",
        "Content-Type": "application/json",
        "Notion-Version": "2022-06-28"
    }

    url = f"https://api.notion.com/v1/blocks/{page_id}/children"

    data = {
        "children": blocks,
        "after": after_block_id  # Insert after this specific block
    }

    response = requests.patch(url, headers=headers, json=data)

    if response.status_code != 200:
        print(f"❌ Error: {response.status_code}")
        print(f"Response: {response.text}")
        raise Exception(f"Failed to append blocks: {response.text}")

    return response.json()


def main():
    """Main function to add Mermaid diagram to Notion page."""

    # Page and block IDs from the user's URL
    # URL: https://www.notion.so/improvado-home/AI-Agent-Call-notion-jira-tickets-update-matching-2be9aec621258070b6dae3699106662e?source=copy_link#2bf9aec6212580fab178ff9f9872e0f9
    page_id = "2be9aec6-2125-8070-b6da-e3699106662e"  # with dashes
    after_block_id = "2bf9aec6-2125-80fa-b178-ff9f9872e0f9"  # anchor block ID

    print("🎯 Adding Mermaid diagram to Notion page...")
    print(f"   Page ID: {page_id}")
    print(f"   After block: {after_block_id}")
    print()

    # Create blocks
    blocks = create_mermaid_diagram_blocks()
    print(f"📦 Created {len(blocks)} blocks to insert")

    # Append blocks after the specified block
    try:
        result = append_blocks_after(page_id, after_block_id, blocks)
        print()
        print("✅ Successfully added blocks to Notion!")
        print(f"   Added {len(result.get('results', []))} blocks")
        print()
        print(f"📄 View page: https://www.notion.so/{page_id.replace('-', '')}")

    except Exception as e:
        print(f"❌ Failed: {e}")
        return 1

    return 0


if __name__ == "__main__":
    sys.exit(main())
