# Jira Configuration

Reference for Jira field IDs, priority mapping, and ticket creation rules.

## Priority Mapping

Jira priorities in this instance (PS project):

| Display Name | Jira ID | Usage |
|--------------|---------|-------|
| Blocker | 1 | Critical blockers |
| Critical | 2 | High priority → maps here |
| Medium | 3 | Default priority |
| Low | 4 | Low priority items |

**Mapping from analysis:**
- "High", "urgent", "blocker" → Critical (id: 2)
- "Medium", "normal" → Medium (id: 3)
- "Low", "nice to have" → Low (id: 4)

---

## Required Fields for Tickets

| Field | Custom Field ID | Description | Example Value |
|-------|-----------------|-------------|---------------|
| **Summary** | - | Ticket title with prefix | `EJ - Dashboard Feedback` |
| **Parent** | `parent` | Link to Epic | `{"key": "PS-4512"}` |
| **Agency Name** | `customfield_10290` | REQUIRED dropdown (domain) | `{"value": "earthjustice.org"}` |
| **Engagement Type** | `customfield_10201` | Billing type | `{"id": "10876"}` |
| **Due Date** | `duedate` | REQUIRED | `"2025-12-10"` |
| **Priority** | `priority` | Issue priority | `{"id": "3"}` |

**Engagement Type Values:**
- `10876` = Billable (default since v2.3.0)
- `10877` = Non Billable

---

## Ticket Title Prefix Convention

Use 2-3 letter client abbreviation:

| Client | Prefix |
|--------|--------|
| Ruder Finn | `RF -` |
| ExampleClient | `EJ -` |
| Kaiser Permanente | `KP -` |
| Novus Media | `NM -` |

**Example:** `EJ - Tableau Date Range Filter Across All Tabs`

---

## Assignee Selection Logic (CRITICAL!)

**Assignee MUST be an Improvado employee from the call participants, NOT Daniel (owner).**

### Selection Priority:
1. If transcript mentions someone as owner/responsible → assign to them
2. If only one Improvado person on call → assign to them
3. If multiple Improvado people → assign to the CSM/Account Manager
4. If unclear → ask user to confirm before creating tickets

### Common Team Members

| Name | Email | Jira Account ID |
|------|-------|-----------------|
| Ekaterina Koshkina | user@improvado.io | `61c4896ee7637900685ff59d` |
| Maxim Ezhak | user@improvado.io | - |
| Nermin Hrnjica | user@improvado.io | - |
| Elyse Waslat | user@improvado.io | - |
| Ilia Kolesnikov | user@improvado.io | - |

**Note:** Look up other assignees via Jira API: `GET /rest/api/3/user/search?query={name}`

---

## Epic Creation (When No Epic Exists)

### Epic Naming Pattern:
```
{Client Name} ({agency_id}) - Engagement
```

**Examples:**
- `Ruder Finn (10399) - Engagement`
- `ExampleClient (10717) - Engagement`

### Required Fields for Epic:

| Field | Custom Field ID | Description | Example |
|-------|-----------------|-------------|---------|
| **Summary** | - | Epic title | `Ruder Finn (10399) - Engagement` |
| **Agency Name** | `customfield_10290` | Domain format | `{"value": "ruderfinn.com"}` |

**Agency Name Dropdown:**
- Uses domain format: `ruderfinn.com`, `earthjustice.org`, `kp.org`
- Find via Google "[client name] website" or check participant emails

---

## ADF Format for Description

Jira uses Atlassian Document Format (ADF) for descriptions:

```python
{
    "type": "doc",
    "version": 1,
    "content": [
        {
            "type": "paragraph",
            "content": [
                {"type": "text", "text": "Task: "},
                {"type": "text", "text": "Description here"}
            ]
        },
        {
            "type": "paragraph",
            "content": [
                {"type": "text", "text": "Background: "},
                {
                    "type": "text",
                    "text": "Gong Recording (02:33-05:00)",
                    "marks": [{"type": "link", "attrs": {"href": "gong_url&t=153"}}]
                }
            ]
        },
        {
            "type": "blockquote",
            "content": [{
                "type": "paragraph",
                "content": [{"type": "text", "text": "Quote from call..."}]
            }]
        },
        {
            "type": "bulletList",
            "content": [
                {"type": "listItem", "content": [{"type": "paragraph", "content": [{"type": "text", "text": "Action item 1"}]}]},
                {"type": "listItem", "content": [{"type": "paragraph", "content": [{"type": "text", "text": "Action item 2"}]}]}
            ]
        }
    ]
}
```

---

## Comment Idempotency (CRITICAL!)

**Problem:** Re-running skill adds duplicate comments.

**Solution:** Check existing comments via API BEFORE adding:

```python
def check_existing_comment(ticket_key: str, call_date: str) -> bool:
    """Check if call comment already exists"""
    url = f"{JIRA_BASE}/rest/api/3/issue/{ticket_key}/comment"
    response = requests.get(url, auth=auth)

    for comment in response.json().get('comments', []):
        body_text = extract_text_from_adf(comment['body'])
        if f"Update from {call_date}" in body_text:
            return True  # Already has comment

    return False  # Needs comment
```

---

## API Examples

### Create Ticket:
```python
response = jira._make_request('POST', 'issue', data={
    "fields": {
        "project": {"key": "PS"},
        "summary": "EJ - Dashboard Feedback",
        "issuetype": {"name": "Task"},
        "parent": {"key": "PS-4512"},
        "priority": {"id": "3"},
        "duedate": "2025-12-10",
        "customfield_10201": {"id": "10876"},  # Billable (default)
        "customfield_10290": {"value": "earthjustice.org"},
        "description": {adf_doc}
    }
})
```

### Add Comment:
```python
response = jira._make_request('POST', f'issue/{ticket_key}/comment', data={
    "body": {adf_comment}
})
```
