#!/usr/bin/env python3
"""
Execute synthesis results - Task List Contract Pattern

Created by: Claude Code
Date: 2025-12-03
Updated: 2025-12-04 (v2.7.0 - Task List Contract)
Purpose: Execute Phase 3 with explicit task tracking and validation

v2.7.0 Key Changes:
- Task List Contract: Explicit list of all tasks before execution
- Validation Gates: Each task validated before marked complete
- Progress Tracking: Clear visibility into what's done/pending
- Resume Capability: Can continue from last completed task
"""

import json
import sys
import os
from pathlib import Path
from datetime import datetime
from typing import Dict, List, Optional

# Resolve paths relative to this file (portable - no hardcoded paths!)
SKILL_DIR = Path(__file__).parent
sys.path.insert(0, str(SKILL_DIR))  # For lib imports

# Use centralized base module for project root and env
from lib.base import PROJECT_ROOT, ensure_env
project_root = PROJECT_ROOT  # Alias for backward compatibility
ensure_env()  # Load .env from project root

# Import lib modules
from lib.jira_executor import JiraExecutor, NewTicketRequest, CommentRequest
from lib.notion_publisher import NotionPublisher, ActionItemsDoc
from lib.task_list import (
    TaskList, Task, TaskType, TaskStatus,
    TaskListGenerator, TaskListRenderer, TaskListPersistence,
    ValidationGate
)


def _get_session_id() -> Optional[str]:
    """
    Get current Claude Code session ID for traceability.
    """
    session_id = os.environ.get('CLAUDE_SESSION_ID')
    if session_id:
        return session_id

    try:
        project_hash = str(PROJECT_ROOT).replace('/', '-').lstrip('-')
        sessions_dir = Path.home() / '.claude' / 'projects' / project_hash

        if sessions_dir.exists():
            session_files = list(sessions_dir.glob('*.jsonl'))
            if session_files:
                latest = max(session_files, key=lambda p: p.stat().st_mtime)
                return latest.stem
    except Exception:
        pass

    return None


class TaskExecutor:
    """
    Execute tasks from the task list with validation.

    This is the core of the Task List Contract pattern:
    1. Get next pending task (respecting dependencies)
    2. Execute the task
    3. Validate the result
    4. Mark task complete or failed
    5. Repeat until all tasks done
    """

    def __init__(
        self,
        task_list: TaskList,
        synthesis: Dict,
        context: Dict,
        epic_key: str,
        dry_run: bool = False
    ):
        self.task_list = task_list
        self.synthesis = synthesis
        self.context = context
        self.epic_key = epic_key
        self.dry_run = dry_run

        # Shared state across tasks
        self.notion_url: Optional[str] = None
        self.notion_page_id: Optional[str] = None
        self.created_tickets: List[Dict] = []
        self.updated_tickets: List[str] = []
        self.errors: List[str] = []

        # Executors (lazy init)
        self._publisher: Optional[NotionPublisher] = None
        self._jira_executor: Optional[JiraExecutor] = None

        # Context data
        self.gong_url = synthesis.get('gong_url', context.get('gong_url', ''))
        self.call_date = synthesis.get('call_date', datetime.now().strftime('%Y-%m-%d'))
        self.client_name = context.get('client', {}).get('name', 'Unknown Client')
        self.agency_id = context.get('client', {}).get('agency_id')

        # Smart Assignee: from context (populated by main.py from Gong call participants)
        self.default_assignee_id = context.get('default_assignee_id')
        self.reporter_id = context.get('reporter_id')

    @property
    def publisher(self) -> NotionPublisher:
        if self._publisher is None:
            self._publisher = NotionPublisher()
        return self._publisher

    @property
    def jira_executor(self) -> JiraExecutor:
        if self._jira_executor is None:
            self._jira_executor = JiraExecutor(
                epic_key=self.epic_key,
                default_assignee_id=self.default_assignee_id,
                reporter_id=self.reporter_id
            )
        return self._jira_executor

    def execute_all(self) -> Dict:
        """
        Execute all tasks in the task list.

        Returns:
            Results dictionary with created_tickets, updated_tickets, notion_page, errors
        """
        print(TaskListRenderer.render(self.task_list, show_details=False))
        print()

        while not self.task_list.is_complete():
            task = self.task_list.get_next_pending()

            if task is None:
                # No more tasks can be executed
                pending = self.task_list.get_pending_count()
                if pending > 0:
                    print(f"\n⚠️ DEADLOCK: {pending} tasks pending but dependencies not met")
                    for t in self.task_list.tasks:
                        if t.status == TaskStatus.PENDING:
                            print(f"   - [{t.id}] {t.description} (depends on: {t.depends_on})")
                break

            # Execute the task
            self._execute_task(task)

            # Update stats
            self.task_list.update_stats()

            # Show mini progress
            completed = self.task_list.completed_tasks
            total = self.task_list.total_tasks
            print(f"   Progress: {completed}/{total} tasks complete")

        # Final summary
        print()
        print(TaskListRenderer.render(self.task_list, show_details=False))

        return {
            'created_tickets': self.created_tickets,
            'updated_tickets': self.updated_tickets,
            'notion_page': {
                'id': self.notion_page_id,
                'url': self.notion_url,
                'title': f"{self.client_name} - Working Session"
            } if self.notion_page_id else None,
            'errors': self.errors,
            'task_list_complete': self.task_list.is_complete(),
            'tasks_completed': self.task_list.completed_tasks,
            'tasks_failed': self.task_list.failed_tasks
        }

    def _execute_task(self, task: Task):
        """Execute a single task"""
        print(f"\n{'─' * 40}")
        print(f"🔄 Executing: [{task.id}] {task.description}")
        print(f"{'─' * 40}")

        task.start()

        try:
            if self.dry_run:
                output = self._simulate_task(task)
            else:
                output = self._run_task(task)

            # Validate
            passed, message = ValidationGate.validate(task)

            if passed:
                task.complete(output)
                print(f"   ✅ Completed: {message}")
            else:
                task.fail(f"Validation failed: {message}")
                print(f"   ❌ Failed validation: {message}")
                self.errors.append(f"Task {task.id}: {message}")

        except Exception as e:
            task.fail(str(e))
            print(f"   ❌ Exception: {e}")
            self.errors.append(f"Task {task.id}: {str(e)}")

    def _run_task(self, task: Task) -> Dict:
        """Actually execute the task"""

        if task.type == TaskType.CREATE_NOTION_PAGE:
            return self._create_notion_page(task)

        elif task.type == TaskType.CREATE_JIRA_TICKET:
            return self._create_jira_ticket(task)

        elif task.type == TaskType.ADD_JIRA_COMMENT:
            return self._add_jira_comment(task)

        elif task.type == TaskType.COMPLETE_NOTION_PAGE:
            return self._complete_notion_page(task)

        else:
            raise ValueError(f"Unknown task type: {task.type}")

    def _simulate_task(self, task: Task) -> Dict:
        """Simulate task execution for dry run"""
        print(f"   [DRY RUN] Would execute: {task.type.value}")

        if task.type == TaskType.CREATE_NOTION_PAGE:
            self.notion_url = "https://www.notion.so/DRY_RUN_PAGE"
            self.notion_page_id = "dry_run_page_id"
            return {"notion_url": self.notion_url, "notion_page_id": self.notion_page_id}

        elif task.type == TaskType.CREATE_JIRA_TICKET:
            ticket_key = f"PS-DRY{task.id}"
            self.created_tickets.append({
                'key': ticket_key,
                'url': f"https://jira.com/{ticket_key}",
                'title': task.metadata.get('ticket_title', 'Unknown')
            })
            return {"ticket_key": ticket_key, "ticket_url": f"https://jira.com/{ticket_key}"}

        elif task.type == TaskType.ADD_JIRA_COMMENT:
            self.updated_tickets.append(task.metadata.get('ticket_key', 'PS-????'))
            return {"success": True}

        elif task.type == TaskType.COMPLETE_NOTION_PAGE:
            return {"blocks_added": 42}

        return {}

    def _create_notion_page(self, task: Task) -> Dict:
        """Create empty Notion page"""
        minimal_doc = ActionItemsDoc(
            title=f"{self.client_name} - Working Session",
            call_date=datetime.strptime(self.call_date, '%Y-%m-%d'),
            gong_url=self.gong_url,
            participants=[],
            summary="",
            new_tickets=[],
            updates=[],
            next_steps=[],
            client_name=self.client_name
        )

        page = self.publisher.create_event_page(minimal_doc, self.client_name, self.agency_id)
        self.notion_url = page.url
        self.notion_page_id = page.page_id

        print(f"   Created: {page.url}")

        task.output = {"notion_url": self.notion_url, "notion_page_id": self.notion_page_id}
        return task.output

    def _create_jira_ticket(self, task: Task) -> Dict:
        """Create Jira ticket"""
        ticket_index = task.metadata.get('ticket_index', 0)
        ticket_data = self.synthesis.get('new_tickets', [])[ticket_index]

        # Normalize priority
        priority_raw = ticket_data.get('priority', 'Medium')
        priority_map = {
            'high': 'Medium',
            'critical': 'Critical',
            'blocker': 'Critical',
            'medium': 'Medium',
            'low': 'Low'
        }
        priority = priority_map.get(priority_raw.lower(), 'Medium')

        req = NewTicketRequest(
            title=ticket_data['title'],
            description=ticket_data.get('description', ''),
            priority=priority,
            assignee_name=None,
            gong_url=self.gong_url,
            gong_timestamp=ticket_data.get('timestamp', 0),
            timestamp_display=ticket_data.get('timestamp_display', '00:00'),
            quote=ticket_data.get('quote'),
            action_items=ticket_data.get('action_items', [])
        )

        result = self.jira_executor.create_ticket(req, notion_url=self.notion_url)

        if result.success:
            self.created_tickets.append({
                'key': result.ticket_key,
                'url': result.url,
                'title': ticket_data['title']
            })
            print(f"   Created: {result.ticket_key}")
            if self.notion_url:
                print(f"   📎 Linked to Notion")

            return {"ticket_key": result.ticket_key, "ticket_url": result.url}
        else:
            raise Exception(result.error)

    def _add_jira_comment(self, task: Task) -> Dict:
        """Add comment to existing Jira ticket"""
        update_index = task.metadata.get('update_index', 0)
        update_data = self.synthesis.get('updates', [])[update_index]

        req = CommentRequest(
            ticket_key=update_data['ticket_key'],
            gong_url=self.gong_url,
            gong_timestamp=update_data.get('timestamp', 0),
            timestamp_display=update_data.get('timestamp_display', ''),
            updates=update_data.get('items', []),
            notion_url=self.notion_url
        )

        result = self.jira_executor.add_comment(req)

        if result.success:
            self.updated_tickets.append(update_data['ticket_key'])
            print(f"   Commented on: {update_data['ticket_key']}")
            return {"success": True}
        else:
            raise Exception(result.error)

    def _complete_notion_page(self, task: Task) -> Dict:
        """Complete Notion page with all content"""
        summary = self.synthesis.get('summary', '')
        tickets_to_show = self.synthesis.get('new_tickets', [])

        full_doc = ActionItemsDoc(
            title=f"{self.client_name} - Working Session",
            call_date=datetime.strptime(self.call_date, '%Y-%m-%d'),
            gong_url=self.gong_url,
            participants=self.synthesis.get('participants', []),
            summary=summary,
            new_tickets=[{
                'title': t['title'],
                'description': t.get('description', ''),
                'priority': t.get('priority', 'Medium'),
                'timestamp': t.get('timestamp', 0),
                'timestamp_display': t.get('timestamp_display', ''),
                'quote': t.get('quote', ''),
                'jira_key': next(
                    (r['key'] for r in self.created_tickets
                     if r['title'].lower().strip() == t['title'].lower().strip()),
                    ''
                ),
                'jira_url': next(
                    (r['url'] for r in self.created_tickets
                     if r['title'].lower().strip() == t['title'].lower().strip()),
                    ''
                )
            } for t in tickets_to_show],
            updates=[{
                'ticket_key': u['ticket_key'],
                'updates': u.get('items', []),
                'timestamp': u.get('timestamp', 0),
                'timestamp_display': u.get('timestamp_display', '')
            } for u in self.synthesis.get('updates', [])],
            next_steps=self.synthesis.get('next_steps', []),
            client_name=self.client_name,
            duration_minutes=self.synthesis.get('duration_minutes'),
            session_id=_get_session_id()
        )

        blocks_added = self.publisher.add_content_blocks(self.notion_page_id, full_doc)

        print(f"   Added {blocks_added} blocks")
        print(f"   📎 {len(self.created_tickets)} tickets linked")

        return {"blocks_added": blocks_added}


def execute_synthesis(
    synthesis_path: str,
    context_path: str,
    epic_key: str,
    dry_run: bool = False
) -> Dict:
    """
    Execute synthesis results using Task List Contract pattern.

    Args:
        synthesis_path: Path to 04_SYNTHESIS.json
        context_path: Path to context.json
        epic_key: Jira Epic key (e.g., PS-3286)
        dry_run: If True, only simulate execution

    Returns:
        Dict with created tickets, Notion page info, and task list status
    """
    # Load files
    synthesis = json.loads(Path(synthesis_path).read_text())
    context = json.loads(Path(context_path).read_text())

    call_id = Path(synthesis_path).parent.name

    # Show header
    print("═" * 60)
    print("🚀 EXECUTING SYNTHESIS (v2.7.0 - Task List Contract)")
    print("═" * 60)
    print(f"Epic: {epic_key}")
    print(f"Call ID: {call_id}")
    print(f"New tickets: {len(synthesis.get('new_tickets', []))}")
    print(f"Updates: {len(synthesis.get('updates', []))}")
    print()

    if dry_run:
        print("🔸 DRY RUN MODE - Simulating execution")
        print()

    # ═══════════════════════════════════════════════════════════════
    # GENERATE TASK LIST CONTRACT
    # ═══════════════════════════════════════════════════════════════
    print("📋 Generating Task List Contract...")
    task_list = TaskListGenerator.from_synthesis(synthesis, call_id)
    print(f"   Generated {task_list.total_tasks} tasks")

    # Save task list
    task_list_path = Path(synthesis_path).parent / '06_TASK_LIST.json'
    TaskListPersistence.save(task_list, task_list_path)
    print(f"   Saved to: {task_list_path.name}")

    # ═══════════════════════════════════════════════════════════════
    # EXECUTE ALL TASKS
    # ═══════════════════════════════════════════════════════════════
    executor = TaskExecutor(
        task_list=task_list,
        synthesis=synthesis,
        context=context,
        epic_key=epic_key,
        dry_run=dry_run
    )

    results = executor.execute_all()

    # Save updated task list
    TaskListPersistence.save(task_list, task_list_path)

    # ═══════════════════════════════════════════════════════════════
    # FINAL SUMMARY
    # ═══════════════════════════════════════════════════════════════
    print()
    print("═" * 60)
    print("📊 EXECUTION SUMMARY")
    print("═" * 60)
    print(f"   Tasks completed: {results['tasks_completed']}/{task_list.total_tasks}")
    print(f"   Tasks failed: {results['tasks_failed']}")
    print(f"   ✅ Created tickets: {len(results['created_tickets'])}")
    print(f"   ✅ Updated tickets: {len(results['updated_tickets'])}")
    print(f"   ✅ Notion page: {'Yes' if results['notion_page'] else 'No'}")
    print(f"   ❌ Errors: {len(results['errors'])}")

    if results['task_list_complete']:
        print()
        print("🎉 ALL TASKS COMPLETED SUCCESSFULLY!")
    else:
        print()
        print("⚠️ INCOMPLETE - Some tasks did not finish")

    return results


if __name__ == "__main__":
    import argparse

    parser = argparse.ArgumentParser(description='Execute synthesis with Task List Contract')
    parser.add_argument('synthesis', help='Path to 04_SYNTHESIS.json')
    parser.add_argument('--context', required=True, help='Path to context.json')
    parser.add_argument('--epic', required=True, help='Jira Epic key')
    parser.add_argument('--dry-run', action='store_true', help='Simulate execution')

    args = parser.parse_args()

    results = execute_synthesis(
        args.synthesis,
        args.context,
        args.epic,
        args.dry_run
    )

    # Save results
    output_path = Path(args.synthesis).parent / '05_EXECUTION_RESULTS.json'

    # Merge with previous results if exists
    if output_path.exists():
        prev_results = json.loads(output_path.read_text())
        results['created_tickets'] = prev_results.get('created_tickets', []) + results['created_tickets']
        results['updated_tickets'] = list(set(prev_results.get('updated_tickets', []) + results['updated_tickets']))
        if not results['notion_page'] and prev_results.get('notion_page'):
            results['notion_page'] = prev_results['notion_page']
        results['errors'] = prev_results.get('errors', []) + results['errors']

    output_path.write_text(json.dumps(results, indent=2, default=str))
    print(f"\n💾 Results saved: {output_path}")
