#!/usr/bin/env python3
"""
Gong-to-Jira/Notion Automation - Main Orchestrator

Created by: Claude Code
Date: 2025-12-03
Purpose: Orchestrate the full workflow: Gong → Analysis → Jira → Notion

Usage:
    python main.py <gong_url>
    python main.py --resume [call_id]
"""

import os
import sys
import json
import argparse
from pathlib import Path
from datetime import datetime
from typing import Optional, Dict, List

# Resolve paths relative to this file (portable - no hardcoded paths!)
SKILL_DIR = Path(__file__).parent
sys.path.insert(0, str(SKILL_DIR))  # For lib imports

# Use centralized base module for project root and env
from lib.base import PROJECT_ROOT, ensure_env
project_root = PROJECT_ROOT  # Alias for backward compatibility
ensure_env()  # Load .env from project root

# Import lib modules
from lib.gong_fetcher import GongFetcher
from lib.agency_discovery import AgencyDiscovery, ClientNotFoundError, ClientContext
from lib.jira_sync import JiraSync
from lib.jira_executor import JiraExecutor, NewTicketRequest, CommentRequest
from lib.notion_publisher import NotionPublisher, ActionItemsDoc
from lib.progress_tracker import ProgressTracker, SkillState


class GongToJiraNotionSkill:
    """
    Main orchestrator for Gong → Jira/Notion automation.

    Hybrid architecture:
    - Python (this file): API calls, file I/O, orchestration
    - Claude Code Agent: Analysis, synthesis (invoked separately)

    This orchestrator handles:
    1. INGEST: Fetch Gong, discover client, sync Jira
    2. ANALYZE: Prepare context for agent (agent does actual analysis)
    3. EXECUTE: Create/update Jira tickets
    4. PUBLISH: Create Notion event page
    """

    def __init__(self, gong_url: str, resume: bool = False, agency_id: int = None):
        """
        Initialize skill.

        Args:
            gong_url: Gong call URL
            resume: Whether to resume from checkpoint
            agency_id: Manual agency ID (bypasses auto-discovery)
        """
        self.gong_url = gong_url
        self.agency_id = agency_id
        self.gong_fetcher = GongFetcher()
        self.agency_discovery = AgencyDiscovery()

        # Extract call ID
        self.call_id = self.gong_fetcher.extract_call_id(gong_url)
        if not self.call_id:
            raise ValueError(f"Cannot extract call ID from URL: {gong_url}")

        # These will be set after discovery
        self.context: Optional[ClientContext] = None
        self.tracker: Optional[ProgressTracker] = None
        self.jira_sync: Optional[JiraSync] = None
        self.jira_executor: Optional[JiraExecutor] = None
        self.notion_publisher: Optional[NotionPublisher] = None

    def _init_tracker(self, working_dir: str):
        """Initialize progress tracker"""
        self.tracker = ProgressTracker(working_dir, self.call_id, self.gong_url)

    def _get_session_id(self) -> Optional[str]:
        """
        Get Claude Code session ID for traceability.

        Tries multiple sources:
        1. CLAUDE_SESSION_ID env var (set by Claude Code)
        2. Most recent session file in ~/.claude/projects/

        Note: Session path is derived from PROJECT_ROOT for portability.
        """
        import glob

        # Try env var first
        if os.environ.get('CLAUDE_SESSION_ID'):
            return os.environ.get('CLAUDE_SESSION_ID')

        # Try to find from recent session files
        try:
            # Derive session folder name from PROJECT_ROOT (portable)
            # Claude Code uses: ~/.claude/projects/-{path_with_slashes_as_dashes}/
            project_path_encoded = str(PROJECT_ROOT).replace('/', '-')
            sessions_pattern = os.path.expanduser(
                f'~/.claude/projects/{project_path_encoded}/*.jsonl'
            )
            session_files = glob.glob(sessions_pattern)
            if session_files:
                # Get most recent file
                latest = max(session_files, key=os.path.getmtime)
                # Extract session ID from filename (UUID format)
                filename = os.path.basename(latest)
                if filename.endswith('.jsonl') and '-' in filename:
                    session_id = filename.replace('.jsonl', '')
                    # Validate UUID-like format
                    if len(session_id) == 36 and session_id.count('-') == 4:
                        return session_id
        except Exception:
            pass

        return None

    def _get_smart_reporter_id(self) -> Optional[str]:
        """
        Smart Reporter detection with cascading fallback.

        Priority:
        1. Epic's reporter (if Epic exists)
        2. SFDC Primary CSM → lookup Jira user by email
        3. Default: Ilia Kolesnikov

        Returns:
            Jira account ID for reporter
        """
        # Default: Ilia Kolesnikov's account ID
        ILIA_ACCOUNT_ID = "6165c9bcd9bb5e0069b8b212"  # user@improvado.io

        # Priority 1: Epic's reporter
        if self.context and self.context.defaults.get('reporter_id'):
            reporter_id = self.context.defaults['reporter_id']
            print(f"   📋 Reporter: from Epic")
            return reporter_id

        # Priority 2: SFDC Primary CSM email → Jira lookup
        if self.context and self.context.primary_csm_email:
            try:
                from lib.base import clients
                jira = clients.jira

                # Search by email
                user = jira.find_user_by_name(self.context.primary_csm_email)
                if user and user.get('accountId'):
                    print(f"   📋 Reporter: SFDC CSM ({self.context.primary_csm_email})")
                    return user['accountId']
            except Exception as e:
                print(f"   ⚠️ CSM lookup failed: {e}")

        # Priority 3: Default to Ilia
        print(f"   📋 Reporter: default (Ilia Kolesnikov)")
        return ILIA_ACCOUNT_ID

    def _get_smart_assignee_id(self) -> Optional[str]:
        """
        Smart Assignee detection from call participants.

        Priority:
        1. First Improvado call participant (from ClickHouse gong_call_improvado_emails_array)
        2. Epic's default assignee
        3. None (unassigned)

        Returns:
            Jira account ID for assignee
        """
        # Priority 1: First Improvado participant from call
        improvado_team = getattr(self, 'actual_improvado_team', [])
        if improvado_team:
            try:
                from lib.base import clients
                jira = clients.jira

                # Try first participant
                for email in improvado_team:
                    user = jira.find_user_by_name(email)
                    if user and user.get('accountId'):
                        print(f"   👤 Assignee: {user.get('displayName', email)} (call participant)")
                        return user['accountId']
            except Exception as e:
                print(f"   ⚠️ Assignee lookup failed: {e}")

        # Priority 2: Epic's default assignee
        if self.context and self.context.defaults.get('assignee_id'):
            print(f"   👤 Assignee: from Epic defaults")
            return self.context.defaults['assignee_id']

        # Priority 3: None
        print(f"   👤 Assignee: unassigned (no Improvado participants found)")
        return None

    def run_ingest_phase(self) -> Dict:
        """
        Phase 1: Ingest - Fetch Gong, discover client, sync Jira.

        Returns:
            Dictionary with ingest results
        """
        print("\n📥 INGEST PHASE")
        print("=" * 50)

        results = {}

        # Step 1.1: Fetch Gong transcript
        print("\n1.1 Fetching Gong transcript...")
        self.tracker.transition_to(SkillState.GONG_FETCH)
        self.tracker.start_checkpoint('GONG_FETCH')

        try:
            transcript = self.gong_fetcher.fetch_and_parse(self.gong_url)

            if not transcript:
                self.tracker.fail_checkpoint('GONG_FETCH', 'No transcript found')
                raise ValueError("No transcript found for this call")

            # Save transcript
            transcript_path = self.gong_fetcher.save_transcript(
                transcript,
                self.context.calls_folder
            )
            json_path = self.gong_fetcher.save_raw_json(
                transcript,
                self.context.processing_folder
            )

            # Filter invited participants by who actually spoke
            actual_speakers_lower = [s.lower() for s in transcript.speakers]

            # Filter improvado_team: only include those who actually spoke
            actual_improvado_team = []
            for email in transcript.metadata.improvado_team:
                # Extract name from email: user@improvado.io -> ilia kolesnikov
                email_name = email.split('@')[0].replace('.', ' ').replace('_', ' ').lower()
                name_parts = email_name.split()

                # Check if this person spoke (match any speaker name)
                spoke = False
                for speaker in actual_speakers_lower:
                    # Match if first name or full name matches
                    if name_parts[0] in speaker or email_name in speaker:
                        spoke = True
                        break
                    # Also check last name for "Kolesnikov" matching "Ilia Kolesnikov"
                    if len(name_parts) > 1 and name_parts[-1] in speaker:
                        spoke = True
                        break

                if spoke:
                    actual_improvado_team.append(email)

            results['transcript'] = {
                'path': transcript_path,
                'json_path': json_path,
                'title': transcript.metadata.title,
                'date': transcript.metadata.date.isoformat(),
                'duration': transcript.total_duration_display,
                'speakers': transcript.speakers,
                'segment_count': len(transcript.segments),
                'char_count': len(transcript.raw_text),
                # Store ACTUAL participants (filtered by who spoke)
                'improvado_team': actual_improvado_team,
                'client_team': transcript.metadata.client_team  # External always included
            }

            # Store on self for use in publish phase (with filtered team)
            self.transcript_metadata = transcript.metadata
            self.actual_improvado_team = actual_improvado_team  # Filtered by speakers

            self.tracker.complete_checkpoint('GONG_FETCH', {
                'chars': len(transcript.raw_text),
                'speakers': len(transcript.speakers),
                'file': transcript_path
            })
            self.tracker.transition_to(SkillState.GONG_DONE)

            print(f"   ✅ Transcript: {len(transcript.segments)} segments, {len(transcript.raw_text):,} chars")
            print(f"   ✅ Speakers: {', '.join(transcript.speakers)}")

        except Exception as e:
            self.tracker.fail_checkpoint('GONG_FETCH', str(e))
            raise

        # Step 1.2: Agency discovery (already done if context exists)
        print("\n1.2 Client context...")
        self.tracker.transition_to(SkillState.AGENCY_DISCOVER)
        self.tracker.start_checkpoint('AGENCY_DISCOVER')

        self.tracker.complete_checkpoint('AGENCY_DISCOVER', {
            'client_name': self.context.client_name,
            'agency_id': self.context.agency_id
        })
        self.tracker.transition_to(SkillState.AGENCY_DONE)

        print(f"   ✅ Client: {self.context.client_name} (ID: {self.context.agency_id})")
        print(f"   ✅ Epic: {self.context.epic_key}")

        results['client'] = {
            'name': self.context.client_name,
            'agency_id': self.context.agency_id,
            'folder': self.context.client_folder,
            'epic': self.context.epic_key
        }

        # Step 1.3: Sync Jira tickets
        print("\n1.3 Syncing Jira tickets...")
        self.tracker.transition_to(SkillState.JIRA_DISCOVER)
        self.tracker.start_checkpoint('JIRA_DISCOVER')

        try:
            if self.context.epic_key:
                sync_state = self.jira_sync.sync(self.context.epic_key)

                results['jira'] = {
                    'epic': sync_state.epic_key,
                    'total_tickets': sync_state.total_tickets,
                    'updated': sync_state.updated_count
                }

                self.tracker.complete_checkpoint('JIRA_DISCOVER', {
                    'epic': sync_state.epic_key,
                    'tickets': sync_state.total_tickets
                })

                print(f"   ✅ Synced: {sync_state.total_tickets} tickets")
                print(f"   ✅ Updated: {sync_state.updated_count} since last sync")
            else:
                results['jira'] = {'epic': None, 'total_tickets': 0}
                self.tracker.complete_checkpoint('JIRA_DISCOVER', {'epic': None, 'tickets': 0})
                print("   ⚠️ No Epic found - skipping Jira sync")

            self.tracker.transition_to(SkillState.JIRA_DONE)

        except Exception as e:
            self.tracker.fail_checkpoint('JIRA_DISCOVER', str(e))
            raise

        return results

    def prepare_analysis_context(self) -> str:
        """
        Prepare context file for Claude Code agent analysis.

        Creates a JSON file with all necessary context for the agent.

        Returns:
            Path to context file
        """
        # Get tickets for analysis
        tickets = self.jira_sync.get_tickets_for_analysis() if self.context.epic_key else []

        # Get smart assignee and reporter IDs for execute phase
        assignee_id = self._get_smart_assignee_id()
        reporter_id = self._get_smart_reporter_id()

        context = {
            'call_id': self.call_id,
            'gong_url': self.gong_url,
            'epic_key': self.context.epic_key,  # Top-level for easy access
            'default_assignee_id': assignee_id,  # CRITICAL: For execute phase Jira ticket assignment
            'reporter_id': reporter_id,  # CRITICAL: For execute phase Jira ticket reporter
            'client': {
                'name': self.context.client_name,
                'agency_id': self.context.agency_id,
                'epic_key': self.context.epic_key
            },
            'tickets': tickets,
            'files': {
                'transcript': os.path.join(self.context.processing_folder, f'{self.call_id}'),
                'analysis_output': os.path.join(self.context.processing_folder, f'{self.call_id}', 'analysis.json'),
                'action_items_output': os.path.join(self.context.processing_folder, f'{self.call_id}', 'ACTION_ITEMS.md')
            }
        }

        context_path = os.path.join(self.context.processing_folder, self.call_id, 'context.json')
        os.makedirs(os.path.dirname(context_path), exist_ok=True)

        with open(context_path, 'w') as f:
            json.dump(context, f, indent=2, default=str)

        return context_path

    def run_execute_phase(self, analysis_path: str, notion_url: Optional[str] = None) -> Dict:
        """
        Phase 4: Execute - Create/update Jira tickets based on analysis.

        Args:
            analysis_path: Path to analysis.json from agent
            notion_url: Optional Notion page URL for links

        Returns:
            Execution results
        """
        print("\n⚡ EXECUTE PHASE")
        print("=" * 50)

        self.tracker.transition_to(SkillState.JIRA_EXECUTE)
        self.tracker.start_checkpoint('JIRA_EXECUTE')

        # Load analysis
        with open(analysis_path, 'r') as f:
            analysis = json.load(f)

        # Prepare requests
        new_ticket_requests = []
        comment_requests = []

        for item in analysis.get('new_tickets', []):
            new_ticket_requests.append(NewTicketRequest(
                title=item.get('title', ''),
                description=item.get('description', ''),
                priority=item.get('priority', 'Medium'),
                assignee_name=item.get('assignee'),
                gong_url=self.gong_url,
                gong_timestamp=item.get('timestamp', 0),
                timestamp_display=item.get('timestamp_display', ''),
                quote=item.get('quote'),
                action_items=item.get('action_items', [])
            ))

        for item in analysis.get('updates', []):
            comment_requests.append(CommentRequest(
                ticket_key=item.get('ticket_key', ''),
                gong_url=self.gong_url,
                gong_timestamp=item.get('timestamp', 0),
                timestamp_display=item.get('timestamp_display', ''),
                updates=item.get('items', [])
            ))

        # Update tracker with totals
        self.tracker.update_jira_progress(
            new_tickets_total=len(new_ticket_requests),
            comments_total=len(comment_requests)
        )

        # Progress callback
        def on_progress(op_type, key, success):
            if op_type == 'created':
                if success:
                    self.tracker.update_jira_progress(new_ticket_created=key)
                    print(f"   ✅ Created: {key}")
                else:
                    print(f"   ❌ Failed to create: {key}")
            elif op_type == 'commented':
                if success:
                    self.tracker.update_jira_progress(comment_added=key)
                    print(f"   ✅ Commented: {key}")
                else:
                    print(f"   ❌ Failed to comment: {key}")

        # Get call date
        call_date = analysis.get('call_date', datetime.now().strftime('%b %d, %Y'))

        # Execute
        results = self.jira_executor.execute_batch(
            new_tickets=new_ticket_requests,
            comments=comment_requests,
            notion_url=notion_url,
            call_date=call_date,
            progress_callback=on_progress
        )

        self.tracker.complete_checkpoint('JIRA_EXECUTE', {
            'created': len(results['created']),
            'commented': len(results['commented']),
            'errors': len(results['errors'])
        })
        self.tracker.transition_to(SkillState.JIRA_EXECUTE_DONE)

        print(f"\n   Summary: {len(results['created'])} created, {len(results['commented'])} commented")

        if results['errors']:
            print(f"   ⚠️ Errors: {len(results['errors'])}")
            for err in results['errors']:
                print(f"      - {err['type']}: {err.get('title', err.get('ticket', '?'))}: {err['error']}")

        return results

    def run_publish_phase(self, action_items_path: str, jira_results: Dict) -> str:
        """
        Phase 5: Publish - Create Notion event page.

        Args:
            action_items_path: Path to ACTION_ITEMS.md from agent
            jira_results: Results from execute phase

        Returns:
            Notion page URL
        """
        print("\n📤 PUBLISH PHASE")
        print("=" * 50)

        self.tracker.transition_to(SkillState.NOTION_PUBLISH)
        self.tracker.start_checkpoint('NOTION_PUBLISH')

        # Load action items (simplified - in practice, parse the markdown)
        with open(action_items_path, 'r') as f:
            content = f.read()

        # Get Claude Code session ID for traceability
        session_id = os.environ.get('CLAUDE_SESSION_ID', self._get_session_id())

        # Create ActionItemsDoc with proper metadata
        # Get call date and participants from transcript metadata
        transcript_meta = getattr(self, 'transcript_metadata', None)
        actual_improvado = getattr(self, 'actual_improvado_team', [])  # Filtered by speakers

        if transcript_meta:
            actual_date = transcript_meta.date
            # Use FILTERED improvado_team (only those who spoke) + all client team
            participants = actual_improvado + transcript_meta.client_team
        else:
            actual_date = datetime.now()
            participants = []

        doc = ActionItemsDoc(
            title=f"{self.context.client_name} Working Session",
            call_date=actual_date,
            gong_url=self.gong_url,
            participants=participants,  # Now uses emails from ClickHouse
            summary="Call summary...",  # Should come from action items
            new_tickets=[
                {
                    'title': t['title'],
                    'jira_key': t['key'],
                    'timestamp': 0,
                    'timestamp_display': ''
                }
                for t in jira_results.get('created', [])
            ],
            updates=[
                {
                    'ticket_key': t['key'],
                    'ticket_summary': '',
                    'timestamp': 0,
                    'timestamp_display': '',
                    'items': []
                }
                for t in jira_results.get('commented', [])
            ],
            next_steps=[],
            session_id=session_id
        )

        # Publish
        page = self.notion_publisher.publish(doc, self.context.client_name)

        self.tracker.complete_checkpoint('NOTION_PUBLISH', {
            'page_id': page.page_id,
            'url': page.url
        })
        self.tracker.transition_to(SkillState.COMPLETE)

        print(f"\n   📓 Notion page: {page.url}")

        return page.url

    def run(self):
        """
        Run full automation workflow.

        Steps:
        1. Discover client from Gong title (requires pre-fetch of title)
        2. Initialize all components
        3. Run ingest phase
        4. Prepare context for agent (agent runs separately)
        5. After agent creates analysis.json, run execute phase
        6. Run publish phase
        """
        print("\n🎯 Gong → Jira/Notion Automation")
        print("━" * 60)
        print(f"Call: {self.gong_url}")
        print(f"Call ID: {self.call_id}")
        print("━" * 60)

        # Step 1: Quick fetch to get title for discovery
        print("\n🔍 Fetching call metadata...")
        raw_data = self.gong_fetcher.fetch_call_from_db(self.gong_url)

        if not raw_data:
            raise ValueError("Cannot fetch Gong call data")

        gong_title = raw_data['title']
        print(f"   Title: {gong_title}")

        # Step 2: Discover client
        print("\n🔍 Discovering client...")
        if self.agency_id:
            print(f"   Using manual agency_id: {self.agency_id}")
            self.context = self.agency_discovery.discover_from_agency_id(self.agency_id)
        else:
            self.context = self.agency_discovery.discover(gong_title)

        # Step 3: Initialize components
        # FIXED: gong_processing goes inside customer_communication/calls/, not client root
        calls_dir = os.path.join(self.context.client_folder, 'customer_communication', 'calls')
        self._init_tracker(os.path.join(calls_dir, 'gong_processing'))
        self.tracker.set_client_context(self.context.client_name, self.context.client_folder)

        self.jira_sync = JiraSync(self.context.jira_folder)
        self.notion_publisher = NotionPublisher(self.context.notion_events_db)

        # Step 4: Run ingest (this populates actual_improvado_team)
        ingest_results = self.run_ingest_phase()

        # Step 4b: Initialize JiraExecutor AFTER ingest (needs participant data for assignee)
        if self.context.epic_key:
            # Smart Reporter detection: Epic reporter → SFDC CSM → Ilia (default)
            reporter_id = self._get_smart_reporter_id()
            # Smart Assignee detection: Call participant → Epic default → None
            assignee_id = self._get_smart_assignee_id()

            self.jira_executor = JiraExecutor(
                epic_key=self.context.epic_key,
                jira_project=self.context.jira_project,
                engagement_type_id=self.context.defaults.get('engagement_type_id', '10876'),
                default_assignee_id=assignee_id,  # From call participants
                agency_domain_id=self.context.domain_id,  # For customfield_10290
                reporter_id=reporter_id
            )

        # Step 5: Prepare context for agent
        context_path = self.prepare_analysis_context()

        print("\n" + "=" * 60)
        print("📋 INGEST COMPLETE - READY FOR TWO-AGENT ANALYSIS!")
        print("=" * 60)
        print(f"\nContext: {context_path}")
        print("\n" + "─" * 60)
        print("🚀 RECOMMENDED: Run parallel Two-Agent analysis:")
        print("─" * 60)
        print(f"\n   python main.py --analyze {self.call_id}")
        print(f"\n   This runs Gemini + Claude IN PARALLEL, then synthesizes.")
        print(f"\n📌 Then execute:")
        print(f"   python main.py --execute {self.call_id}")
        print("\n" + "─" * 60)
        print("📋 Alternative: Manual steps (if parallel fails):")
        print("─" * 60)
        print("\n📌 Step 1: GEMINI ANALYSIS")
        print(f"   python lib/gemini_analyzer.py \\")
        print(f'     "{self.context.calls_folder}/*_TRANSCRIPT.md" \\')
        print(f'     --tickets "{self.context.jira_folder}/tickets" \\')
        print(f'     --gong-url "{self.gong_url}" \\')
        print(f'     --output "{self.context.processing_folder}/{self.call_id}/03_gemini_analysis.json"')
        print("\n📌 Step 2: CLAUDE ANALYSIS")
        print(f"   python lib/claude_analyzer.py \\")
        print(f'     "{self.context.calls_folder}/*_TRANSCRIPT.md" \\')
        print(f'     --tickets "{self.context.jira_folder}/tickets" \\')
        print(f'     --gong-url "{self.gong_url}" \\')
        print(f'     --output "{self.context.processing_folder}/{self.call_id}/03_claude_analysis.json"')
        print("\n📌 Step 3: SYNTHESIZE BOTH")
        print(f"   python lib/synthesis.py \\")
        print(f'     --gemini "{self.context.processing_folder}/{self.call_id}/03_gemini_analysis.json" \\')
        print(f'     --claude "{self.context.processing_folder}/{self.call_id}/03_claude_analysis.json" \\')
        print(f'     --output "{self.context.processing_folder}/{self.call_id}/04_SYNTHESIS.json"')

        return {
            'status': 'ingest_complete',
            'call_id': self.call_id,
            'context_path': context_path,
            'ingest_results': ingest_results
        }

    def run_post_analysis(self):
        """
        Run execute and publish phases after agent analysis.

        Assumes analysis.json and ACTION_ITEMS.md exist.
        """
        # Load context
        context_path = os.path.join(
            self.context.processing_folder,
            self.call_id,
            'context.json'
        )

        with open(context_path, 'r') as f:
            context_data = json.load(f)

        analysis_path = os.path.join(
            self.context.processing_folder,
            self.call_id,
            'analysis.json'
        )
        action_items_path = os.path.join(
            self.context.processing_folder,
            self.call_id,
            'ACTION_ITEMS.md'
        )

        if not os.path.exists(analysis_path):
            raise FileNotFoundError(f"Analysis not found: {analysis_path}")

        # Run execute
        jira_results = self.run_execute_phase(analysis_path)

        # Run publish
        if os.path.exists(action_items_path):
            notion_url = self.run_publish_phase(action_items_path, jira_results)
        else:
            print("⚠️ ACTION_ITEMS.md not found, skipping Notion publish")
            notion_url = None

        # Final progress
        print("\n" + self.tracker.render_progress())

        return {
            'status': 'complete',
            'jira': jira_results,
            'notion_url': notion_url
        }


def main():
    parser = argparse.ArgumentParser(
        description='Gong → Jira/Notion Automation Skill'
    )
    parser.add_argument('url', nargs='?', help='Gong call URL')
    parser.add_argument('--resume', metavar='CALL_ID', nargs='?', const=True,
                       help='Resume from checkpoint (optionally specify call ID)')
    parser.add_argument('--execute', metavar='CALL_ID',
                       help='Run execute/publish phases for completed analysis')
    parser.add_argument('--agency-id', type=int, metavar='ID',
                       help='Manual agency ID (bypasses auto-discovery from title)')
    parser.add_argument('--mode', choices=['auto', 'approve'], default='auto',
                       help='Execution mode: auto (run to completion) or approve (show findings, wait for confirmation)')
    parser.add_argument('--analyze', metavar='CALL_ID',
                       help='Run parallel Two-Agent analysis (Gemini + Claude) for a call')

    args = parser.parse_args()

    # ═══════════════════════════════════════════════════════════════
    # ANALYZE MODE: Run Gemini + Claude in parallel
    # ═══════════════════════════════════════════════════════════════
    if args.analyze:
        call_id = args.analyze

        print(f"═" * 60)
        print(f"🔬 PARALLEL TWO-AGENT ANALYSIS for call: {call_id}")
        print(f"═" * 60)

        # Find gong_processing folder with this call_id
        import glob
        pattern = str(PROJECT_ROOT / 'client_cases' / '*' / 'customer_communication' / 'calls' / 'gong_processing' / call_id)
        matches = glob.glob(pattern)

        if not matches:
            print(f"❌ No processing folder found for call_id: {call_id}")
            print(f"   Searched: {pattern}")
            return

        processing_folder = Path(matches[0])
        print(f"📁 Found: {processing_folder}")

        # Find transcript
        calls_folder = processing_folder.parent.parent
        transcript_files = list(calls_folder.glob("*_TRANSCRIPT.md"))

        if not transcript_files:
            print(f"❌ No transcript found in: {calls_folder}")
            return

        transcript_path = transcript_files[0]
        print(f"📄 Transcript: {transcript_path.name}")

        # Find tickets folder
        client_folder = calls_folder.parent.parent
        tickets_folder = client_folder / 'jira_tickets' / 'tickets'

        if not tickets_folder.exists():
            print(f"❌ Tickets folder not found: {tickets_folder}")
            return

        print(f"🎫 Tickets: {tickets_folder}")

        # Load context.json for gong_url
        context_path = processing_folder / 'context.json'
        if not context_path.exists():
            print(f"❌ context.json not found")
            return

        with open(context_path) as f:
            context = json.load(f)

        gong_url = context.get('gong_url', '')
        print(f"🔗 Gong URL: {gong_url[:50]}...")

        # Run parallel analysis
        from lib.parallel_analyzer import run_parallel_analysis

        results = run_parallel_analysis(
            transcript_path=str(transcript_path),
            tickets_folder=str(tickets_folder),
            gong_url=gong_url,
            output_dir=str(processing_folder)
        )

        if results.get('synthesis'):
            print(f"\n✅ Ready for execution!")
            print(f"   Run: python main.py --execute {call_id}")
        else:
            print(f"\n⚠️ Analysis incomplete - check errors above")

        return

    if args.execute:
        # Run post-analysis phases (Phase 3: Execute)
        call_id = args.execute
        mode = args.mode

        print(f"═" * 60)
        print(f"🚀 EXECUTING Phase 3 for call: {call_id}")
        print(f"   Mode: {mode.upper()}")
        print(f"═" * 60)

        # Find gong_processing folder with this call_id
        import glob
        pattern = str(PROJECT_ROOT / 'client_cases' / '*' / 'customer_communication' / 'calls' / 'gong_processing' / call_id)
        matches = glob.glob(pattern)

        if not matches:
            print(f"❌ No processing folder found for call_id: {call_id}")
            print(f"   Searched: {pattern}")
            return

        processing_folder = Path(matches[0])
        print(f"📁 Found: {processing_folder}")

        # Load context.json
        context_path = processing_folder / 'context.json'
        if not context_path.exists():
            print(f"❌ context.json not found in {processing_folder}")
            return

        with open(context_path) as f:
            context = json.load(f)

        epic_key = context.get('epic_key')
        if not epic_key:
            print(f"❌ No epic_key in context.json")
            return

        print(f"📋 Epic: {epic_key}")

        # Find synthesis file
        synthesis_path = processing_folder / '04_SYNTHESIS.json'
        if not synthesis_path.exists():
            print(f"❌ 04_SYNTHESIS.json not found")
            print(f"   Run Two-Agent Analysis first (Gemini + Claude → Synthesis)")
            return

        print(f"📄 Synthesis: {synthesis_path}")

        # ═══════════════════════════════════════════════════════════════
        # APPROVE MODE: Show findings and wait for confirmation
        # ═══════════════════════════════════════════════════════════════
        if mode == 'approve':
            with open(synthesis_path) as f:
                synthesis = json.load(f)

            # Try to load individual agent analyses
            gemini_path = processing_folder / '03_gemini_analysis.json'
            claude_path = processing_folder / '03_claude_analysis.json'

            print(f"\n{'─' * 60}")
            print(f"📊 TWO-AGENT ANALYSIS RESULTS")
            print(f"{'─' * 60}")

            # Show Gemini findings
            if gemini_path.exists():
                with open(gemini_path) as f:
                    gemini = json.load(f)
                print(f"\n🤖 GEMINI found {len(gemini.get('new_tickets', []))} new tickets:")
                for i, t in enumerate(gemini.get('new_tickets', [])[:5], 1):
                    print(f"   {i}. {t.get('title', 'No title')[:60]}")
                if len(gemini.get('new_tickets', [])) > 5:
                    print(f"   ... and {len(gemini.get('new_tickets', [])) - 5} more")
            else:
                print(f"\n⚠️ Gemini analysis not found")

            # Show Claude findings
            if claude_path.exists():
                with open(claude_path) as f:
                    claude = json.load(f)
                print(f"\n🧠 CLAUDE found {len(claude.get('new_tickets', []))} new tickets:")
                for i, t in enumerate(claude.get('new_tickets', [])[:5], 1):
                    print(f"   {i}. {t.get('title', 'No title')[:60]}")
                if len(claude.get('new_tickets', [])) > 5:
                    print(f"   ... and {len(claude.get('new_tickets', [])) - 5} more")
            else:
                print(f"\n⚠️ Claude analysis not found")

            # Show synthesis results
            print(f"\n{'─' * 60}")
            print(f"🔀 SYNTHESIS RESULT (merged & deduplicated):")
            print(f"{'─' * 60}")

            new_tickets = synthesis.get('new_tickets', [])
            updates = synthesis.get('updates', [])
            next_steps = synthesis.get('next_steps', [])

            print(f"\n📋 NEW TICKETS TO CREATE ({len(new_tickets)}):")
            for i, t in enumerate(new_tickets, 1):
                conf = t.get('confidence', {})
                conf_level = conf.get('level', '?')
                conf_emoji = '🟢' if conf_level == 'HIGH' else '🟡' if conf_level == 'MEDIUM' else '🔴'
                print(f"   {i}. {conf_emoji} [{conf_level}] {t.get('title', 'No title')[:55]}")

            print(f"\n📝 UPDATES TO EXISTING ({len(updates)}):")
            for u in updates[:5]:
                print(f"   • {u.get('ticket_key', '?')}: {len(u.get('items', []))} updates")

            print(f"\n📌 NEXT STEPS ({len(next_steps)}):")
            for s in next_steps[:3]:
                print(f"   • {s.get('owner', '?')}: {s.get('task', '?')[:50]}")

            # Ask for confirmation
            print(f"\n{'═' * 60}")
            print(f"⚠️  APPROVE MODE: Review the above findings")
            print(f"{'═' * 60}")
            print(f"")
            print(f"   This will:")
            print(f"   • Create {len(new_tickets)} new Jira tickets")
            print(f"   • Add comments to {len(updates)} existing tickets")
            print(f"   • Create Notion meeting notes page")
            print(f"")

            try:
                response = input("   Proceed? [y/N]: ").strip().lower()
            except EOFError:
                response = 'n'

            if response not in ['y', 'yes']:
                print(f"\n❌ Execution cancelled by user")
                print(f"   To re-run: python main.py --execute {call_id} --mode approve")
                return

            print(f"\n✅ Approved! Proceeding with execution...")

        # ═══════════════════════════════════════════════════════════════
        # AUTO MODE: Just run everything
        # ═══════════════════════════════════════════════════════════════

        # Import and run execute_synthesis
        from execute_synthesis import execute_synthesis
        results = execute_synthesis(
            str(synthesis_path),
            str(context_path),
            epic_key,
            dry_run=False
        )

        # Save results
        output_path = processing_folder / '05_EXECUTION_RESULTS.json'
        with open(output_path, 'w') as f:
            json.dump(results, f, indent=2, default=str)

        print(f"\n{'═' * 60}")
        print(f"✅ EXECUTION COMPLETE")
        print(f"{'═' * 60}")
        print(f"   Created tickets: {len(results.get('created_tickets', []))}")
        print(f"   Updated tickets: {len(results.get('updated_tickets', []))}")
        print(f"   Notion page: {'Yes' if results.get('notion_page') else 'No'}")
        print(f"   Errors: {len(results.get('errors', []))}")
        print(f"\n💾 Results: {output_path}")
        return

    if args.resume:
        # Resume from checkpoint
        print("Resuming from checkpoint...")
        # Implementation would find incomplete progress and continue
        print("Not yet implemented - use full workflow")
        return

    if not args.url:
        parser.print_help()
        return

    # Full workflow
    skill = GongToJiraNotionSkill(args.url, agency_id=getattr(args, 'agency_id', None))
    result = skill.run()

    print(f"\n✅ Result: {result['status']}")


if __name__ == "__main__":
    main()
