---
name: graph-find
description: "Parallel multi-algorithm Obsidian vault search. Without arguments — auto-summarizes current session and maps it to vault nodes. With query — runs 3 search methods in parallel (LLM semantic, MCP text+dataview, filesystem), deduplicates & ranks results. Use when user says: 'find in vault', 'what do we have about [topic]', 'найди в обсидиане', 'search obsidian', 'vault search', 'найди контекст', 'graph find', 'map session'."
version: "2.0.0"
---

# /graph_find — Parallel Multi-Algorithm Vault Search

**Thesis:** Without a query — auto-summarize the current session into a blended query and map it to vault nodes. With a query — run 3 independent search methods in parallel, merge & deduplicate results, compute composite relevance scores, and present a ranked list with confidence classification.

## Execution Protocol

When the user invokes `/graph_find <query>` or `/graph_find` (no arguments):

### Step 0: Query Resolution (if no query provided)

If no query argument was passed, **auto-generate a session summary** to use as the search query.

#### 0a. Analyze Current Session

Review the full conversation history and identify:
- **All distinct tasks/topics** worked on (there may be multiple!)
- **Key entities** mentioned: customer names, product names, technologies, people
- **Artifacts created**: files, skills, configs, documents
- **Domain areas**: which parts of the vault this touches (Improvado, Miras, Daniel Personal, etc.)

#### 0b. Generate Blended Summary Query

Write a single blended query (1-3 sentences) that covers ALL session aspects. This is used as ONE search — not split per topic.

**Format:**
```
[Main topic/action]. Related: [entity1], [entity2], [entity3]. Areas: [domain1], [domain2].
```

**Example:**
```
Obsidian vault skills for graph search and content addition. Related: graph-find, graph-add, suggest_placement, CLAUDE.md. Areas: Claude Code Infrastructure, Obsidian Knowledge Base.
```

**Why blended, not per-topic:** Running N separate searches is slow. A single well-composed blended query captures enough keywords for each method to find relevant hits — LLM catches the semantic meaning, MCP catches exact terms, Grep catches keywords.

#### 0c. Present Session Mapping

After searching (Steps 1-4), present the results as a **session-to-vault mapping**:

```
SESSION MAPPING

Summary: [blended query used]

BEST MATCHES:
1. [0.89] Data sources and vendors/Claude code/ — Sources: LLM, MCP
2. [0.72] Miras Knowledge Platform/ — Sources: LLM

RECOMMENDED related_to LINKS:
  - "[[Claude code Skills base.base]]"
  - "[[Miras Knowledge Platform]]"
```

---

### Step 1: Launch All 3 Search Methods in Parallel

Use the Task tool or direct tool calls to run ALL methods simultaneously. Do NOT run them sequentially.

#### Method 1: LLM Semantic (suggest_placement)

```bash
$PROJECT_ROOT/claude_venv/bin/python -c "
import sys; sys.path.insert(0, '$PROJECT_ROOT')
from data_sources.obsidian.ops import suggest_placement
results = suggest_placement('QUERY_HERE')
for r in results:
    print(f'{r.score:.2f}|{r.path}|{r.reason}')
"
```

- **Score range:** 0.6–1.0 (pre-filtered by suggest_placement)
- **Strength:** Best for fuzzy topic matching, understands semantic context, analyzes vault hierarchy
- **Label results as:** source=`LLM`

#### Method 2: MCP Search (text + dataview)

Run both MCP tools in parallel:

**Text search:**
```
mcp__obsidian__search_vault_simple(query="QUERY_HERE")
```
- Score: 0.7 for all matches (exact text match)
- Label as: source=`MCP-text`

**Dataview/structured search:**
```
mcp__obsidian__search_vault(query="QUERY_HERE")
```
- Score: 0.7 for matches (frontmatter, tags, properties)
- Label as: source=`MCP-dv`

- **Fallback:** If MCP unavailable (Obsidian not running), log warning and skip both. This method degrades gracefully.

#### Method 3: Filesystem (Grep + Glob)

Run both in parallel:

**Content search (Grep):**
```
Grep(pattern="KEYWORD", path="data_sources/obsidian/vault/", glob="*.md", output_mode="files_with_matches", head_limit=20)
```
- Score: 0.6 for content matches
- Label as: source=`Grep`

**Filename search (Glob):**
```
Glob(pattern="**/*KEYWORD*", path="data_sources/obsidian/vault/")
```
- Score: 0.5 for filename matches
- Label as: source=`Glob`

- **Strength:** Fastest, zero LLM cost, always available, catches exact keywords
- **Query preprocessing:** Extract 1-3 most distinctive keywords from query (skip articles, prepositions)

### Step 2: Merge & Deduplicate

1. Collect all results into flat list: `[{path, source, score, snippet}]`
2. **Normalize paths:** Strip vault root prefix, lowercase for comparison, treat `folder/` and `folder/index.md` as same
3. **Group by normalized path** (deduplicate same file from multiple sources)
4. For each unique path, track:
   - `sources_count` — how many methods found it (max 3: LLM, MCP, Filesystem)
   - `max_score` — highest relevance score from any method
   - `sources_list` — which methods found it (e.g., "LLM, MCP-text, Grep")

### Step 3: Compute Composite Score

For each unique result:

```
composite = (sources_count / 3) * 0.4 + max_score * 0.6
```

- **Multi-source bonus:** Files found by all 3 methods score highest
- **Quality anchor:** High relevance from even 1 method still ranks well (0.6 * max_score)
- Sort by composite score descending

### Step 4: Classify & Present

**Confidence levels:**

| Level | Criteria |
|-------|----------|
| **HIGH** | composite >= 0.8 AND sources_count >= 2 |
| **MEDIUM** | composite >= 0.6 OR sources_count >= 2 |
| **LOW** | composite < 0.6 AND sources_count == 1 |
| **NO_MATCH** | No results from any method |

**Output format:**

```
VAULT SEARCH: "query"

TOP MATCH: path/to/file.md [0.91]
  Found by: LLM(0.92), MCP-text, Grep

RANKED RESULTS:
1. [0.91] path/to/file1.md — Sources: LLM, MCP-text, Grep
   Snippet: first relevant line or reason
2. [0.78] path/to/folder/ — Sources: LLM, MCP-dv
   Snippet: ...
3. [0.65] path/to/file2.md — Sources: Grep, Glob
   Snippet: ...

RECOMMENDATION:
  Best file: path/to/file1.md
  Best folder: path/to/folder/
  Confidence: HIGH
```

- Show top 10-15 results max
- Include snippet or reason for each
- State confidence level

## Graceful Degradation

If any method fails, log warning and continue with remaining methods.

| Working methods | Quality |
|----------------|---------|
| **3/3** | Full confidence |
| **2/3** | Good results, note failure |
| **1/3** (filesystem always works) | Usable, warn about reduced confidence |
| **0/3** | Report failure, suggest manual browsing |

Warning: `[!] Method X unavailable: [reason]. Continuing with N remaining methods.`

## Query Preprocessing

Before searching:
1. **Extract keywords:** For Grep/Glob, use 1-3 distinctive words
2. **Keep full query:** For LLM and MCP methods, use full natural language
3. **Russian support:** If query in Russian, also search English equivalents
4. **Folder hints:** If query mentions known entity (customer, product, person), add vault path hints

## Integration with /graph_add

When called from `/graph_add`, return structured data (not just formatted output):
- Full results list with scores and sources
- Confidence classification
- Best file and best folder recommendations

## Related Skills

| Skill | Relationship |
|-------|-------------|
| `/graph_add` | Uses graph_find results to decide create/update/refactor |
| `/obsidian` | Primary vault operations (MCP tools, Python ops) |
| `/find-right-folder-for-context` | Simpler folder recommendation (repo-level, not vault search) |

---
Created: 2026-02-05 | Version: 2.0.0
Changes: v2 — honest 3 methods (removed invented search_vault_smart), blended query for session mode
