# Описание структуры таблиц HubSpot в ClickHouse

## Таблицы HubSpot

### 1. stg_hubspot_contacts
Таблица с информацией о контактах из HubSpot.

**Ключевые поля:**
- `primary_key` - Первичный ключ записи
- `hubspot_contacts_id` - Уникальный идентификатор контакта в HubSpot
- `email` - Email адрес контакта
- `hubspot_first_name` - Имя контакта
- `hubspot_last_name` - Фамилия контакта
- `hubspot_job_title` - Должность контакта
- `hubspot_contact_segment` - Сегмент контакта
- `hubspot_contact_pre_qualification_status` - Статус предварительной квалификации
- `hubspot_create_datetime` - Дата и время создания контакта
- `hubspot_country` - Страна контакта
- `hubspot_city` - Город контакта
- `hubspot_contact_linkedin` - Ссылка на профиль LinkedIn
- `hubspot_buyer_persona` - Персона покупателя
- `hubspot_contact_how_we_can_help` - Как мы можем помочь
- `hubspot_contact_lead_score` - Скоринг лида
- `hubspot_contact_likelihood_to_close` - Вероятность закрытия сделки

### 2. stg_hubspot_engagements
Центральная таблица для всех типов взаимодействий в HubSpot.

**Ключевые поля:**
- `primary_key` - Первичный ключ записи
- `hubspot_engagements_id` - Уникальный идентификатор взаимодействия
- `hubspot_engagements_type` - Тип взаимодействия (EMAIL, TASK, MEETING, CALL, INCOMING_EMAIL)
- `hubspot_engagements_created_at` - Дата и время создания взаимодействия
- `hubspot_engagements_last_updated` - Дата и время последнего обновления
- `hubspot_engagements_timestamp` - Временная метка взаимодействия
- `hubspot_engagements_owner_id` - ID владельца взаимодействия
- `hubspot_contact_id_arr` - Массив ID контактов, связанных с взаимодействием
- `hubspot_company_id_arr` - Массив ID компаний, связанных с взаимодействием
- `hubspot_engagements_body` - Тело/содержание взаимодействия
- `hubspot_engagements_body_preview` - Предпросмотр содержания
- `hubspot_engagements_subject` - Тема взаимодействия

### 3. stg_hubspot_engagements_meetings
Таблица с информацией о встречах, являющихся подтипом взаимодействий.

**Ключевые поля:**
- `primary_key` - Первичный ключ записи
- `hubspot_event_id` - ID встречи
- `hubspot_event_title` - Название встречи
- `hubspot_event_start_datetime` - Дата и время начала встречи
- `hubspot_event_timestamp` - Временная метка события
- `hubspot_companies` - Массив ID компаний, связанных со встречей
- `hubspot_contacts` - Массив ID контактов, связанных со встречей
- `hubspot_owner_id` - ID владельца встречи
- `hubspot_body` - Описание встречи
- `hubspot_body_preview` - Предпросмотр описания
- `hubspot_created_datetime` - Дата и время создания записи о встрече
- `hubspot_end_time` - Дата и время окончания встречи
- `hubspot_event_location` - Местоположение встречи
- `hubspot_event_external_url` - Внешний URL встречи (например, ссылка на Zoom)

### 4. stg_hubspot_engagements_calls
Таблица с информацией о звонках.

**Ключевые поля:**
- `hubspot_engagement_call_status` - Статус звонка
- `hubspot_engagement_call_disposition` - Результат звонка
- `hubspot_engagement_call_companies_arr` - Массив компаний, связанных со звонком
- `hubspot_engagement_call_contacts_arr` - Массив контактов, связанных со звонком
- `hubspot_engagement_call_duration_milliseconds` - Продолжительность звонка

### 5. stg_hubspot_engagements_tasks
Таблица с информацией о задачах.

**Ключевые поля:**
- `hubspot_engagement_task_subject` - Тема задачи
- `hubspot_engagement_task_status` - Статус задачи
- `hubspot_engagement_task_priority` - Приоритет задачи
- `hubspot_engagement_task_assigned_date` - Дата назначения задачи
- `hubspot_engagement_task_associated_contacts_number` - Количество связанных контактов

### 6. stg_hubspot_form_submissions
Таблица с информацией о заполнениях форм на сайте.

**Ключевые поля:**
- `hubspot_form_id` - ID формы
- `hubspot_form_submitted_at` - Дата и время заполнения формы
- `hubspot_page_url` - URL страницы, на которой заполнена форма
- `email` - Email пользователя, заполнившего форму
- `hubspot_first_name` - Имя из формы
- `hubspot_last_name` - Фамилия из формы
- `hubspot_values_extract_map` - Карта значений полей формы

## Связи между таблицами

В базе данных HubSpot в ClickHouse таблицы связаны следующим образом:

1. **Связь между контактами и взаимодействиями:**
   - Таблица `stg_hubspot_contacts` связана с `stg_hubspot_engagements` через поле `hubspot_contacts_id` и массив `hubspot_contact_id_arr` соответственно
   - Один контакт может быть связан с несколькими взаимодействиями
   - Одно взаимодействие может быть связано с несколькими контактами (через массив)

2. **Связь между взаимодействиями и их подтипами:**
   - Все специфические таблицы подтипов (`stg_hubspot_engagements_meetings`, `stg_hubspot_engagements_calls`, `stg_hubspot_engagements_tasks`) связаны с основной таблицей `stg_hubspot_engagements` через поле `hubspot_engagements_id`
   - Тип взаимодействия определяется полем `hubspot_engagements_type` в таблице `stg_hubspot_engagements`

3. **Связь между формами и контактами:**
   - Таблица `stg_hubspot_form_submissions` связана с `stg_hubspot_contacts` через поле `email`
   - Одна форма может быть связана с одним контактом (через email)

4. **Связь компаний (которая отсутствует как отдельная таблица):**
   - Информация о компаниях хранится в виде массивов идентификаторов в таблицах `stg_hubspot_engagements` (поле `hubspot_company_id_arr`) и `stg_hubspot_engagements_meetings` (поле `hubspot_companies`)
   - Для получения информации о компании часто используются данные контакта, включая домен email

## Особенности работы с данными HubSpot

1. **Работа с массивами:**
   - Многие таблицы содержат поля-массивы (например, `hubspot_contact_id_arr`, `hubspot_company_id_arr`, `hubspot_contacts`, `hubspot_companies`)
   - Для работы с элементами массивов используется функция `arrayJoin` и `arrayElement`
   - В JOIN нельзя напрямую использовать arrayExists или другие функции массивов

2. **Типы коммуникаций:**
   - EMAIL - электронные письма
   - TASK - задачи
   - MEETING - встречи
   - CALL - звонки
   - INCOMING_EMAIL - входящие электронные письма

3. **Связь через домены:**
   - Для определения компании часто используется домен email-адреса контакта
   - Для этого можно использовать функцию `arrayElement(splitByChar('@', email), 2)`

4. **Отсутствие отдельной таблицы компаний:**
   - В системе нет отдельной таблицы компаний, и информация о них хранится только как ID
   - Для аналитики компаний часто используются данные связанных контактов

5. **Временные метки:**
   - В системе используется несколько временных меток для разных целей:
     - `hubspot_create_datetime` - создание записи
     - `hubspot_engagements_created_at` - создание взаимодействия
     - `hubspot_event_start_datetime` - начало встречи
     - `hubspot_form_submitted_at` - время заполнения формы

## Примеры запросов

### Получение встреч, созданных после заполнения форм

Этот запрос объединяет данные о заполненных формах и встречах, чтобы найти встречи, которые были созданы после заполнения формы контактом:

```sql
WITH 
-- Шаг 1: Получаем информацию о заполненных формах и контактах
form_contacts AS (
    SELECT 
        fs.email as email,
        fs.hubspot_form_id as form_id,
        fs.hubspot_form_submitted_at as form_submitted_at,
        fs.hubspot_page_url as page_url,
        c.hubspot_contacts_id as contact_id
    FROM stg_hubspot_form_submissions fs 
    JOIN stg_hubspot_contacts c ON fs.email = c.email
), 
-- Шаг 2: Получаем информацию о контактах
contact_info AS (
    SELECT 
        hubspot_contacts_id as contact_id,
        hubspot_first_name as first_name,
        hubspot_last_name as last_name,
        hubspot_job_title as job_title
    FROM stg_hubspot_contacts
),
-- Шаг 3: Получаем все встречи с контактами через массив контактов
meetings_with_contacts AS (
    SELECT 
        e.hubspot_engagements_id as engagement_id,
        e.hubspot_engagements_created_at as created_at,
        arrayJoin(e.hubspot_contact_id_arr) AS contact_id
    FROM stg_hubspot_engagements e 
    WHERE e.hubspot_engagements_type = 'MEETING'
)
-- Шаг 4: Объединяем данные и получаем все встречи, которые были созданы после заполнения формы
SELECT 
    fc.email,
    ci.first_name,
    ci.last_name,
    ci.job_title,
    fc.form_id,
    fc.form_submitted_at,
    fc.page_url,
    m.hubspot_event_title,
    m.hubspot_event_start_datetime,
    m.hubspot_end_time,
    m.hubspot_event_external_url,
    mwc.created_at as meeting_created_at,
    dateDiff('second', fc.form_submitted_at, mwc.created_at) as seconds_between_form_and_meeting
FROM form_contacts fc 
JOIN contact_info ci ON fc.contact_id = ci.contact_id
JOIN meetings_with_contacts mwc ON fc.contact_id = mwc.contact_id
JOIN stg_hubspot_engagements_meetings m ON mwc.engagement_id = m.hubspot_event_id
WHERE 
    -- Только встречи, созданные после заполнения формы
    mwc.created_at >= fc.form_submitted_at
ORDER BY 
    fc.form_submitted_at DESC,
    seconds_between_form_and_meeting ASC
LIMIT 10
```

Этот запрос показывает, как работать с массивами контактов в таблице `stg_hubspot_engagements` через `arrayJoin` и объединять данные из разных таблиц HubSpot. Он может быть использован для анализа автоматизации и выявления паттернов в процессе записи на встречи после заполнения форм. 